<?php

/**
 * WooCommerce Payment Method Blocks Integration.
 *
 * @since 3.7.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

/**
 * Class for integrating with WooCommerce payment method blocks scripts
 *
 * @since 3.7.0
 */
class WAL_WC_Payment_Method_Blocks_Integration extends AbstractPaymentMethodType {

	/**
	 * Payment method name defined by payment methods extending this class.
	 *
	 * @since 3.7.0
	 * @var string
	 */
	protected $name = 'wal_wallet';

	/**
	 * Payment gateway.
	 *
	 * @since 3.7.0
	 * @var object/null
	 */
	protected $payment_gateway;

	/**
	 * Whether the integration has been initialized.
	 *
	 * @since 3.7.0
	 * @var boolean
	 */
	protected $is_initialized;

	/**
	 * The single instance of the class.
	 *
	 * @since 3.7.0
	 * @var WAL_WC_Payment_Method_Blocks_Integration
	 */
	protected static $_instance = null;

	/**
	 * Main WAL_WC_Payment_Method_Blocks_Integration instance. Ensures only one instance of WAL_WC_Payment_Method_Blocks_Integration is loaded or can be loaded.
	 *
	 * @since 3.7.0
	 * @static
	 * @return WAL_WC_Payment_Method_Blocks_Integration
	 */
	public static function instance() {
		if (is_null(self::$_instance)) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * Cloning is forbidden.
	 * 
	 * @since 3.7.0
	 */
	public function __clone() {
		_doing_it_wrong(__FUNCTION__, esc_html__('Foul!', 'wallet-for-woocommerce'), '3.7.0');
	}

	/**
	 * Unserializing instances of this class is forbidden.
	 * 
	 * @since 3.7.0
	 */
	public function __wakeup() {
		_doing_it_wrong(__FUNCTION__, esc_html__('Foul!', 'wallet-for-woocommerce'), '3.7.0');
	}

	/**
	 * When called invokes any initialization/setup for the integration.
	 * 
	 * @since 3.7.0
	 */
	public function initialize() {
		if ($this->is_initialized) {
			return;
		}

		$this->payment_gateway = new WAL_Gateway_Wallet();

		// Enqueue block assets for the front-end.
		add_action('enqueue_block_assets', array( $this, 'enqueue_block_assets' ));
	}

	/**
	 * Returns if this payment method should be active. If false, the scripts will not be enqueued.
	 *
	 * @since 3.7.0
	 * @return boolean
	 */
	public function is_active() {
		return 'yes' === $this->payment_gateway->enabled;
	}

	/**
	 * Returns an array of key=>value pairs of data made available to the payment methods script.
	 *
	 * @since 3.7.0
	 * @return array
	 */
	public function get_payment_method_data() {
		return array(
			'id' => $this->payment_gateway->id,
			'title' => $this->payment_gateway->get_title(),
			'description' => $this->payment_gateway->get_description(),
			'supports' => $this->payment_gateway->supports,
		);
	}

	/**
	 * Returns an array of scripts/handles to be registered for this payment method.
	 *
	 * @since 3.7.0
	 *
	 * @return void
	 */
	public function get_payment_method_script_handles() {
		return array( 'wal-wc-payment-blocks' );
	}

	/**
	 * Enqueue block assets for the front-end.
	 *
	 * @since 3.7.0
	 *
	 * @return void
	 */
	public function enqueue_block_assets() {
		// Load script.
		$script_asset_details = $this->get_script_asset_details();

		wp_register_script(
				'wal-wc-payment-blocks',
				WAL_PLUGIN_URL . '/assets/blocks/payment-method/index.js',
				$script_asset_details['dependencies'],
				$script_asset_details['version'],
				true
		);
	}

	/**
	 * Get the script asset details from the file if exists.
	 * 
	 * @since 3.7.0
	 * @param string $site
	 * @return array
	 */
	private function get_script_asset_details() {
		$script_asset_path = WAL_PLUGIN_PATH . '/assets/blocks/payment-method/index.asset.php';

		return file_exists($script_asset_path) ? require $script_asset_path : array(
			'dependencies' => array(),
			'version' => WAL_VERSION,
		);
	}
}
