<?php

/**
 * WooCommerce Blocks Store API.
 *
 * @since 3.7.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

use Automattic\WooCommerce\StoreApi\Schemas\V1\CartSchema;
use Automattic\WooCommerce\StoreApi\Exceptions\RouteException;

/**
 * Class for extend store API of cart/checkout.
 *
 * @since 3.7.0
 */
class WAL_WC_Blocks_Store_API {

	/**
	 * Plugin Identifier, unique to each plugin.
	 *
	 * @since 3.7.0
	 * @var string
	 */
	const IDENTIFIER = 'wal-wallet';

	/**
	 * Bootstrap.
	 * 
	 * @since 3.7.0
	 */
	public static function init() {
		// Extend StoreAPI.
		self::extend_store();

		// Filter topup cart item quantity.
		add_filter('woocommerce_store_api_product_quantity_minimum', array( __CLASS__, 'filter_cart_item_qty' ), 10, 3);
		add_filter('woocommerce_store_api_product_quantity_maximum', array( __CLASS__, 'filter_cart_item_qty' ), 10, 3);
	}

	/**
	 * Register extensibility points.
	 * 
	 * @since 3.7.0
	 */
	protected static function extend_store() {
		if (function_exists('woocommerce_store_api_register_endpoint_data')) {
			woocommerce_store_api_register_endpoint_data(
					array(
						'endpoint' => CartSchema::IDENTIFIER,
						'namespace' => self::IDENTIFIER,
						'data_callback' => array( 'WAL_WC_Blocks_Store_API', 'extend_cart_data' ),
						'schema_callback' => array( 'WAL_WC_Blocks_Store_API', 'extend_cart_schema' ),
						'schema_type' => ARRAY_A,
					)
			);
		}

		if (function_exists('woocommerce_store_api_register_update_callback')) {
			woocommerce_store_api_register_update_callback(
					array(
						'namespace' => self::IDENTIFIER,
						'callback' => array( 'WAL_WC_Blocks_Store_API', 'rest_handle_endpoint' ),
					)
			);
		}
	}

	/**
	 * Register gift wrapper schema in the cart schema.
	 * 
	 * @since 3.7.0
	 * @return array
	 */
	public static function extend_cart_schema() {
		return array();
	}

	/**
	 * Register gift wrapper data in the cart API.
	 * 
	 * @since 3.7.0
	 * @return array
	 */
	public static function extend_cart_data() {
		/**
		 * This hook is used to alter the extend cart data.
		 * 
		 * @since 3.7.0
		 */
		return apply_filters('wal_extend_cart_data', array(
			'redeem_wallet_fund_title' => wal_get_checkout_partial_usage_title_label(),
			'wallet_fee_html' => wal_get_block_wallet_fee_html(),
			'notices' => wal_get_store_api_cart_notices(),
			'cart_redeem_wallet_fund_form_html' => wal_get_cart_block_redeem_wallet_fund_html(),
			'checkout_redeem_wallet_fund_form_html' => wal_get_checkout_block_redeem_wallet_fund_html(),
		));
	}

	/**
	 * Filter the Top-up cart item quantity.
	 * 
	 * @since 3.7.0
	 * @param int $value
	 * @param object $product
	 * @param array $cart_item
	 * @return int
	 */
	public static function filter_cart_item_qty( $value, $product, $cart_item ) {
		// Omit it if the current cart item is not topup cart item.
		if (!isset($cart_item['wal_topup'])) {
			return $value;
		}

		return 1;
	}

	/**
	 * Handles wallet rest endpoints.
	 * 
	 * @since 3.7.0
	 * @param array $args
	 */
	public static function rest_handle_endpoint( $args ) {
		if (!isset($args['action'])) {
			return;
		}

		switch ($args['action']) {
			case 'apply_redeem_wallet_fund':
				self::process_apply_redeem_wallet_fund($args);
				break;

			case 'remove_redeem_wallet_fund':
				self::process_remove_redeem_wallet_fund();
				break;
		}
	}

	/**
	 * Process the apply redeem wallet fund in the WC session.
	 * 
	 * @since 3.7.0
	 * @param array $args
	 * @throws RouteException
	 */
	public static function process_apply_redeem_wallet_fund( $args ) {
		$post_fund = isset($args['fund']) ? wc_clean(wp_unslash($args['fund'])) : '';
		// Validate if the fund exists.
		if (!$post_fund) {
			throw new RouteException('woocommerce_blocks_redeem_wallet_fund_empty', wp_kses_post(get_option('wal_messages_partial_fund_empty')), 400);
		}

		// Validate if the given fund is non numeric.
		if (!is_numeric($post_fund)) {
			throw new RouteException('woocommerce_blocks_redeem_wallet_fund_not_numeric', wp_kses_post(get_option('wal_messages_partial_fund_numeric')), 400);
		}

		// Validate if the current user is valid to apply wallet fund.
		if (!WAL_Current_User_Wallet::is_valid_fund_usage()) {
			throw new RouteException('woocommerce_blocks_redeem_wallet_fund_is_not_valid', wp_kses_post(get_option('wal_messages_partial_fund_user_restricted')), 400);
		}

		$fund = floatval(wal_convert_price($post_fund, true));
		// Return if the fund is greater than wallet balance.
		if (WAL_Current_User_Wallet::get_balance() < $fund) {
			/* translators: %s wallet balance */
			throw new RouteException('woocommerce_blocks_redeem_wallet_fund_is_greater_than_wallet', wp_kses_post(str_replace('{wallet_balance}', wal_convert_price_by_currency(WAL_Current_User_Wallet::get_balance()), get_option('wal_messages_partial_wallet_insufficient_fund'))), 400);
		}

		$fund = WAL_Current_User_Wallet::get_fund_to_apply($fund);

		if ( empty($fund) ) {
			throw new RouteException('woocommerce_blocks_redeem_wallet_fund_is_invalid_for_product', wp_kses_post(get_option('wal_messages_product_filter_error_for_usage')), 400);
		}

		$maximum_amount = wal_get_maximum_fund_usage_limit();
		// Return if the fund is greater than maximum amount.
		if (false !== $maximum_amount && $maximum_amount < $fund) {
			/* translators: %s maximum amount */
			throw new RouteException('woocommerce_blocks_redeem_wallet_fund_is_more_than_maximum_amount', wp_kses_post(str_replace('{restricted_funds}', wal_convert_price_by_currency($maximum_amount), get_option('wal_messages_partial_allow_maximum_fund'))), 400);
		}

		// Return if the wallet balance is less than order total.
		if ('2' != get_option('wal_general_partial_payments_mode') && WAL_Current_User_Wallet::get_balance() >= wal_get_wc_cart_total()) {
			throw new RouteException('woocommerce_blocks_redeem_wallet_fund_is_less_than_order_total', esc_html__('Sorry, You are not valid to use fund.', 'wallet-for-woocommerce'), 400);
		}

		// Set the partial fund in the WC session.
		WC()->session->set('wal_partial_fund', $fund);
	}

	/**
	 * Process the remove redeem wallet fund from the WC session.
	 * 
	 * @since 3.7.0
	 * @throws RouteException
	 */
	public static function process_remove_redeem_wallet_fund() {
		// Validate if the redeem wallet fund applied.
		if (!WC()->session->get('wal_partial_fund')) {
			throw new RouteException('woocommerce_blocks_redeem_wallet_fund_not_applied', esc_html__('You have already removed the applied funds.', 'wallet-for-woocommerce'), 400);
		}

		// Remove the redeem wallet fund from the WC session.
		WC()->session->set('wal_partial_fund', null);
	}
}
