<?php

/**
 * WooCommerce Blocks Integration.
 *
 * @since 3.7.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

use Automattic\WooCommerce\Blocks\Integrations\IntegrationInterface;

/**
 * Class for integrating with WooCommerce Blocks scripts
 *
 * @since 3.7.0
 */
class WAL_WC_Blocks_Integration implements IntegrationInterface {

	/**
	 * Whether the integration has been initialized.
	 *
	 * @since 3.7.0
	 * @var boolean
	 */
	protected $is_initialized;

	/**
	 * The single instance of the class.
	 *
	 * @since 3.7.0
	 * @var WAL_WC_Blocks_Integration
	 */
	protected static $_instance = null;

	/**
	 * Main WAL_WC_Blocks_Integration instance. Ensures only one instance of WAL_WC_Blocks_Integration is loaded or can be loaded.
	 *
	 * @since 3.7.0
	 * @static
	 * @return WAL_WC_Blocks_Integration
	 */
	public static function instance() {
		if (is_null(self::$_instance)) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * Cloning is forbidden.
	 * 
	 * @since 3.7.0
	 */
	public function __clone() {
		_doing_it_wrong(__FUNCTION__, esc_html__('Foul!', 'wallet-for-woocommerce'), '3.7.0');
	}

	/**
	 * Unserializing instances of this class is forbidden.
	 * 
	 * @since 3.7.0
	 */
	public function __wakeup() {
		_doing_it_wrong(__FUNCTION__, esc_html__('Foul!', 'wallet-for-woocommerce'), '3.7.0');
	}

	/**
	 * The name of the integration.
	 *
	 * @since 3.7.0
	 * @return string
	 */
	public function get_name() {
		return 'wal-wc-blocks';
	}

	/**
	 * When called invokes any initialization/setup for the integration.
	 * 
	 * @since 3.7.0
	 */
	public function initialize() {
		if ($this->is_initialized) {
			return;
		}

		// Enqueue block assets for the editor.
		add_action('enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ));
		// Enqueue block assets for the front-end.
		add_action('enqueue_block_assets', array( $this, 'enqueue_block_assets' ));
	}

	/**
	 * Returns an array of script handles to enqueue in the frontend context.
	 *
	 * @since 3.7.0
	 * @return string[]
	 */
	public function get_script_handles() {
		return array( 'wal-wc-blocks' );
	}

	/**
	 * Returns an array of script handles to enqueue in the editor context.
	 *
	 * @since 3.7.0
	 * @return string[]
	 */
	public function get_editor_script_handles() {
		return array( 'wal-wc-blocks' );
	}

	/**
	 * Enqueue block assets for the editor.
	 *
	 * @since 3.7.0
	 *
	 * @return void
	 */
	public function enqueue_block_editor_assets() {
		// Load script.
		$script_asset_details = $this->get_script_asset_details('admin');

		wp_register_script(
				'wal-wc-blocks',
				WAL_PLUGIN_URL . '/assets/blocks/admin/index.js',
				$script_asset_details['dependencies'],
				$script_asset_details['version'],
				true
		);
	}

	/**
	 * Get the script asset details from the file if exists.
	 * 
	 * @since 3.7.0
	 * @param string $site
	 * @return array
	 */
	private function get_script_asset_details( $site = 'frontend' ) {
		$script_asset_path = WAL_PLUGIN_PATH . '/assets/blocks/' . sanitize_file_name($site) . '/index.asset.php';

		return file_exists($script_asset_path) ? require WAL_PLUGIN_PATH . '/assets/blocks/' . sanitize_file_name($site) . '/index.asset.php' : array(
			'dependencies' => array(),
			'version' => WAL_VERSION,
		);
	}

	/**
	 * Enqueue block assets for the front-end.
	 *
	 * @since 3.7.0
	 *
	 * @return void
	 */
	public function enqueue_block_assets() {
		// Load script.
		$script_asset_details = $this->get_script_asset_details();

		wp_register_script(
				'wal-wc-blocks',
				WAL_PLUGIN_URL . '/assets/blocks/frontend/index.js',
				$script_asset_details['dependencies'],
				$script_asset_details['version'],
				true
		);

		wp_enqueue_style(
				'wal-wc-blocks',
				WAL_PLUGIN_URL . '/assets/blocks/frontend/index.css',
				'',
				$script_asset_details['version']
		);
	}

	/**
	 * An array of key, value pairs of data made available to the block on the client side.
	 *
	 * @since 3.7.0
	 * @return array
	 */
	public function get_script_data() {
		if (!is_admin()) {
			return array(
				'apply_discount_for_gateway' => get_option('wal_general_apply_discount_for_wallet_gateway'),
				'redeem_wallet_fund_added_message' => get_option('wal_messages_partial_fund_added'),
				'redeem_wallet_fund_removed_message' => get_option('wal_messages_partial_fund_removed'),
			);
		} else {
			return array(
				'redeem_wallet_fund_form_title' => wal_get_checkout_partial_usage_title_label(),
			);
		}
	}
}
