<?php

/**
 * Store API functions.
 * 
 * @since 3.7.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!function_exists('wal_store_api_format_price')) {

	/**
	 * Format the price by money formatter.
	 * 
	 * @since 3.7.0
	 * @param float $price
	 * @return float
	 */
	function wal_store_api_format_price( $price ) {
		return woocommerce_store_api_get_formatter('money')->format($price);
	}

}

if (!function_exists('wal_get_store_api_cart_notices')) {

	/**
	 * Get the store API cart notices.
	 * 
	 * @since 3.7.0
	 * @return array
	 */
	function wal_get_store_api_cart_notices() {
		/**
		 * This hook is used to alter the store API cart notices.
		 * 
		 * @since 3.7.0
		 */
		$notices = apply_filters('wal_store_api_cart_notices', array());

		return array_filter($notices, 'wal_array_filter');
	}

}

if (!function_exists('wal_is_block_cart')) {

	/**
	 * Is a block cart page?.
	 *
	 * @since 3.7.0
	 * @return boolean
	 */
	function wal_is_block_cart() {
		global $post;
		$is_singular = true;
		if (!is_a($post, 'WP_Post')) {
			$is_singular = false;
		}

		// Consider as block cart while the request call via Store API.
		if (isset($GLOBALS['wp']->query_vars['rest_route']) && false !== strpos($GLOBALS['wp']->query_vars['rest_route'], '/wc/store/v1')) {
			return true;
		}

		return $is_singular && has_block('woocommerce/cart', $post);
	}

}

if (!function_exists('wal_is_block_checkout')) {

	/**
	 * Is a block checkout page?.
	 *
	 * @since 3.7.0
	 * @return boolean
	 */
	function wal_is_block_checkout() {
		global $post;
		$is_singular = true;
		if (!is_a($post, 'WP_Post')) {
			$is_singular = false;
		}

		// Consider as block checkout while the request call via Store API.
		if (isset($GLOBALS['wp']->query_vars['rest_route']) && false !== strpos($GLOBALS['wp']->query_vars['rest_route'], '/wc/store/v1/cart')) {
			return true;
		}

		return $is_singular && has_block('woocommerce/checkout', $post);
	}

}

if (!function_exists('wal_get_cart_block_redeem_wallet_fund_html')) {

	/**
	 * Get the cart block redeem wallet fund HTML.
	 *
	 * @since 3.7.0
	 * @return HTML
	 */
	function wal_get_cart_block_redeem_wallet_fund_html() {
		// Return if the redeem wallet fund form is not valid to display.
		if (!wal_can_render_redeem_wallet_fund_form_in_cart()) {
			return '';
		}

		return wal_get_template_html('block/cart-redeem-wallet-fund-form.php');
	}

}

if (!function_exists('wal_get_checkout_block_redeem_wallet_fund_html')) {

	/**
	 * Get the checkout block redeem wallet fund HTML.
	 *
	 * @since 3.7.0
	 * @return HTML
	 */
	function wal_get_checkout_block_redeem_wallet_fund_html() {
		// Return if the redeem wallet fund form is not valid to display.
		if (!wal_can_render_redeem_wallet_fund_form_in_checkout()) {
			return '';
		}

		return wal_get_template_html('block/checkout-redeem-wallet-fund-form.php');
	}

}

if (!function_exists('wal_get_block_wallet_fee_html')) {

	/**
	 * Get the block wallet fee HTML.
	 *
	 * @since 3.7.0
	 * @return HTML
	 */
	function wal_get_block_wallet_fee_html() {
		// Return if the wallet fee session does not exists in the WC session.
		if (!WC()->session->get('wal_partial_fund')) {
			return;
		}
				
		$fees = WC()->cart->get_fees();
		if (!wal_check_is_array($fees)) {
			return;
		}

		// Return if wallet fee does not exists.
		if (!array_key_exists(WAL()->wallet_fee_name(), $fees)) {
			return;
		}

		return wal_get_template_html('block/wallet-fee-wrapper.php', array( 'fee' => $fees[WAL()->wallet_fee_name()] ));
	}

}
