<?php

/**
 * Post functions.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! function_exists( 'wal_create_new_wallet' ) ) {

	/**
	 * Create a new wallet.
	 *
	 * @since 1.0.0
	 * @param array $meta_args
	 * @param array $post_args
	 * @return integer/string
	 */
	function wal_create_new_wallet( $meta_args, $post_args = array() ) {

		$object = new WAL_Wallet();
		$id     = $object->create( $meta_args, $post_args );

		return $id;
	}

}

if ( ! function_exists( 'wal_get_wallet' ) ) {

	/**
	 * Get the wallet object.
	 *
	 * @since 1.0.0
	 * @param int $id
	 * @return object
	 */
	function wal_get_wallet( $id ) {

		$object = new WAL_Wallet( $id );

		return $object;
	}

}

if ( ! function_exists( 'wal_update_wallet' ) ) {

	/**
	 * Update the wallet.
	 *
	 * @since 1.0.0
	 * @param array $meta_args
	 * @param array $post_args
	 * @return object
	 */
	function wal_update_wallet( $id, $meta_args, $post_args = array() ) {

		$object = new WAL_Wallet( $id );
		$object->update( $meta_args, $post_args );

		return $object;
	}

}

if ( ! function_exists( 'wal_delete_wallet' ) ) {

	/**
	 * Delete the wallet.
	 *
	 * @since 1.0.0
	 * @param int  $id
	 * @param bool $force
	 * @return bool
	 */
	function wal_delete_wallet( $id, $force = true ) {

		wp_delete_post( $id, $force );

		return true;
	}

}

if ( ! function_exists( 'wal_get_wallet_statuses' ) ) {

	/**
	 * Get the wallet statuses.
	 *
	 * @since 1.0.0
	 * @return array
	 */
	function wal_get_wallet_statuses() {
		/**
		 * This hook is used to alter the wallet statuses.
		 *
		 * @since 1.0.0
		 */
		return apply_filters( 'wal_wallet_statuses', array( 'wal_active', 'wal_inactive', 'wal_expired' ) );
	}

}

if ( ! function_exists( 'wal_create_new_transaction_log' ) ) {

	/**
	 * Create a new transaction log.
	 *
	 * @since 1.0.0
	 * @param array $meta_args
	 * @param array $post_args
	 * @return integer/string
	 */
	function wal_create_new_transaction_log( $meta_args, $post_args = array() ) {

		$object = new WAL_Transaction_Log();
		$id     = $object->create( $meta_args, $post_args );

		return $id;
	}

}

if ( ! function_exists( 'wal_get_transaction_log' ) ) {

	/**
	 * Get the transaction log object.
	 *
	 * @since 1.0.0
	 * @param int $id
	 * @return object
	 */
	function wal_get_transaction_log( $id ) {

		$object = new WAL_Transaction_Log( $id );

		return $object;
	}

}

if ( ! function_exists( 'wal_update_transaction_log' ) ) {

	/**
	 * Update the transaction log.
	 *
	 * @since 1.0.0
	 * @param int   $id
	 * @param array $meta_args
	 * @param array $post_args
	 * @return object
	 */
	function wal_update_transaction_log( $id, $meta_args, $post_args = array() ) {

		$object = new WAL_Transaction_Log( $id );
		$object->update( $meta_args, $post_args );

		return $object;
	}

}

if ( ! function_exists( 'wal_delete_transaction_log' ) ) {

	/**
	 * Delete the transaction log.
	 *
	 * @since 1.0.0
	 * @param int  $id
	 * @param bool $force
	 * @return bool
	 */
	function wal_delete_transaction_log( $id, $force = true ) {

		wp_delete_post( $id, $force );

		return true;
	}

}

if ( ! function_exists( 'wal_get_transaction_log_statuses' ) ) {

	/**
	 * Get the transaction log statuses.
	 *
	 * @since 1.0.0
	 * @return array
	 */
	function wal_get_transaction_log_statuses() {
		/**
		 * This hook is used to alter the transaction log statuses.
		 *
		 * @since 1.0
		 */
		return apply_filters( 'wal_transaction_log_statuses', array( 'wal_credit', 'wal_debit' ) );
	}

}

if ( ! function_exists( 'wal_display_post_status' ) ) {

	/**
	 * Display the formatted status.
	 *
	 * @since 1.0.0
	 * @param string $status
	 * @param bool   $html
	 * @return string
	 */
	function wal_display_post_status( $status, $html = true ) {

		$status_object = get_post_status_object( $status );

		if ( ! isset( $status_object ) ) {
			return '';
		}

		return $html ? '<mark class="wal_status_label ' . esc_attr( $status ) . '_status"><span >' . esc_html( $status_object->label ) . '</span></mark>' : esc_html( $status_object->label );
	}

}

if ( ! function_exists( 'wal_get_wallet_id_by_user_id' ) ) {

	/**
	 * Get the wallet ID by user ID.
	 *
	 * @since 1.0.0
	 * @param int  $user_id
	 * @param bool $status
	 * @return int/bool
	 */
	function wal_get_wallet_id_by_user_id( $user_id, $status = false ) {

		if ( ! $status ) {
			$status = wal_get_wallet_statuses();
		}

		$args = array(
			'post_type'      => WAL_Register_Post_Types::WALLET_POSTTYPE,
			'post_status'    => $status,
			'post_parent'    => $user_id,
			'posts_per_page' => 1,
			'fields'         => 'ids',
		);

		$wallet_id = get_posts( $args );

		if ( ! wal_check_is_array( $wallet_id ) ) {
			return false;
		}

		return reset( $wallet_id );
	}

}

if ( ! function_exists( 'wal_get_wallet_by_user_id' ) ) {

	/**
	 * Get the wallet by user ID.
	 *
	 * @since 1.0.0
	 * @param int  $user_id
	 * @param bool $status
	 * @return object/bool
	 */
	function wal_get_wallet_by_user_id( $user_id, $status = false ) {
		$wallet_id = wal_get_wallet_id_by_user_id( $user_id, $status );
		if ( ! $wallet_id ) {
			return false;
		}

		$wallet = wal_get_wallet( $wallet_id );
		if ( ! $wallet->exists() ) {
			return false;
		}

		return $wallet;
	}

}

if ( ! function_exists( 'wal_user_transaction_logs' ) ) {

	/**
	 * Get the user transaction logs.
	 *
	 * @since 1.0.0
	 * @param bool/int $user_id
	 * @return array
	 */
	function wal_user_transaction_logs( $user_id = false ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$post_args = array(
			'post_type'   => WAL_Register_Post_Types::TRANSACTION_LOG_POSTTYPE,
			'post_status' => wal_get_transaction_log_statuses(),
			'fields'      => 'ids',
			'numberposts' => '-1',
			'meta_key'    => 'wal_user_id',
			'meta_value'  => $user_id,
			'orderby'     => 'ID',
		);

		return get_posts( $post_args );
	}

}

if ( ! function_exists( 'wal_users_transaction_logs' ) ) {

	/**
	 * Get the users transaction logs.
	 *
	 * @since 3.5.0
	 * @param array $user_ids User IDs.
	 * @return array
	 */
	function wal_users_transaction_logs( $user_ids ) {
		return get_posts(
			array(
				'post_type'   => WAL_Register_Post_Types::TRANSACTION_LOG_POSTTYPE,
				'post_status' => wal_get_transaction_log_statuses(),
				'fields'      => 'ids',
				'numberposts' => '-1',
				'orderby'     => 'ID',
				'meta_query'  => array(
					array(
						'key'     => 'wal_user_id',
						'value'   => $user_ids,
						'compare' => 'IN',
					),
				),
			)
		);
	}

}
