<?php

/**
 * Module functions.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!function_exists('wal_create_new_fund_transfer')) {

	/**
	 * Create a new fund transfer.
	 *
	 * @return integer/string
	 */
	function wal_create_new_fund_transfer( $meta_args, $post_args = array() ) {

		$object = new WAL_Fund_Transfer();
		$id = $object->create($meta_args, $post_args);

		return $id;
	}

}

if (!function_exists('wal_get_fund_transfer')) {

	/**
	 * Get the fund transfer object.
	 *
	 * @return object
	 */
	function wal_get_fund_transfer( $id ) {

		$object = new WAL_Fund_Transfer($id);

		return $object;
	}

}

if (!function_exists('wal_update_fund_transfer')) {

	/**
	 * Update the fund transfer.
	 *
	 * @return object
	 */
	function wal_update_fund_transfer( $id, $meta_args, $post_args = array() ) {

		$object = new WAL_Fund_Transfer($id);
		$object->update($meta_args, $post_args);

		return $object;
	}

}

if (!function_exists('wal_delete_fund_transfer')) {

	/**
	 * Delete the fund transfer.
	 *
	 * @return bool
	 */
	function wal_delete_fund_transfer( $id, $force = true ) {

		wp_delete_post($id, $force);

		return true;
	}

}

if (!function_exists('wal_get_fund_transfer_statuses')) {

	/**
	 * Get the fund transfer statuses.
	 *
	 * @return array
	 */
	function wal_get_fund_transfer_statuses() {
		/**
		 * This hook is used to alter the fund transfer statuses.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_fund_transfer_statuses', array(
			'wal_transfered',
			'wal_received',
			'wal_request_received',
			'wal_requested',
			'wal_request_declined',
			'wal_request_cancel',
		));
	}

}

if (!function_exists('wal_create_new_fund_transfer_log')) {

	/**
	 * Create a new fund transfer log.
	 *
	 * @return integer/string
	 */
	function wal_create_new_fund_transfer_log( $meta_args, $post_args = array() ) {

		$object = new WAL_Fund_Transfer_Log();
		$id = $object->create($meta_args, $post_args);

		return $id;
	}

}

if (!function_exists('wal_get_fund_transfer_log')) {

	/**
	 * Get the fund transfer log object.
	 *
	 * @return object
	 */
	function wal_get_fund_transfer_log( $id ) {

		$object = new WAL_Fund_Transfer_Log($id);

		return $object;
	}

}

if (!function_exists('wal_update_fund_transfer_log')) {

	/**
	 * Update the fund transfer log.
	 *
	 * @return object
	 */
	function wal_update_fund_transfer_log( $id, $meta_args, $post_args = array() ) {

		$object = new WAL_Fund_Transfer_Log($id);
		$object->update($meta_args, $post_args);

		return $object;
	}

}

if (!function_exists('wal_delete_fund_transfer_log')) {

	/**
	 * Delete the fund transfer log.
	 *
	 * @return bool
	 */
	function wal_delete_fund_transfer_log( $id, $force = true ) {

		wp_delete_post($id, $force);

		return true;
	}

}

if (!function_exists('wal_get_fund_transfer_log_statuses')) {

	/**
	 * Get the fund transfer log statuses.
	 *
	 * @return array
	 */
	function wal_get_fund_transfer_log_statuses() {
		/**
		 * This hook is used to alter the fund transfer log statuses.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_fund_transfer_log_statuses', array(
			'wal_transfered',
			'wal_received',
			'wal_request_received',
			'wal_requested',
			'wal_request_declined',
			'wal_request_cancel',
		));
	}

}

if (!function_exists('wal_get_auto_topup_statuses')) {

	/**
	 * Get the auto topup statuses.
	 *
	 * @return array
	 */
	function wal_get_auto_topup_statuses() {
		/**
		 * This hook is used to alter the wallet auto top-up statuses.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_wallet_auto_topup_statuses', array( 'wal_active', 'wal_cancelled' ));
	}

}

if (!function_exists('wal_create_new_auto_topup')) {

	/**
	 * Create a new auto topup.
	 *
	 * @return integer/string
	 */
	function wal_create_new_auto_topup( $meta_args, $post_args = array() ) {

		$object = new WAL_Auto_Topup();
		$id = $object->create($meta_args, $post_args);

		return $id;
	}

}

if (!function_exists('wal_get_auto_topup')) {

	/**
	 * Get the auto topup object.
	 *
	 * @return object
	 */
	function wal_get_auto_topup( $id ) {

		$object = new WAL_Auto_Topup($id);

		return $object;
	}

}

if (!function_exists('wal_update_auto_topup')) {

	/**
	 * Update the auto topup.
	 *
	 * @return object
	 */
	function wal_update_auto_topup( $id, $meta_args, $post_args = array() ) {

		$object = new WAL_Auto_Topup($id);
		$object->update($meta_args, $post_args);

		return $object;
	}

}

if (!function_exists('wal_create_new_gift_voucher')) {

	/**
	 * Create a new gift voucher.
	 *
	 * @return integer/string
	 */
	function wal_create_new_gift_voucher( $meta_args, $post_args = array() ) {

		$object = new WAL_Gift_Voucher();
		$id = $object->create($meta_args, $post_args);

		return $id;
	}

}

if (!function_exists('wal_get_gift_voucher')) {

	/**
	 * Get the gift voucher object.
	 *
	 * @return object
	 */
	function wal_get_gift_voucher( $id ) {

		$object = new WAL_Gift_Voucher($id);

		return $object;
	}

}

if (!function_exists('wal_update_gift_voucher')) {

	/**
	 * Update the gift voucher.
	 *
	 * @return object
	 */
	function wal_update_gift_voucher( $id, $meta_args, $post_args = array() ) {

		$object = new WAL_Gift_Voucher($id);
		$object->update($meta_args, $post_args);

		return $object;
	}

}

if (!function_exists('wal_delete_gift_voucher')) {

	/**
	 * Delete the gift voucher.
	 *
	 * @return bool
	 */
	function wal_delete_gift_voucher( $id, $force = true ) {

		wp_delete_post($id, $force);

		return true;
	}

}

if (!function_exists('wal_get_gift_voucher_statuses')) {

	/**
	 * Get the gift voucher statuses.
	 *
	 * @return array
	 */
	function wal_get_gift_voucher_statuses() {
		/**
		 * This hook is used to alter the gift voucher statuses.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_gift_voucher_statuses', array(
			'wal_unused',
			'wal_used',
			'wal_expired',
		));
	}

}

if (!function_exists('wal_get_auto_topup_id_by_wallet_id')) {

	/**
	 * Get the auto topup ID by wallet ID.
	 *
	 * @return int/bool
	 */
	function wal_get_auto_topup_id_by_wallet_id( $wallet_id, $status = false ) {

		if (!$status) {
			$status = wal_get_auto_topup_statuses();
		}

		$args = array(
			'post_type' => WAL_Register_Post_Types::AUTO_TOPUP_USERS_POSTTYPE,
			'post_status' => $status,
			'post_parent' => $wallet_id,
			'posts_per_page' => 1,
			'fields' => 'ids',
		);

		$auto_topup_id = get_posts($args);

		if (!wal_check_is_array($auto_topup_id)) {
			return false;
		}

		return reset($auto_topup_id);
	}

}

if (!function_exists('wal_user_fund_transfers')) {

	/**
	 * Get the user fund transfers.
	 *
	 * @return array
	 */
	function wal_user_fund_transfers( $user_id = false ) {
		if (!$user_id) {
			$user_id = get_current_user_id();
		}

		$post_args = array(
			'post_type' => WAL_Register_Post_Types::FUND_TRANSFER_POSTTYPE,
			'post_status' => wal_get_fund_transfer_statuses(),
			'fields' => 'ids',
			'numberposts' => '-1',
			'author' => $user_id,
		);

		return get_posts($post_args);
	}

}

if (!function_exists('wal_get_user_fund_transfered_particular_user_count')) {

	/**
	 * Get the user fund transfered count to specific user.
	 *
	 * @return int
	 */
	function wal_get_user_fund_transfered_particular_user_count( $receiver_id, $user_id = false ) {
		if (!$user_id) {
			$user_id = get_current_user_id();
		}

		$args = array(
			'post_type' => WAL_Register_Post_Types::FUND_TRANSFER_LOG_POSTTYPE,
			'post_status' => array( 'wal_transfered' ),
			'author' => $user_id,
			'posts_per_page' => -1,
			'fields' => 'ids',
			'date_query' => array(
				'column' => 'post_date_gmt',
				'day' => gmdate('d'),
				'year' => gmdate('Y'),
				'month' => gmdate('m'),
			),
			'meta_key' => 'wal_receiver_id',
			'meta_value' => $receiver_id,
		);

		$ids = get_posts($args);

		if (!wal_check_is_array($ids)) {
			return 0;
		}

		return count($ids);
	}

}

if (!function_exists('wal_get_user_fund_transfered_count')) {

	/**
	 * Get the user fund transferred count.
	 *
	 * @return int
	 */
	function wal_get_user_fund_transfered_count( $user_id = false ) {
		if (!$user_id) {
			$user_id = get_current_user_id();
		}

		$args = array(
			'post_type' => WAL_Register_Post_Types::FUND_TRANSFER_LOG_POSTTYPE,
			'post_status' => array( 'wal_transfered' ),
			'author' => $user_id,
			'posts_per_page' => -1,
			'fields' => 'ids',
			'date_query' => array(
				'column' => 'post_date_gmt',
				'day' => gmdate('d'),
				'year' => gmdate('Y'),
				'month' => gmdate('m'),
			),
		);

		$ids = get_posts($args);

		if (!wal_check_is_array($ids)) {
			return 0;
		}

		return count($ids);
	}

}

if (!function_exists('wal_get_user_fund_requested_particular_user_count')) {

	/**
	 * Get the user fund requested count to specific user.
	 *
	 * @return int
	 */
	function wal_get_user_fund_requested_particular_user_count( $receiver_id, $user_id = false ) {
		if (!$user_id) {
			$user_id = get_current_user_id();
		}

		$args = array(
			'post_type' => WAL_Register_Post_Types::FUND_TRANSFER_LOG_POSTTYPE,
			'post_status' => array( 'wal_requested' ),
			'author' => $user_id,
			'posts_per_page' => -1,
			'fields' => 'ids',
			'date_query' => array(
				'column' => 'post_date_gmt',
				'day' => gmdate('d'),
				'year' => gmdate('Y'),
				'month' => gmdate('m'),
			),
			'meta_key' => 'wal_receiver_id',
			'meta_value' => $receiver_id,
		);

		$ids = get_posts($args);

		if (!wal_check_is_array($ids)) {
			return 0;
		}

		return count($ids);
	}

}

if (!function_exists('wal_get_user_fund_requested_count')) {

	/**
	 * Get the user fund requested count.
	 *
	 * @return int
	 */
	function wal_get_user_fund_requested_count( $user_id = false ) {
		if (!$user_id) {
			$user_id = get_current_user_id();
		}

		$args = array(
			'post_type' => WAL_Register_Post_Types::FUND_TRANSFER_LOG_POSTTYPE,
			'post_status' => array( 'wal_requested' ),
			'author' => $user_id,
			'posts_per_page' => -1,
			'fields' => 'ids',
			'date_query' => array(
				'column' => 'post_date_gmt',
				'day' => gmdate('d'),
				'year' => gmdate('Y'),
				'month' => gmdate('m'),
			),
		);

		$ids = get_posts($args);

		if (!wal_check_is_array($ids)) {
			return 0;
		}

		return count($ids);
	}

}

if (!function_exists('wal_user_fund_transfer')) {

	/**
	 * User fund transfer to other users.
	 *
	 * @return int
	 */
	function wal_user_fund_transfer( $args ) {
		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'amount' => 0,
			'fee' => 0,
			'reason' => '',
			'sender_status' => 'wal_transfered',
			'receiver_status' => 'wal_received',
			'currency' => get_woocommerce_currency(),
		);

		$args = wp_parse_args($args, $default_args);
		
		$args['sender_transfer_id'] = wal_sender_fund_transfer($args);
		$args['receiver_transfer_id'] = wal_receiver_fund_transfer($args);
		$args['sender_transfer_log_id'] = wal_sender_fund_transfer_log($args);
		$args['receiver_transfer_log_id'] = wal_receiver_fund_transfer_log($args);

		//Update transfer log meta.
		update_post_meta($args['sender_transfer_log_id'], 'wal_receiver_log_id', $args['receiver_transfer_log_id']);
		update_post_meta($args['receiver_transfer_log_id'], 'wal_receiver_log_id', $args['sender_transfer_log_id']);

		$args['sender_transaction_id'] = wal_fund_debit_sender($args);
		$args['receiver_transaction_id'] = wal_fund_credit_receiver($args);
			   
		/**
		 * This hook is used to do extra action after user fund transferred.
		 * 
		 * @since 1.0
		 */
		do_action('wal_user_fund_transfered', $args);
	}

}

if (!function_exists('wal_user_fund_request')) {

	/**
	 * User fund request to other users.
	 *
	 * @return int
	 */
	function wal_user_fund_request( $args ) {

		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'amount' => 0,
			'reason' => '',
			'type' => 'request',
			'sender_status' => 'wal_requested',
			'receiver_status' => 'wal_request_received',
			'currency' => get_woocommerce_currency(),
		);

		$args = wp_parse_args($args, $default_args);

		$args['sender_transfer_id'] = wal_sender_fund_transfer($args);
		$args['receiver_transfer_id'] = wal_receiver_fund_transfer($args);
		$args['sender_transfer_log_id'] = wal_sender_fund_transfer_log($args);
		$args['receiver_transfer_log_id'] = wal_receiver_fund_transfer_log($args);

		//Update transfer log meta.
		update_post_meta($args['sender_transfer_log_id'], 'wal_receiver_log_id', $args['receiver_transfer_log_id']);
		update_post_meta($args['receiver_transfer_log_id'], 'wal_receiver_log_id', $args['sender_transfer_log_id']);
		/**
		 * This hook is used to do extra action after user fund requested.
		 * 
		 * @since 1.0
		 */
		do_action('wal_user_fund_requested', $args);
	}

}

if (!function_exists('wal_accept_fund_request')) {

	/**
	 * Accept the user fund request.
	 *
	 * @return int
	 */
	function wal_accept_fund_request( $args ) {

		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'amount' => 0,
			'fee' => 0,
			'sender_status' => 'wal_transfered',
			'receiver_status' => 'wal_received',
			'sender_log_id' => 0,
			'receiver_log_id' => 0,
			'currency' => get_woocommerce_currency(),
		);

		$args = wp_parse_args($args, $default_args);

		$args['sender_transfer_id'] = wal_sender_fund_transfer($args);
		$args['receiver_transfer_id'] = wal_receiver_fund_transfer($args);

		//Update sender transfer log.
		wal_update_fund_transfer_log($args['sender_log_id'], array(), array( 'post_status' => $args['sender_status'] ));
		//Update receiver transfer log.
		wal_update_fund_transfer_log($args['receiver_log_id'], array(), array( 'post_status' => $args['receiver_status'] ));

		$args['sender_transaction_id'] = wal_fund_debit_sender($args);
		$args['receiver_transaction_id'] = wal_fund_credit_receiver($args);
		/**
		 * This hook is used to do extra action after user fund request accepted.
		 * 
		 * @since 1.0
		 */
		do_action('wal_user_fund_request_accepted', $args);
	}

}

if (!function_exists('wal_decline_fund_request')) {

	/**
	 * Decline the user fund request.
	 *
	 * @return int
	 */
	function wal_decline_fund_request( $args ) {

		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'amount' => 0,
			'fee' => 0,
			'sender_status' => 'wal_request_declined',
			'receiver_status' => 'wal_request_declined',
			'sender_log_id' => 0,
			'receiver_log_id' => 0,
			'currency' => get_woocommerce_currency(),
			'type' => 'decline',
		);

		$args = wp_parse_args($args, $default_args);

		$args['sender_transfer_id'] = wal_sender_fund_transfer($args);
		$args['receiver_transfer_id'] = wal_receiver_fund_transfer($args);

		//Update sender transfer log.
		wal_update_fund_transfer_log($args['sender_log_id'], array(), array( 'post_status' => $args['sender_status'] ));
		//Update receiver transfer log.
		wal_update_fund_transfer_log($args['receiver_log_id'], array(), array( 'post_status' => $args['receiver_status'] ));
		/**
		 * This hook is used to do extra action after user fund request declined.
		 * 
		 * @since 1.0
		 */
		do_action('wal_user_fund_request_declined', $args);
	}

}

if (!function_exists('wal_cancel_fund_request')) {

	/**
	 * Cancel the user fund request.
	 *
	 * @return int
	 */
	function wal_cancel_fund_request( $args ) {

		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'amount' => 0,
			'fee' => 0,
			'sender_status' => 'wal_request_cancel',
			'receiver_status' => 'wal_request_cancel',
			'sender_log_id' => 0,
			'receiver_log_id' => 0,
			'currency' => get_woocommerce_currency(),
			'type' => 'cancel',
		);

		$args = wp_parse_args($args, $default_args);

		$args['sender_transfer_id'] = wal_sender_fund_transfer($args);
		$args['receiver_transfer_id'] = wal_receiver_fund_transfer($args);

		//Update sender transfer log.
		wal_update_fund_transfer_log($args['sender_log_id'], array(), array( 'post_status' => $args['sender_status'] ));
		//Update receiver transfer log.
		wal_update_fund_transfer_log($args['receiver_log_id'], array(), array( 'post_status' => $args['receiver_status'] ));
		/**
		 * This hook is used to do extra action after user fund request cancelled.
		 * 
		 * @since 1.0
		 */
		do_action('wal_user_fund_request_cancelled', $args);
	}

}

if (!function_exists('wal_sender_fund_transfer')) {

	/**
	 * Handle the sender fund transfer.
	 *
	 * @return int
	 */
	function wal_sender_fund_transfer( $args ) {

		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'amount' => 0,
			'sender_status' => 'wal_transfered',
			'receiver_status' => 'wal_received',
			'last_activity_date' => WAL_Date_Time::get_mysql_date_time_format('now', true),
			'currency' => get_woocommerce_currency(),
			'type' => 'transfer',
		);

		$args = wp_parse_args($args, $default_args);            
		$sender_transfer_id = wal_get_sender_fund_transfer_id($args['sender_id'], $args['receiver_id']);
		$meta_args = array( 'wal_last_activity_date' => $args['last_activity_date'] );
		
		//Check if the sender transfer id already exists.
		if ($sender_transfer_id) {
			if ('request' == $args['type']) {
				$meta_args['wal_requested_total'] = floatval(get_post_meta($sender_transfer_id, 'wal_requested_total', true)) + floatval($args['amount'] );
			} elseif ('transfer' == $args['type']) {                                        
				$meta_args['wal_transfered_total'] = floatval(get_post_meta($sender_transfer_id, 'wal_transfered_total', true)) + floatval($args['amount']); 
			}
			   
			$post_args = array( 'post_status' => $args['sender_status'] );
						
			//Update the sender transer details.
			wal_update_fund_transfer($sender_transfer_id, $meta_args, $post_args);
		} else {

			if ('request' == $args['type']) {
				$meta_args['wal_requested_total'] = floatval($args['amount']);
			} elseif ('transfer' == $args['type']) {
				$meta_args['wal_transfered_total'] = floatval($args['amount']);
			}
						 
			$post_args = array(
				'post_status' => $args['sender_status'],
				'post_author' => $args['sender_id'],
				'post_parent' => $args['receiver_id'],
			);

			// Create the new sender transfer id.
			$sender_transfer_id = wal_create_new_fund_transfer($meta_args, $post_args);
		}

		return $sender_transfer_id;
	}

}

if (!function_exists('wal_receiver_fund_transfer')) {

	/**
	 * Handle the receiver fund transfer.
	 *
	 * @return int
	 */
	function wal_receiver_fund_transfer( $args ) {

		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'amount' => 0,
			'receiver_status' => 'wal_received',
			'last_activity_date' => WAL_Date_Time::get_mysql_date_time_format('now', true),
			'currency' => get_woocommerce_currency(),
			'type' => 'transfer',
		);

		$args = wp_parse_args($args, $default_args);                
		$receiver_transfer_id = wal_get_sender_fund_transfer_id($args['receiver_id'], $args['sender_id']);
		$meta_args = array( 'wal_last_activity_date' => $args['last_activity_date'] );

		//Check if the receiver transfer id already exists.
		if ($receiver_transfer_id) {

			if ('transfer' == $args['type']) {
				$meta_args['wal_received_total'] = floatval(get_post_meta($receiver_transfer_id, 'wal_received_total', true)) + floatval($args['amount']);
			}
						
			$post_args = array( 'post_status' => $args['receiver_status'] );

			//Update the receiver transer details.
			wal_update_fund_transfer($receiver_transfer_id, $meta_args, $post_args);
		} else {

			if ('transfer' == $args['type']) {
				$meta_args['wal_received_total'] = floatval($args['amount']);
			}

			$post_args = array(
				'post_status' => $args['receiver_status'],
				'post_author' => $args['receiver_id'],
				'post_parent' => $args['sender_id'],
			);

			// Create the new receiver transfer id.
			$receiver_transfer_id = wal_create_new_fund_transfer($meta_args, $post_args);
		}

		return $receiver_transfer_id;
	}

}

if (!function_exists('wal_sender_fund_transfer_log')) {

	/**
	 * Handle the sender fund transfer log.
	 *
	 * @return int
	 */
	function wal_sender_fund_transfer_log( $args ) {

		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'receiver_transfer_id' => 0,
			'sender_transfer_id' => 0,
			'amount' => 0,
			'fee' => 0,
			'sender_status' => 'wal_transfered',
			'currency' => get_woocommerce_currency(),
			'reason' => '',
		);

		$args = wp_parse_args($args, $default_args);

		$meta_args = array(
			'wal_receiver_id' => $args['receiver_id'],
			'wal_receiver_transfer_id' => $args['receiver_transfer_id'],
			'wal_amount' => $args['amount'],
			'wal_fee' => $args['fee'],
			'wal_currency' => $args['currency'],
			'wal_reason' => $args['reason'],
			'wal_sent_by' => 'yes',
		);
				
		$post_args = array(
			'post_author' => $args['sender_id'],
			'post_parent' => $args['sender_transfer_id'],
			'post_status' => $args['sender_status'],
		);

		return wal_create_new_fund_transfer_log($meta_args, $post_args);
	}

}

if (!function_exists('wal_receiver_fund_transfer_log')) {

	/**
	 * Handle the receiver fund transfer log.
	 *
	 * @return int
	 */
	function wal_receiver_fund_transfer_log( $args ) {

		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'receiver_transfer_id' => 0,
			'sender_transfer_id' => 0,
			'amount' => 0,
			'fee' => 0,
			'receiver_status' => 'wal_received',
			'currency' => get_woocommerce_currency(),
			'reason' => '',
		);

		$args = wp_parse_args($args, $default_args);

		$meta_args = array(
			'wal_receiver_id' => $args['sender_id'],
			'wal_receiver_transfer_id' => $args['sender_transfer_id'],
			'wal_amount' => $args['amount'],
			'wal_fee' => $args['fee'],
			'wal_currency' => $args['currency'],
			'wal_reason' => $args['reason'],
			'wal_sent_by' => 'no',
		);
			   
		$post_args = array(
			'post_author' => $args['receiver_id'],
			'post_parent' => $args['receiver_transfer_id'],
			'post_status' => $args['receiver_status'],
		);

		return wal_create_new_fund_transfer_log($meta_args, $post_args);
	}

}

if (!function_exists('wal_fund_debit_sender')) {

	/**
	 * Fund debit from sender.
	 *
	 * @return int/boo;
	 */
	function wal_fund_debit_sender( $args ) {

		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'amount' => 0,
			'fee' => 0,
			'currency' => get_woocommerce_currency(),
			'transferred_total' => true,
		);

		$args = wp_parse_args($args, $default_args);              
		$amount = $args['amount'] + $args['fee'];  
				
		$log = get_option('wal_module_fund_transfer_log_message');
		$event_message = str_replace(array( '{amount}', '{user_name}', '{fee}' ), array( wal_price($amount), get_userdata($args['receiver_id'])->display_name, $args['fee'] ), $log);

		$post_args = array(
			'user_id' => $args['sender_id'],
			'amount' => $amount,
			'event_id' => 10,
			'event_message' => $event_message,
		);

		$transaction_id = wal_debit_wallet_fund($post_args);
				
		// Update transferred total.
		if ($transaction_id) {
			$wallet_id = wal_get_wallet_id_by_user_id($args['sender_id']);
			$transferred_total = floatval(get_post_meta($wallet_id, 'wal_transferred_total', true));
						
			update_post_meta($wallet_id, 'wal_transferred_total', $transferred_total + $args['amount']);

			return $transaction_id;
		}

		return false;
	}

}

if (!function_exists('wal_fund_credit_receiver')) {

	/**
	 * Fund credit to receiver.
	 *
	 * @return int/bool;
	 */
	function wal_fund_credit_receiver( $args ) {

		$default_args = array(
			'sender_id' => 0,
			'receiver_id' => 0,
			'amount' => 0,
			'fee' => 0,
			'currency' => get_woocommerce_currency(),
		);

		$args = wp_parse_args($args, $default_args);

		$amount = $args['amount'] + $args['fee']; 
				
		$log = get_option('wal_module_fund_received_log_message');
		$event_message = str_replace(array( '{amount}', '{user_name}' ), array( wal_price($amount), get_userdata($args['sender_id'])->display_name ), $log);

		$post_args = array(
			'user_id' => $args['receiver_id'],
			'amount' => $amount,
			'event_id' => 11,
			'event_message' => $event_message,
		);

		$transaction_id = wal_credit_wallet_fund($post_args);

		// Update received total.
		if ($transaction_id) {
			$wallet_id = wal_get_wallet_id_by_user_id($args['receiver_id']);
			$received_total = floatval(get_post_meta($wallet_id, 'wal_received_total', true));

			update_post_meta($wallet_id, 'wal_received_total', $received_total + $amount);

			return $transaction_id;
		}

		return false;
	}

}

if (!function_exists('wal_get_sender_fund_transfer_id')) {

	/**
	 * Get the sender fund transfer ID.
	 *
	 * @return int/bool
	 */
	function wal_get_sender_fund_transfer_id( $sender_id, $receiver_id ) {
		$args = array(
			'post_type' => WAL_Register_Post_Types::FUND_TRANSFER_POSTTYPE,
			'post_status' => wal_get_fund_transfer_statuses(),
			'author' => $sender_id,
			'post_parent' => $receiver_id,
			'posts_per_page' => 1,
			'fields' => 'ids',
		);

		$sender_transfer_id = get_posts($args);

		if (!wal_check_is_array($sender_transfer_id)) {
			return false;
		}

		return reset($sender_transfer_id);
	}

}

if (!function_exists('wal_get_fund_transfer_log_ids')) {

	/**
	 * Get the fund transfer log IDs.
	 *
	 * @return int/bool
	 */
	function wal_get_fund_transfer_log_ids( $fund_transfer_id ) {
		$args = array(
			'post_type' => WAL_Register_Post_Types::FUND_TRANSFER_LOG_POSTTYPE,
			'post_status' => wal_get_fund_transfer_log_statuses(),
			'post_parent' => $fund_transfer_id,
			'posts_per_page' => '-1',
			'fields' => 'ids',
		);

		$fund_transfer_log_ids = get_posts($args);

		if (!wal_check_is_array($fund_transfer_log_ids)) {
			return array();
		}

		return $fund_transfer_log_ids;
	}

}

if (!function_exists('wal_get_gift_voucher_id_by_code')) {

	/**
	 * Get the gift voucher ID by code.
	 *
	 * @return int/bool
	 */
	function wal_get_gift_voucher_id_by_code( $code, $status = false ) {
		if (!$status) {
			$status = wal_get_gift_voucher_statuses();
		}

		$args = array(
			'post_type' => WAL_Register_Post_Types::GIFT_VOUCHER_POSTTYPE,
			'post_status' => $status,
			'posts_per_page' => '1',
			'fields' => 'ids',
			'meta_key' => 'wal_code',
			'meta_value' => $code,
		);

		$gift_voucher_id = get_posts($args);

		if (!wal_check_is_array($gift_voucher_id)) {
			return false;
		}

		return reset($gift_voucher_id);
	}

}

if (!function_exists('wal_fund_transfer_fee_enabled')) {

	/**
	 * Check the fund transfer fee enabled?.
	 *
	 *  @return bool
	 */
	function wal_fund_transfer_fee_enabled() {
		/**
		 * This hook is used to check if the fund transfer fee is enabled.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_fund_transfer_fee_enabled', 'yes' == get_option('wal_module_fund_transfer_fee_enabled'));
	}

}

if (!function_exists('wal_fund_transfer_otp_enabled')) {

	/**
	 * Check the fund transfer OTP enabled?.
	 *
	 *  @return bool
	 */
	function wal_fund_transfer_otp_enabled() {
		/**
		 * This hook is used to check if the fund transfer OTP is enabled.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_fund_transfer_otp_enabled', 'yes' == get_option('wal_module_fund_transfer_otp_enabled'));
	}

}

if (!function_exists('wal_fund_transfer_enabled')) {

	/**
	 * Check the fund transfer enabled?.
	 *
	 *  @return bool
	 */
	function wal_fund_transfer_enabled() {
		/**
		 * This hook is used to check if the fund transfer is enabled.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_fund_transfer_enabled', 'yes' == get_option('wal_module_transfer_enabled'));
	}

}

if (!function_exists('wal_fund_request_enabled')) {

	/**
	 * Check the fund request enabled?.
	 *
	 *  @return bool
	 */
	function wal_fund_request_enabled() {
		/**
		 * This hook is used to check if the fund request is enabled.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_fund_request_enabled', 'yes' == get_option('wal_module_fund_request_enabled'));
	}

}

if (!function_exists('wal_get_gift_vouchers_page_url')) {

	/**
	 * Get the gift vouchers page URL.
	 *
	 * @return URL
	 */
	function wal_get_gift_vouchers_page_url( $args = array() ) {
		$url = add_query_arg(array( 'page' => 'wal_modules', 'module' => 'gift_voucher', 'tab' => 'gift_voucher' ), admin_url('admin.php'));

		if (wal_check_is_array($args)) {
			$url = add_query_arg($args, $url);
		}

		return $url;
	}

}

if (!function_exists('wal_create_new_wallet_withdrawal')) {

	/**
	 * Create a new wallet withdrawal.
	 *
	 * @since 1.8
	 * @return integer/string
	 */
	function wal_create_new_wallet_withdrawal( $meta_args, $post_args = array() ) {

		$object = new WAL_Wallet_Withdrawal();
		$id = $object->create($meta_args, $post_args);

		return $id;
	}

}

if (!function_exists('wal_get_wallet_withdrawal')) {

	/**
	 * Get the wallet withdrawal object.
	 *
	 * @since 1.8
	 * @return object
	 */
	function wal_get_wallet_withdrawal( $id ) {

		$object = new WAL_Wallet_Withdrawal($id);

		return $object;
	}

}

if (!function_exists('wal_update_wallet_withdrawal')) {

	/**
	 * Update the wallet withdrawal.
	 *
	 * @since 1.8
	 * @return object
	 */
	function wal_update_wallet_withdrawal( $id, $meta_args, $post_args = array() ) {

		$object = new WAL_Wallet_Withdrawal($id);
		$object->update($meta_args, $post_args);

		return $object;
	}

}

if (!function_exists('wal_delete_wallet_withdrawal')) {

	/**
	 * Delete the wallet withdrawal.
	 *
	 * @since 1.8
	 * @return bool
	 */
	function wal_delete_wallet_withdrawal( $id, $force = true ) {

		wp_delete_post($id, $force);

		return true;
	}

}

if (!function_exists('wal_get_wallet_withdrawal_statuses')) {

	/**
	 * Get the wallet withdrawal statuses.
	 *
	 * @since 1.8
	 * @return array
	 */
	function wal_get_wallet_withdrawal_statuses() {
		/**
		 * This hook is used to alter the wallet withdrawal statuses.
		 * 
		 * @since 1.8
		 * @param array
		 */
		return apply_filters('wal_wallet_withdrawal_statuses', array( 'wal_pending', 'wal_approved', 'wal_cancelled' ));
	}

}

if (!function_exists('wal_wallet_withdrawal_otp_enabled')) {

	/**
	 * Check the wallet withdrawal OTP enabled?.
	 *
	 * @since 1.8
	 * @return bool
	 */
	function wal_wallet_withdrawal_otp_enabled() {
		/**
		 * This hook is used to check if the wallet withdrawal OTP is enabled.
		 * 
		 * @since 1.8
		 * @param bool
		 */
		return apply_filters('wal_wallet_withdrawal_otp_enabled', 'yes' === get_option('wal_module_wallet_withdrawal_otp_enabled'));
	}

}

if (!function_exists('wal_get_wallet_withdrawal_payment_gateways_options')) {

	/**
	 * Get the wallet withdrawal payment gateways.
	 *
	 * @since 1.8
	 * @return array
	 */
	function wal_get_wallet_withdrawal_payment_gateways_options() {
		$options = array( '' => wal_get_withdrawal_request_payment_gateway_option_label() );
		$available_gateways = WAL_Module_Instances::get_module_by_id('wallet_withdrawal')->available_payment_gateways();

		if (wal_check_is_array($available_gateways)) {
			foreach ($available_gateways as $gateway_id => $gateway) {
				$options[$gateway_id] = $gateway->get_title();
			}
		}

		/**
		 * This hook is used to alter the wallet withdrawal payment gateways options.
		 * 
		 * @since 1.8
		 * @param array
		 */
		return apply_filters('wal_wallet_withdrawal_payment_gateways_options', $options);
	}

}


if (!function_exists('wal_unset_global_variable_values')) {

	/**
	 * Unset the global variable values.
	 * 
	 * @since 1.8
	 * @param array
	 */
	function wal_unset_global_variable_values( $keys ) {

		foreach ($keys as $key) {

			// Request key.
			if (isset($_REQUEST[$key])) {
				unset($_REQUEST[$key]);
			}
		}
	}

}

if (!function_exists('wal_user_wallet_withdrawals')) {

	/**
	 * Get the user wallet withdrawals.
	 * 
	 * @since 1.8
	 * @return array
	 */
	function wal_user_wallet_withdrawals( $wallet_id = false ) {
		if (!$wallet_id) {
			$wallet_id = WAL_Current_User_Wallet::get_wallet_id();
		}

		$post_args = array(
			'post_type' => WAL_Register_Post_Types::WALLET_WITHDRAWAL_POSTTYPE,
			'post_status' => wal_get_wallet_withdrawal_statuses(),
			'fields' => 'ids',
			'numberposts' => '-1',
			'post_parent' => $wallet_id,
		);

		return get_posts($post_args);
	}

}

if (!function_exists('wal_get_user_wallet_withdrew_count')) {

	/**
	 * Get the user wallet withdrew count.
	 *
	 * @since 1.8
	 * @return int
	 */
	function wal_get_user_wallet_withdrew_count( $wallet_id = false ) {
		if (!$wallet_id) {
			$wallet_id = WAL_Current_User_Wallet::get_wallet_id();
		}

		$args = array(
			'post_type' => WAL_Register_Post_Types::WALLET_WITHDRAWAL_POSTTYPE,
			'post_status' => wal_get_wallet_withdrawal_statuses(),
			'post_parent' => $wallet_id,
			'posts_per_page' => -1,
			'fields' => 'ids',
			'date_query' => array(
				'column' => 'post_date_gmt',
				'day' => gmdate('d'),
				'year' => gmdate('Y'),
				'month' => gmdate('m'),
			),
		);

		$ids = get_posts($args);
		if (!wal_check_is_array($ids)) {
			return 0;
		}

		return count($ids);
	}

}

if (!function_exists('wal_wallet_withdrawal_pagination_count')) {

	/**
	 * Get the wallet withdrawal pagination count.
	 *
	 * @since 1.8
	 * @return string
	 */
	function wal_wallet_withdrawal_pagination_count() {
		/**
		 * This hook is used to alert the wallet withdrawal pagination count.
		 * 
		 * @since 1.8
		 * @param string
		 */
		return apply_filters('wal_wallet_withdrawal_pagination_count', get_option('wal_module_wallet_withdrawal_transactions_pagination_count', 5));
	}

}

if (!function_exists('wal_approve_wallet_withdrawal')) {

	/**
	 * Approve the wallet withdrawal.
	 * 
	 * @since 1.8
	 * @return int/bool
	 */
	function wal_approve_wallet_withdrawal( $withdrawal_id, $force = false ) {
		if (!$withdrawal_id) {
			return false;
		}

		$withdrawal = wal_get_wallet_withdrawal($withdrawal_id);
		if (!$withdrawal->exists()) {
			return false;
		}

		if ($withdrawal->has_status('wal_approved') || ( !$force && !$withdrawal->has_status('wal_pending') )) {
			return false;
		}

		$gateway = WAL_Module_Instances::get_module_by_id('wallet_withdrawal')->get_payment_gateway_by_id($withdrawal->get_payment_method());
		if (!$gateway) {
			return false;
		}

		$response = $gateway->process_payment($withdrawal);
		if (!$response || is_wp_error($response)) {
			return false;
		}

		$withdrawal->update_status('wal_approved');
		/**
		 * This hook is used to do extra action after wallet withdrawal approved.
		 * 
		 * @since 1.8
		 * @param int $transaction_id
		 * @param int $withdrawal_id
		 * @param object $withdrawal
		 */
		do_action('wal_wallet_withdrawal_request_approved', $withdrawal_id, $withdrawal);

		return false;
	}

}

if (!function_exists('wal_cancel_wallet_withdrawal')) {

	/**
	 * Cancel the wallet withdrawal.
	 * 
	 * @param int $withdrawal_id
	 * @param string $reason
	 * @param boolean $force
	 * @return boolean/int
	 */
	function wal_cancel_wallet_withdrawal( $withdrawal_id, $reason = false, $force = false ) {
		if (!$withdrawal_id) {
			return false;
		}

		$withdrawal = wal_get_wallet_withdrawal($withdrawal_id);
		if (!$withdrawal->exists()) {
			return false;
		}

		if ('wal_cancelled' === $withdrawal->get_status() || ( !$force && !$withdrawal->has_status('wal_pending') ) ) {
			return false;
		}

		$transaction_id = wal_wallet_withdrawal_credit_fund($withdrawal_id);
		if ($transaction_id) {
			if ($reason) {
				$withdrawal->update_meta('wal_withdrawal_cancelled_reason', $reason);
			}

			$withdrawal->update_status('wal_cancelled');

			/**
			 * This hook is used to do extra action after wallet withdrawal cancelled.
			 * 
			 * @since 1.8
			 * @param int $transaction_id
			 * @param int $withdrawal_id
			 * @param object $withdrawal
			 */
			do_action('wal_wallet_withdrawal_request_cancelled', $withdrawal_id, $withdrawal);

			return $transaction_id;
		}

		return false;
	}

}

if (!function_exists('wal_wallet_withdrawal_gateway_title_by_id')) {

	/**
	 * Get the wallet withdrawal gateway title by ID.
	 *
	 * @since 1.8
	 * @param int $gateway_id
	 * @return string
	 */
	function wal_wallet_withdrawal_gateway_title_by_id( $gateway_id ) {
		$gateway = WAL_Module_Instances::get_module_by_id('wallet_withdrawal')->get_payment_gateway_by_id($gateway_id);
		if (!$gateway) {
			return '';
		}

		return $gateway->get_title();
	}

}

if (!function_exists('wal_wallet_withdrawal_credit_fund')) {

	/**
	 * Wallet withdrawal credit amount.
	 * 
	 * @since 1.8
	 * @return int/bool
	 */
	function wal_wallet_withdrawal_credit_fund( $withdrawal_id ) {
		if (!$withdrawal_id) {
			return false;
		}

		$withdrawal = wal_get_wallet_withdrawal($withdrawal_id);
		if (!$withdrawal->exists()) {
			return false;
		}
					
		$post_args = array(
			'user_id' => $withdrawal->get_wallet()->get_user_id(),
			'amount' => $withdrawal->get_total(),
			'event_id' => 19,
			'event_message' => get_option('wal_module_wallet_withdrawal_credit_log_message'),
		);
			   
		$transaction_id = wal_credit_wallet_fund($post_args);
		if ($transaction_id) {
			// Update withdrew total.
			$withdrew_total = floatval($withdrawal->get_wallet()->get_withdrew_total()) - $withdrawal->get_amount();
			$total = ( $withdrew_total < 0 ) ? 0 : $withdrew_total;

			$withdrawal->get_wallet()->update_meta('wal_withdrew_total', $total);
		}

		return $transaction_id;
	}

}

if (!function_exists('wal_wallet_withdrawal_debit_fund')) {

	/**
	 * Wallet withdrawal debit amount.
	 * 
	 * @since 1.8
	 * @return int/bool
	 */
	function wal_wallet_withdrawal_debit_fund( $withdrawal_id ) {
		if (!$withdrawal_id) {
			return false;
		}

		$withdrawal = wal_get_wallet_withdrawal($withdrawal_id);
		if (!$withdrawal->exists()) {
			return false;
		}

		$post_args = array(
			'user_id' => $withdrawal->get_wallet()->get_user_id(),
			'amount' => $withdrawal->get_total(),
			'event_id' => 18,
			'event_message' => get_option('wal_module_wallet_withdrawal_debit_log_message'),
		);
			
		$transaction_id = wal_debit_wallet_fund($post_args);
		if ($transaction_id) {
			// Update withdrew total.
			$withdrew_total = floatval($withdrawal->get_wallet()->get_withdrew_total());
			$withdrawal->get_wallet()->update_meta('wal_withdrew_total', $withdrew_total + $withdrawal->get_amount());
		}

		return $transaction_id;
	}

}

if (!function_exists('wal_get_wallet_withdrawal_edit_post_stauses')) {

	/**
	 * Get the wallet withdrawal edit post statuses.
	 * 
	 * @since 1.8
	 * @return array
	 */
	function wal_get_wallet_withdrawal_edit_post_stauses( $status ) {

		$statuses = array(
			'wal_pending' => __('Pending', 'wallet-for-woocommerce'),
			'wal_approved' => __('Approved', 'wallet-for-woocommerce'),
			'wal_cancelled' => __('Cancelled', 'wallet-for-woocommerce'),
		);

		switch ($status) {
			case 'wal_approved':
				$statuses = array(
					'wal_approved' => __('Approved', 'wallet-for-woocommerce'),
				);
				break;
			case 'wal_cancelled':
				$statuses = array(
					'wal_cancelled' => __('Cancelled', 'wallet-for-woocommerce'),
				);
				break;
		}

		return $statuses;
	}

}

if (!function_exists('wal_format_payment_gateway_credentials')) {

	/**
	 * Format payment gateway credentials.
	 * 
	 * @since 1.8
	 * @param array $credentials
	 * @param array $credential_labels
	 * @return int/bool
	 */
	function wal_format_payment_gateway_credentials( $credentials, $credential_labels ) {
		$formatted_credentials = array();

		if (wal_check_is_array($credential_labels)) {
			foreach ($credential_labels as $id => $label) {
				$formatted_credentials[$id] = array(
					'label' => $label,
					'value' => isset($credentials[$id]) ? $credentials[$id] : '',
				);
			}
		}

		return $formatted_credentials;
	}

}

if (!function_exists('wal_get_pending_wallet_withdrawal_ids')) {

	/**
	 * Get the pending wallet withdrawal ids.
	 *
	 * @since 2.8.0
	 * @return int
	 */
	function wal_get_pending_wallet_withdrawal_ids() {
		$args = array(
			'post_type' => WAL_Register_Post_Types::WALLET_WITHDRAWAL_POSTTYPE,
			'post_status' => 'wal_pending',
			'fields' => 'ids',
			'numberposts' => -1,
			'meta_key' => '_edit_lock',
			'meta_compare' => 'NOT EXISTS',
		);

		$ids = get_posts($args);
		if (!wal_check_is_array($ids)) {
			return 0;
		}

		return count($ids);
	}

}
