<?php

/**
 * Front end functions.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

include_once 'wal-template-functions.php';
include_once 'wal-module-template-functions.php';

if (!function_exists('wal_get_wc_cart_subtotal')) {

	/**
	 * Get the WC cart Subtotal.
	 *
	 * @return string/float
	 */
	function wal_get_wc_cart_subtotal() {

		if (method_exists(WC()->cart, 'get_subtotal')) {
			$subtotal = ( 'incl' == get_option('woocommerce_tax_display_cart') ) ? WC()->cart->get_subtotal() + WC()->cart->get_subtotal_tax() : WC()->cart->get_subtotal();
		} else {
			$subtotal = ( 'incl' == get_option('woocommerce_tax_display_cart') ) ? WC()->cart->subtotal + WC()->cart->subtotal_tax : WC()->cart->subtotal;
		}

		return $subtotal;
	}

}

if (!function_exists('wal_get_wc_cart_total')) {

	/**
	 * Get the WC cart total.
	 *
	 * @return string/float
	 */
	function wal_get_wc_cart_total( $exclude_shipping = false, $exclude_tax = false ) {
		$cart_obj = WC()->cart;
		$exclude_shipping_cost = get_option('wal_general_exclude_shipping_for_fund_usage');
		$exclude_tax_cost = get_option('wal_general_exclude_tax_for_fund_usage');
		if ( $exclude_tax && ( 'yes' == $exclude_tax_cost ) ) {
			$shipping_tax_cost = 0;
			$tax_cost = 0;
			$fee_tax = 0;
		} else {
			$shipping_tax_cost = $cart_obj->get_shipping_tax();
			$tax_cost = $cart_obj->get_cart_contents_tax();
			$fee_tax = $cart_obj->get_fee_tax();
		}

		$shipping_amount = ( $exclude_shipping && ( 'yes' == $exclude_shipping_cost ) ) ? 0 : $cart_obj->get_shipping_total() + $shipping_tax_cost;
		$total = $cart_obj->get_cart_contents_total() + $tax_cost + $shipping_amount + $cart_obj->get_fee_total() + $fee_tax;

		return $total;
	}

}

if (!function_exists('wal_cart_topup_product_exists')) {

	/**
	 * Exists top up product in the cart?
	 *
	 * @return bool.
	 */
	function wal_cart_topup_product_exists() {
		$bool = false;
		if (!is_object(WC()->cart)) {
			return $bool;
		}

		foreach (WC()->cart->get_cart() as $key => $value) {
			if (!isset($value['wal_topup'])) {
				continue;
			}

			$bool = true;
		}

		return $bool;
	}

}

if (!function_exists('wal_topup_products_only_exists_in_cart')) {

	/**
	 * To check if the cart contains only top up products.
	 * 
	 * @since 2.0
	 * 
	 * @return boolean
	 */
	function wal_topup_products_only_exists_in_cart() {
		if (!is_object(WC()->cart)) {
			return false;
		}

		$bool = true;
		foreach (WC()->cart->get_cart() as $key => $value) {
			// Continue if the product is a topup product.
			if (isset($value['wal_topup'])) {
				continue;
			}

			$bool = false;
			break;
		}

		return $bool;
	}

}

if (!function_exists('wal_get_price_to_display')) {

	/**
	 * Returns the fomatted price including or excluding tax, based on the 'woocommerce_tax_display_cart' setting.
	 *
	 * @return float
	 */
	function wal_get_price_to_display( $product, $price, $qty = 1 ) {

		return wal_price(wal_get_cart_taxable_price($product, $price, $qty));
	}

}

if (!function_exists('wal_get_cart_taxable_price')) {

	/**
	 * Returns the cart taxable price including or excluding tax, based on the 'woocommerce_tax_display_cart' setting.
	 *
	 * @return float
	 */
	function wal_get_cart_taxable_price( $product, $price, $qty = 1 ) {

		if (empty($price)) {
			return 0;
		}

		$args = array(
			'qty' => 1,
			'price' => $price,
		);

		if (wal_display_prices_including_tax()) {
			$product_price = wc_get_price_including_tax($product, $args);
		} else {
			$product_price = wc_get_price_excluding_tax($product, $args);
		}

		return $product_price;
	}

}

if (!function_exists('wal_display_prices_including_tax')) {

	/**
	 * Check the display prices including tax.
	 *
	 * @return bool
	 */
	function wal_display_prices_including_tax() {
		$tax_display_cart = get_option('woocommerce_tax_display_cart');

		if (is_object(WC()->cart) && method_exists(WC()->cart, 'display_prices_including_tax')) {
			return WC()->cart->display_prices_including_tax();
		}

		return 'incl' == $tax_display_cart;
	}

}

if (!function_exists('wal_dashboard_menu_endpoint_url')) {

	/**
	 * Get the dashboard menu endpoint URL.
	 *
	 * @return string
	 */
	function wal_dashboard_menu_endpoint_url( $endpoint ) {
		$endpoints = wal_wallet_endpoints();
		$customize_endpoint = isset($endpoints[$endpoint]) ? $endpoints[$endpoint] : $endpoint;

		$url = wc_get_endpoint_url($customize_endpoint, '', wc_get_page_permalink('wal_dashboard'));
		/**
		 * This hook is used to alter the dashboard menu endpoint URL.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_dashboard_menu_endpoint_url', $url, $endpoint);
	}

}

if (!function_exists('wal_wallet_endpoints')) {

	/**
	 * Get the wallet endpoints.
	 *
	 * @return array
	 */
	function wal_wallet_endpoints() {
		static $wal_dashboard_endpoints;
		if (!is_null($wal_dashboard_endpoints)) {
			return $wal_dashboard_endpoints;
		}

		/**
		 * This hook is used to alter the wallet endpoints.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_wallet_endpoints', array(
			'overview' => get_option('wal_localization_dashboard_overview_endpoint'),
			'activity' => get_option('wal_localization_dashboard_activity_endpoint'),
			'topup' => get_option('wal_localization_dashboard_topup_endpoint'),
		));
	}

}

if (!function_exists('wal_dashboard_menus')) {

	/**
	 * Get the dashboard menus.
	 *
	 * @return string
	 */
	function wal_dashboard_menus() {
		static $wal_dashboard_menus;
		if (!is_null($wal_dashboard_menus)) {
			return $wal_dashboard_menus;
		}

		/**
		 * This hook is used to alter the dashboard menus.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_dashboard_menus', array(
			'overview' => array(
				'name' => get_option('wal_localization_dashboard_overview_menu_label'),
				'endpoint' => get_option('wal_localization_dashboard_overview_endpoint'),
			),
			'activity' => array(
				'name' => get_option('wal_localization_dashboard_activity_menu_label'),
				'endpoint' => get_option('wal_localization_dashboard_activity_endpoint'),
			),
			'topup' => array(
				'name' => get_option('wal_localization_dashboard_topup_form_menu_label'),
				'endpoint' => get_option('wal_localization_dashboard_topup_endpoint'),
			),
		));
	}

}

if (!function_exists('wal_dashboard_menu_item_classes')) {

	/**
	 * Get the dashboard menu item classes.
	 *
	 * @return string
	 */
	function wal_dashboard_menu_item_classes( $endpoint, $current_endpoint, $dashboard_menu ) {
		global $wp;

		$endpoints = wal_wallet_endpoints();
		$customize_endpoint = isset($endpoints[$endpoint]) ? $endpoints[$endpoint] : '';

		$classes = array(
			'wal-dashboard-menu-item',
			'wal-dashboard-menu-item-' . $customize_endpoint,
		);

		// Set current item.
		if (isset($wp->query_vars[$customize_endpoint])) {
			$classes[] = 'wal-active';
		} elseif (isset($dashboard_menu['submenus'][$current_endpoint])) {
			$classes[] = 'wal-active';
		}
		/**
		 * This hook is used to alter the dashboard menu item classes.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_dashboard_menu_item_classes', implode(' ', $classes), $endpoint, $dashboard_menu);
	}

}


if (!function_exists('wal_dashboard_submenu_item_classes')) {

	/**
	 * Get the dashboard sub menu item classes.
	 *
	 * @return string
	 */
	function wal_dashboard_submenu_item_classes( $endpoint ) {
		global $wp;

		$endpoints = wal_wallet_endpoints();
		$customize_endpoint = isset($endpoints[$endpoint]) ? $endpoints[$endpoint] : '';

		$classes = array(
			'wal-dashboard-submenu-item',
			'wal-dashboard-submenu-item-' . $customize_endpoint,
		);

		// Set current item.
		if (isset($wp->query_vars[$customize_endpoint])) {
			$classes[] = 'wal-active';
		}
		/**
		 * This hook is used to alter the dashboard submenu item classes.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_dashboard_submenu_item_classes', implode(' ', $classes), $endpoint);
	}

}

if (!function_exists('wal_print_dashboard_notices')) {

	/**
	 * Print the dashboard notices.
	 *
	 * @return array
	 */
	function wal_print_dashboard_notices() {
		return WAL_Dashboard_Handler::show_messages();
	}

}

if (!function_exists('wal_number_field_step_value')) {

	/**
	 * Get the number field step value.
	 *
	 * @return string/int
	 */
	function wal_number_field_step_value() {
		$decimals = wc_get_price_decimals();
		$step_value = 1;

		if ($decimals) {
			$num_value = 1;
			for ($i = 1; $i <= $decimals; $i++) {
				$num_value = $num_value * 10;
			}

			$step_value = 1 / $num_value;
		}

		return $step_value;
	}

}

if (!function_exists('wal_get_maximum_fund_usage_limit')) {

	/**
	 * Get the limit of maximum fund usage in an order.
	 *
	 * @return bool/float
	 */
	function wal_get_maximum_fund_usage_limit() {
		$amount = false;
		$value = floatval(get_option('wal_general_partial_payments_max_amount'));
		// Consider unlimited if the value is not set.
		if (!$value) {
			return $amount;
		}

		$type = get_option('wal_general_partial_payments_max_amount_type');
		switch ($type) {
			// Percentage of wallet balance.
			case '2':
				$cart_total = wal_convert_price(wal_get_wc_cart_total(), true);
				// consider the amount is zero if the cart total is zero.
				$amount = ( $cart_total ) ? ( $value / 100 ) * $cart_total : 0;
				break;
			// Fixed price.
			default:
				$amount = $value;
				break;
		}

		return $amount;
	}

}

if (!function_exists('wal_get_wallet_myaccount_menu_position')) {

	/**
	 * Get the wallet MyAccount menu position.
	 *
	 * @return string
	 */
	function wal_get_wallet_myaccount_menu_position() {
		return get_option('wal_general_wallet_myaccount_menu_position');
	}

}

if (!function_exists('wal_can_render_redeem_wallet_fund_form_in_cart')) {

	/**
	 * Can render redeem wallet fund form in the cart.
	 * 
	 * @since 3.7.0
	 * @return bool
	 */
	function wal_can_render_redeem_wallet_fund_form_in_cart() {
		// Return if the cart object is not initialized.
		if (!is_object(WC()->cart)) {
			return false;
		}

		// Return if the session object is not initialized.
		if (!is_object(WC()->session)) {
			return false;
		}

		// Return if the cart is empty.
		if (WC()->cart->get_cart_contents_count() == 0) {
			return false;
		}

		// Return if the topup product is in the cart.
		if (wal_cart_topup_product_exists()) {
			return false;
		}

		// Return if the partial fund is disabled.
		if ('yes' != get_option('wal_general_allow_partial_payments')) {
			return false;
		}

		// Return if the cart page partial fund is disabled.
		if ('yes' != get_option('wal_general_allow_partial_payments_cart_page')) {
			return false;
		}

		// Return if the partial fund already applied.
		if (WC()->session->get('wal_partial_fund')) {
			return false;
		}

		// Return if the current wallet not valid fund usage.
		if (!WAL_Current_User_Wallet::is_valid_fund_usage()) {
			return false;
		}

		//Return if the wallet balance is less than order total.
		if ('2' != get_option('wal_general_partial_payments_mode') && wal_convert_price(WAL_Current_User_Wallet::get_balance()) >= wal_get_wc_cart_total()) {
			return false;
		}

		/**
		 * This hook is used to validate the partial fund form display.
		 * 
		 * @since 3.7.0
		 */
		return apply_filters('wal_can_render_redeem_wallet_fund_form_in_checkout', true);
	}

}

if (!function_exists('wal_can_render_redeem_wallet_fund_form_in_checkout')) {

	/**
	 * Can render redeem wallet fund form in the checkout.
	 * 
	 * @since 3.7.0
	 * @return bool
	 */
	function wal_can_render_redeem_wallet_fund_form_in_checkout() {
		// Return if the cart object is not initialized.
		if (!is_object(WC()->cart)) {
			return false;
		}

		// Return if the session object is not initialized.
		if (!is_object(WC()->session)) {
			return false;
		}

		// Return if the cart is empty.
		if (WC()->cart->get_cart_contents_count() == 0) {
			return false;
		}

		// Return if the topup product is in the cart.
		if (wal_cart_topup_product_exists()) {
			return false;
		}

		// Return if the partial fund is disabled.
		if ('yes' != get_option('wal_general_allow_partial_payments')) {
			return false;
		}

		// Return if the checkout page partial fund is disabled.
		if ('yes' != get_option('wal_general_allow_partial_payments_checkout_page')) {
			return false;
		}

		// Return if the partial fund already applied.
		if (WC()->session->get('wal_partial_fund')) {
			return false;
		}

		// Return if the current wallet not valid fund usage.
		if (!WAL_Current_User_Wallet::is_valid_fund_usage()) {
			return false;
		}

		//Return if the wallet balance is less than order total
		if ('2' != get_option('wal_general_partial_payments_mode') && wal_convert_price(WAL_Current_User_Wallet::get_balance()) >= wal_get_wc_cart_total()) {
			return false;
		}

		/**
		 * This hook is used to validate the partial fund form display.
		 * 
		 * @since 3.7.0
		 */
		return apply_filters('wal_can_render_redeem_wallet_fund_form_in_checkout', true);
	}

}

if (!function_exists('wal_add_wc_notice')) {

	/**
	 * Add a WC notice.
	 * 
	 * @since 3.7.0
	 * @param string $message
	 * @param string $notice_type
	 * @param array $data
	 */
	function wal_add_wc_notice( $message, $notice_type = 'success', $data = array() ) {
		if (wal_is_block_cart() || wal_is_block_checkout()) {
			return;
		}

		wc_add_notice($message, $notice_type, $data);
	}

}

if (!function_exists('wal_is_cart')) {

	/**
	 * Is a cart page?.
	 *
	 * @since 3.7.0
	 * @return boolean
	 */
	function wal_is_cart() {
		if (is_cart()) {
			return true;
		}

		return wal_is_block_cart();
	}

}

if (!function_exists('wal_is_checkout')) {

	/**
	 * Is a checkout page?.
	 *
	 * @since 3.7.0
	 * @return boolean
	 */
	function wal_is_checkout() {
		if (is_checkout()) {
			return true;
		}

		return wal_is_block_checkout();
	}

}

if (!function_exists('wal_get_category_name')) {

	/**
	 * Get Category Name
	 *
	 * @since 5.1.0
	 * @return string
	 */
	function wal_get_category_name( $category_id ) {
		$term = get_term($category_id);

		return is_object($term) ? $term->name : '';
	}

}

if (!function_exists('wal_array_filter')) {

	/**
	 * Get Category Name
	 *
	 * @since 5.1.0
	 */
	function wal_array_filter( $value ) {
		return ! empty( $value );
	}

}
