<?php

/**
 * Core functions.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Event ID Details.
 * 
 * 1  - Manual Credit.
 * 2  - Manual Debit.
 * 3  - Topup Credit.
 * 4  - Topup Debit.
 * 5  - Partial Fund Credit.
 * 6  - Partial Fund Debit.
 * 7  - Gateway Fund Credit.
 * 8  - Gateway Fund Debit.
 * 9  - Expired Full Amount Debit.
 * 10 - Fund transfer Debit.
 * 11 - Fund receive Credit.
 * 12 - Auto Topup Credit.
 * 13 - Auto Topup Debit.
 * 14 - Gift Voucher Credit.
 * 15 - Signup Action Credit.
 * 16 - Daily Login Action Credit.
 * 17 - Product Review Action Credit.
 * 18 - Wallet Withdraw Debit.
 * 19 - Wallet Withdraw Credit.
 * 20 - Wallet Refund Credit.
 * 21 - Topup Bonus Credit.
 * 22 - Topup Bonus Debit.
 * 23 - Product Purchase Credit.
 * 24 - Product Purchase Debit.
 */
if (!function_exists('wal_get_wallet_current_expiry_date')) {

	/**
	 * Get the wallet current expiry date.
	 *
	 * @return string
	 */
	function wal_get_wallet_current_expiry_date() {
		$expiry_date = '';
		$expiry_days = get_option('wal_general_wallet_balance_expired_days');

		if (!empty($expiry_days)) {
			// Get the current UTC time.
			$date_object = WAL_Date_Time::get_date_time_object('now', true);
			$date_object->modify('+ ' . $expiry_days . ' days');
			$expiry_date = $date_object->format('Y-m-d H:i:s');
		}
		/**
		 * This hook is used to alter the wallet expiry day.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_wallet_expiry_day', $expiry_date, $expiry_days);
	}

}

if (!function_exists('wal_credit_wallet_fund')) {

	/**
	 * Credit the amount to user wallet.
	 *
	 * @return int
	 */
	function wal_credit_wallet_fund( $args ) {

		$default_args = array(
			'user_id' => get_current_user_id(),
			'order_id' => '',
			'amount' => 0,
			'event_id' => 1,
			'event_message' => '',
			'currency' => get_woocommerce_currency(),
			'update_usage_total' => false,
			'update_topup_total' => false,
			'mode' => 'automatic',
		);

		$args = wp_parse_args($args, $default_args);

		//Get the wallet ID by user ID.
		$wallet_id = wal_get_wallet_id_by_user_id($args['user_id']);

		// Update the wallet details if the wallet ID exists. 
		if ($wallet_id) {
			$wallet = wal_get_wallet($wallet_id);

			$meta_args = array(
				'wal_balance' => floatval($wallet->get_balance()) + floatval($args['amount']),
				'wal_current_expiry_date' => wal_get_wallet_current_expiry_date(),
				'wal_previous_expiry_date' => $wallet->get_current_expiry_date(),
			);

			if ($args['update_usage_total']) {
				$meta_args['wal_usage_total'] = floatval($wallet->get_usage_total()) - floatval($args['amount']);
			}

			if ($args['update_topup_total']) {
				$meta_args['wal_topup_total'] = floatval($wallet->get_topup_total()) + floatval($args['amount']);
			}

			wal_update_wallet($wallet_id, $meta_args, array( 'post_status' => 'wal_active' ));
		} else {
			// Create the wallet ID by user ID if wallet ID does not exists. 
			$meta_args = array(
				'wal_balance' => $args['amount'],
				'wal_current_expiry_date' => wal_get_wallet_current_expiry_date(),
				'wal_currency' => get_woocommerce_currency(),
			);

			if ($args['update_topup_total']) {
				$meta_args['wal_topup_total'] = $args['amount'];
			}

			$wallet_id = wal_create_new_wallet($meta_args, array( 'post_parent' => $args['user_id'] ));
		}

		//Insert Transaction log
		$transaction_meta_args = array(
			'wal_user_id' => $args['user_id'],
			'wal_order_id' => $args['order_id'],
			'wal_event_id' => $args['event_id'],
			'wal_event_message' => $args['event_message'],
			'wal_amount' => $args['amount'],
			'wal_total' => $meta_args['wal_balance'],
			'wal_currency' => $args['currency'],
		);

		$transaction_log_id = wal_create_new_transaction_log($transaction_meta_args, array( 'post_parent' => $wallet_id, 'post_status' => 'wal_credit' ));

		//Update last wallet activity.
		update_post_meta($wallet_id, 'wal_last_activity_date', WAL_Date_Time::get_mysql_date_time_format('now', true));

		$transaction_log = wal_get_transaction_log($transaction_log_id);
		if ($transaction_log->has_status('wal_credit')) {
			/**
			 * This hook is used to do extra action after the wallet amount credited.
			 * 
			 * @since 1.0
			 */
			do_action('wal_wallet_amount_credited', $transaction_log_id, $transaction_log, $args['mode']);

			return $transaction_log_id;
		} else {
			return false;
		}
	}

}

if (!function_exists('wal_debit_wallet_fund')) {

	/**
	 * Debit the amount from the user wallet.
	 *
	 * @return int
	 */
	function wal_debit_wallet_fund( $args ) {
		$default_args = array(
			'user_id' => get_current_user_id(),
			'order_id' => '',
			'amount' => 0,
			'event_id' => 2,
			'event_message' => '',
			'currency' => get_woocommerce_currency(),
			'update_usage_total' => false,
			'update_topup_total' => false,
			'update_wallet' => true,
			'mode' => 'automatic',
		);

		$args = wp_parse_args($args, $default_args);

		//Get the wallet ID by user ID.
		$wallet_id = wal_get_wallet_id_by_user_id($args['user_id']);

		// Return false if the wallet does not exists for the user.
		if (!$wallet_id) {
			return false;
		}

		// Update the wallet details if the wallet ID exists. 
		if ($args['update_wallet']) {
			$wallet = wal_get_wallet($wallet_id);
			
			$meta_args = array(
				'wal_balance' => floatval($wallet->get_balance()) - floatval($args['amount']),
			);

			if ($args['update_usage_total']) {
				$meta_args['wal_usage_total'] = floatval($wallet->get_usage_total()) + floatval($args['amount']);
			}

			if ($args['update_topup_total']) {
				$meta_args['wal_topup_total'] = floatval($wallet->get_topup_total()) - floatval($args['amount']);
			}

			wal_update_wallet($wallet_id, $meta_args);
		}

		//Insert Transaction log
		$transaction_meta_args = array(
			'wal_user_id' => $args['user_id'],
			'wal_order_id' => $args['order_id'],
			'wal_event_id' => $args['event_id'],
			'wal_event_message' => $args['event_message'],
			'wal_amount' => $args['amount'],
			'wal_total' => $meta_args['wal_balance'],
			'wal_currency' => $args['currency'],
		);

		$transaction_log_id = wal_create_new_transaction_log($transaction_meta_args, array( 'post_parent' => $wallet_id, 'post_status' => 'wal_debit' ));

		//Update last wallet activity.
		update_post_meta($wallet_id, 'wal_last_activity_date', WAL_Date_Time::get_mysql_date_time_format('now', true));

		$transaction_log = wal_get_transaction_log($transaction_log_id);
		if ($transaction_log->has_status('wal_debit')) {
			/**
			 * This hook is used to do extra action after the wallet amount debited.
			 * 
			 * @since 1.0
			 */
			do_action('wal_wallet_amount_debited', $transaction_log_id, $transaction_log, $args['mode']);

			return $transaction_log_id;
		} else {
			return false;
		}
	}

}

if (!function_exists('wal_customer_wallet_usage_count_per_day')) {

	/**
	 * Get the customer wallet usage count per day.
	 *
	 *  @return array
	 */
	function wal_customer_wallet_usage_count_per_day( $user_id = false ) {
		$order_data = wal_customer_wallet_usage_order_data_per_day($user_id);

		return ( isset($order_data['count']) ) ? intval($order_data['count']) : 0;
	}

}

if (!function_exists('wal_customer_wallet_usage_total_per_day')) {

	/**
	 * Get the customer wallet usage total per day.
	 *
	 *  @return array
	 */
	function wal_customer_wallet_usage_total_per_day( $user_id = false ) {
		$order_data = wal_customer_wallet_usage_order_data_per_day($user_id);

		return isset($order_data['total']) ? floatval($order_data['total']) : 0;
	}

}

if (!function_exists('wal_customer_wallet_usage_order_data_per_day')) {

	/**
	 * Get the customer wallet usage order data per day.
	 *
	 *  @return array
	 */
	function wal_customer_wallet_usage_order_data_per_day( $user_id = false ) {
		global $wpdb;

		if (!$user_id) {
			$user_id = get_current_user_id();
		}

		$transient_name = 'wal_customer_wallet_usage_order_data_per_day_' . md5($user_id);
		$transient_version = WC_Cache_Helper::get_transient_version('orders');
		$transient_value = get_transient($transient_name);

		if (isset($transient_value['value'], $transient_value['version']) && $transient_value['version'] === $transient_version) {
			$result = $transient_value['value'];
		} else {
			// Get the current UTC time.
			$date_object = WAL_Date_Time::get_date_time_object('now', true);
			$start_date = $date_object->format('Y-m-d 00:00:00');
			$end_date = $date_object->format('Y-m-d 23:59:59');
			$statuses = array( 'wc-completed', 'wc-processing', 'wc-on-hold' );

			$post_query = new WAL_Query($wpdb->posts, 'p');
			$post_query->select('COUNT(`pm3`.meta_value) as count,SUM(`pm3`.meta_value) as total')
					->leftJoin($wpdb->postmeta, 'pm1', '`p`.`ID` = `pm1`.`post_id`')
					->leftJoin($wpdb->postmeta, 'pm2', '`pm1`.`post_id` = `pm2`.`post_id`')
					->leftJoin($wpdb->postmeta, 'pm3', '`pm2`.`post_id` = `pm3`.`post_id`')
					->whereIn('`p`.post_status', $statuses)
					->whereBetween('`p`.post_date_gmt', $start_date, $end_date)
					->where('`pm1`.meta_key', 'wal_wallet_fund_debited')
					->where('`pm1`.meta_value', 'yes')
					->where('`pm2`.meta_key', '_customer_user')
					->where('`pm2`.meta_value', $user_id)
					->where('`pm3`.meta_key', 'wal_wallet_fund');

			$result = $post_query->fetchArray();

			$transient_value = array(
				'version' => $transient_version,
				'value' => $result[0],
			);

			set_transient($transient_name, $transient_value, DAY_IN_SECONDS * 30);
		}

		return $result;
	}

}

if (!function_exists('wal_customer_wallet_topup_count_per_day')) {

	/**
	 * Get the customer wallet top up count per day.
	 *
	 *  @return array
	 */
	function wal_customer_wallet_topup_count_per_day( $user_id = false ) {
		$order_data = wal_customer_wallet_topup_order_data_per_day($user_id);

		return ( isset($order_data['count']) ) ? intval($order_data['count']) : 0;
	}

}

if (!function_exists('wal_customer_wallet_topup_total_per_day')) {

	/**
	 * Get the customer wallet top up total per day.
	 *
	 *  @return array
	 */
	function wal_customer_wallet_topup_total_per_day( $user_id = false ) {
		$order_data = wal_customer_wallet_topup_order_data_per_day($user_id);

		return isset($order_data['total']) ? floatval($order_data['total']) : 0;
	}

}

if (!function_exists('wal_customer_wallet_topup_order_data_per_day')) {

	/**
	 * Get the customer wallet top up order data per day.
	 *
	 *  @return array
	 */
	function wal_customer_wallet_topup_order_data_per_day( $user_id = false ) {
		global $wpdb;

		if (!$user_id) {
			$user_id = get_current_user_id();
		}

		$transient_name = 'wal_customer_wallet_topup_order_data_per_day_' . md5($user_id);
		$transient_version = WC_Cache_Helper::get_transient_version('orders');
		$transient_value = get_transient($transient_name);

		if (isset($transient_value['value'], $transient_value['version']) && $transient_value['version'] === $transient_version) {
			$result = $transient_value['value'];
		} else {
			// Get the current UTC time.
			$date_object = WAL_Date_Time::get_date_time_object('now', true);
			$start_date = $date_object->format('Y-m-d 00:00:00');
			$end_date = $date_object->format('Y-m-d 23:59:59');
			$statuses = array( 'wc-completed', 'wc-processing', 'wc-on-hold' );

			$post_query = new WAL_Query($wpdb->posts, 'p');
			$post_query->select('COUNT(`pm3`.meta_value) as count,SUM(`pm3`.meta_value) as total')
					->leftJoin($wpdb->postmeta, 'pm1', '`p`.`ID` = `pm1`.`post_id`')
					->leftJoin($wpdb->postmeta, 'pm2', '`pm1`.`post_id` = `pm2`.`post_id`')
					->leftJoin($wpdb->postmeta, 'pm3', '`pm2`.`post_id` = `pm3`.`post_id`')
					->whereIn('`p`.post_status', $statuses)
					->whereBetween('`p`.post_date_gmt', $start_date, $end_date)
					->where('`pm1`.meta_key', 'wal_topup_fund_credited')
					->where('`pm1`.meta_value', 'yes')
					->where('`pm2`.meta_key', '_customer_user')
					->where('`pm2`.meta_value', $user_id)
					->where('`pm3`.meta_key', 'wal_topup_price');

			$result = $post_query->fetchArray();

			$transient_value = array(
				'version' => $transient_version,
				'value' => $result[0],
			);

			set_transient($transient_name, $transient_value, DAY_IN_SECONDS * 30);
		}

		return $result;
	}

}

if (!function_exists('wal_generate_otp')) {

	/**
	 * Generate the OTP.
	 *
	 * @return string
	 */
	function wal_generate_otp( $length = 4 ) {
		$generated_otp = '';
		$generator = implode('', range(0, 9));

		for ($i = 1; $i <= $length; $i++) {
			$generated_otp .= substr($generator, ( rand() % ( strlen($generator) ) ), 1);
		}
		/**
		 * This hook is used to alter the generated OTP.
		 * 
		 * @since 1.0
		 */
		return apply_filters('wal_generate_otp', $generated_otp);
	}

}

if (!function_exists('wal_add_html_inline_style')) {

	/**
	 * Add the custom CSS to HTML elements.
	 *
	 * @return Mixed
	 */
	function wal_add_html_inline_style( $content, $css, $full_content = false ) {
		if (!$css || !$content) {
			return $content;
		}

		// Return the content with style css when DOMDocument class not exists.
		if (!class_exists('DOMDocument')) {
			return '<style type="text/css">' . $css . '</style>' . $content;
		}

		if (class_exists('\Pelago\Emogrifier\CssInliner')) {
			// To create a instance with original HTML.
			$css_inliner_class = 'Pelago\Emogrifier\CssInliner';
			$domDocument = $css_inliner_class::fromHtml($content)->inlineCss($css)->getDomDocument();
			// Removing the elements with display:none style declaration from the content.
			$html_pruner_class = 'Pelago\Emogrifier\HtmlProcessor\HtmlPruner';
			$html_pruner_class::fromDomDocument($domDocument)->removeElementsWithDisplayNone();
			// Converts a few style attributes values to visual HTML attributes.
			$attribute_converter_class = 'Pelago\Emogrifier\HtmlProcessor\CssToAttributeConverter';
			$visual_html = $attribute_converter_class::fromDomDocument($domDocument)->convertCssToVisualAttributes();

			$content = ( $full_content ) ? $visual_html->render() : $visual_html->renderBodyContent();
		} elseif (class_exists('\Pelago\Emogrifier')) {
			$emogrifier_class = 'Pelago\Emogrifier';
			$emogrifier = new Emogrifier($content, $css);
			$content = ( $full_content ) ? $emogrifier->emogrify() : $emogrifier->emogrifyBodyContent();
		} elseif (version_compare(WC_VERSION, '4.0', '<')) {
			$emogrifier_class = 'Emogrifier';
			if (!class_exists($emogrifier_class)) {
				include_once dirname(WC_PLUGIN_FILE) . '/includes/libraries/class-emogrifier.php';
			}

			$emogrifier = new Emogrifier($content, $css);
			$content = ( $full_content ) ? $emogrifier->emogrify() : $emogrifier->emogrifyBodyContent();
		}

		return $content;
	}

}

if (!function_exists('wal_convert_price')) {

	/**
	 * Convert the price.
	 *
	 * @since 3.0.0
	 * @param float $price
	 * @param bool $convert_to_default_currency
	 * @return int/float
	 */
	function wal_convert_price( $price, $convert_to_default_currency = false ) {
		/**
		 * This hook is used to convert the price.
		 * 
		 * @since 3.0.0
		 * @param int/float $price
		 * @param bool $convert_to_default_currency
		 */
		return apply_filters('wal_convert_price', $price, $convert_to_default_currency);
	}

}

if (!function_exists('wal_convert_price_by_currency')) {

	/**
	 * Convert the price based on the currency.
	 * 
	 * @since 3.0.0
	 * @param float $price
	 * @param bool/string $currency
	 * @return int/float
	 */
	function wal_convert_price_by_currency( $price ) {
		/**
		 * This hook is used to convert the price based on the currency.
		 * 
		 * @since 3.0.0
		 * @param int/float $price
		 */
		$price = apply_filters('wal_convert_price_by_currency', $price);

		return wc_price($price);
	}

}

if (!function_exists('wal_get_filter_types')) {

	/**
	 * Get the filter types
	 *
	 * @since 4.5.0
	 * @return array
	 */
	function wal_get_filter_types() {
		static $types;
		if (isset($types)) {
			return $types;
		}

		/**
		 * This hook is used to alter the filter types.
		 * 
		 * @since 4.5.0
		 */
		$types = apply_filters('wal_filter_types', array(
			'3' => __('Top-up - Credit', 'wallet-for-woocommerce'),
			'4' => __('Top-up - Debit', 'wallet-for-woocommerce'),
			'21' => __('Top-up Bonus - Credit', 'wallet-for-woocommerce'),
			'22' => __('Top-up Bonus - Debit', 'wallet-for-woocommerce'),
			'5' => __('Partial Payment - Credit', 'wallet-for-woocommerce'),
			'6' => __('Partial Payment - Debit', 'wallet-for-woocommerce'),
			'7' => __('Wallet Gateway - Credit', 'wallet-for-woocommerce'),
			'8' => __('Wallet Gateway - Debit', 'wallet-for-woocommerce'),
			'10' => __('Fund Transfer - Debit', 'wallet-for-woocommerce'),
			'11' => __('Fund Received - Credit', 'wallet-for-woocommerce'),
			'12' => __('Auto Top-up - Credit', 'wallet-for-woocommerce'),
			'13' => __('Auto Top-up - Debit', 'wallet-for-woocommerce'),
			'18' => __('Wallet Withdraw - Debit', 'wallet-for-woocommerce'),
			'19' => __('Wallet Withdraw - Credit', 'wallet-for-woocommerce'),
			'14' => __('Gift Voucher - Credit', 'wallet-for-woocommerce'),
			'15' => __('Signup - Credit', 'wallet-for-woocommerce'),
			'16' => __('Daily Login - Credit', 'wallet-for-woocommerce'),
			'17' => __('Product Review - Credit', 'wallet-for-woocommerce'),
			'23' => __('Product Purchase - Credit', 'wallet-for-woocommerce'),
			'24' => __('Product Purchase - Debit', 'wallet-for-woocommerce'),
			'1' => __('Manually Updated - Credit', 'wallet-for-woocommerce'),
			'2' => __('Manually Updated - Debit', 'wallet-for-woocommerce'),
			'9' => __('Expired Fund - Debit', 'wallet-for-woocommerce'),
			'20' => __('Wallet Refund - Credit', 'wallet-for-woocommerce'),
		));

		return $types;
	}

}
