<?php

/**
 * Common functions.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

include_once 'wal-layout-functions.php';
include_once 'wal-post-functions.php';
include_once 'wal-core-functions.php';
include_once 'wal-module-functions.php';
include_once 'wal-store-api-functions.php';

if (!function_exists('wal_check_is_array')) {

	/**
	 * Check if resource is array.
	 *
		 * @since 1.0.0
		 * @param array $data
	 * @return bool
	 */
	function wal_check_is_array( $data ) {
		return ( is_array($data) && !empty($data) );
	}

}

if (!function_exists('wal_price')) {

	/**
	 * Display the price based wc_price function.
	 *
		 * @since 1.0.0
		 * @param float $price
		 * @param array $args
		 * @param bool $echo
	 * @return string
	 */
	function wal_price( $price, $args = array(), $echo = false ) {

		if ($echo) {
			echo wp_kses_post(wc_price($price, $args));
		}

		return wc_price($price, $args);
	}

}

if (!function_exists('wal_filter_readable_products')) {

	/**
	 * Filter the readable products.
	 *
		 * @since 1.0.0
		 * @param array $product_ids
	 * @return array
	 */
	function wal_filter_readable_products( $product_ids ) {

		if (!wal_check_is_array($product_ids)) {
			return array();
		}

		if (function_exists('wc_products_array_filter_readable')) {
			return array_filter(array_map('wc_get_product', $product_ids), 'wc_products_array_filter_readable');
		} else {
			return array_filter(array_map('wc_get_product', $product_ids), 'wal_products_array_filter_readable');
		}
	}

}

if (!function_exists('wal_products_array_filter_readable')) {

	/**
	 * Check the product is readable?.
	 *
		 * @since 1.0.0
		 * @param object $product
	 * @return bool
	 */
	function wal_products_array_filter_readable( $product ) {
		return $product && is_a($product, 'WC_Product') && current_user_can('read_product', $product->get_id());
	}

}

if (!function_exists('wal_customize_array_position')) {

	/**
	 * Customize the array position.
	 * 
		 * @since 1.0.0
		 * @param array $array
		 * @param string $key
		 * @param string/array $new_value
	 * @return array
	 */
	function wal_customize_array_position( $array, $key, $new_value ) {
		$keys = array_keys($array);
		$index = array_search($key, $keys);
		$pos = false === $index ? count($array) : $index + 1;

		$new_value = is_array($new_value) ? $new_value : array( $new_value );

		return array_merge(array_slice($array, 0, $pos), $new_value, array_slice($array, $pos));
	}

}

if (!function_exists('wal_user_id_exists')) {

	/**
	 * Check whether the given user ID exists?.
	 * 
		 * @since 1.0.0
		 * @param int $user_id
	 * @return array
	 */
	function wal_user_id_exists( $user_id ) {
		return (bool) get_user_by('id', $user_id);
	}

}

if (!function_exists('wal_generate_random_code')) {

	/**
	 * Generate the random code.
	 * 
		 * @since 1.0.0
		 * @param array $args
	 * @return string
	 */
	function wal_generate_random_code( $args ) {
		$default_args = array(
			'prefix' => '',
			'suffix' => '',
			'type' => '1',
			'length' => 0,
			'exclude_alphabets' => '',
		);

		$args = wp_parse_args($args, $default_args);

		if (!$args['length']) {
			return '';
		}

		// Prepare the characters.
		if ('2' == $args['type']) {
			$characters = array_merge(range('a', 'z'), range('0', '9'));
			$exclude_characters = explode(',', $args['exclude_alphabets']);
			$characters = array_diff($characters, $exclude_characters);
		} else {
			$characters = range('0', '9');
		}

		// Pick the random characters.
		$_code = '';
		for ($i = 0; $i < $args['length']; $i++) {
			$random_key = array_rand($characters);
			$_code .= $characters[$random_key];
		}

		// Prepare the random code.
		$random_code = $args['prefix'] . $_code . $args['suffix'];
		/**
		 * This hook is used to alter the random code.
		 * 
		 * @since 1.0.0
				 * @param string $random_code
				 * @param string $_code
				 * @param array $args
				 * @return string
		 */
		return apply_filters('wal_generate_random_code', $random_code, $_code, $args);
	}

}

if (!function_exists('wal_get_pagination_classes')) {

	/**
	 * Get the pagination classes.
	 *
	 * @since 3.4.0
	 * @param int $page_no
	 * @param int $current_page
	 *
	 * @return array
	 */
	function wal_get_pagination_classes( $page_no, $current_page ) {
		$classes = array( 'wal-pagination', 'wal-pagination-' . $page_no );
		if ($current_page == $page_no) {
			$classes[] = 'current';
		}
		/**
		 * This hook is used to alter the pagination classes.
		 * 
		 * @since 3.4.0
		 * @param array $classes
		 * @param int $page_no
		 * @param int $current_page
		 * 
		 * @return array
		 */
		return apply_filters('wal_pagination_classes', $classes, $page_no, $current_page);
	}

}

if (!function_exists('wal_get_pagination_number')) {

	/**
	 * Get the pagination number.
	 *
	 * @since 3.4.0
	 * @param int $start
	 * @param int $page_count
	 * @param int $current_page
	 *
	 * @return int
	 */
	function wal_get_pagination_number( $start, $page_count, $current_page ) {
		$page_no = false;
		if ($current_page <= $page_count && $start <= $page_count) {
			$page_no = $start;
		} else if ($current_page > $page_count) {
			$overall_count = $current_page - $page_count + $start;
			if ($overall_count <= $current_page) {
				$page_no = $overall_count;
			}
		}
		/**
		 * This hook is used to alter the pagination number.
		 * 
		 * @since 3.4.0
				 * @param int $page_no
		 * @param int $start
		 * @param int $page_count
		 * @param int $current_page
		 * @return int 
		 */
		return apply_filters('wal_pagination_number', $page_no, $start, $page_count, $current_page);
	}

}
