<?php

/**
 * Customer - Wallet withdrawal request cancelled.
 * 
 * @since 1.8
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Customer_Wallet_Withdrawal_Request_Cancelled_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Customer_Wallet_Withdrawal_Request_Cancelled_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'customer_wallet_withdrawal_request_cancelled' ;
			$this->type        = 'customer' ;
			$this->title       = __( 'Customer - Wallet Withdrawal Request Cancelled', 'wallet-for-woocommerce' ) ;
			$this->description = __( 'This email will be sent to the user when the site admin has cancelled the withdrawal request.', 'wallet-for-woocommerce' ) ;

			//Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_wallet_withdrawal_request_cancelled' ), array( $this, 'trigger' ), 10, 1 ) ;

			parent::__construct() ;
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {
			return '{site_name} - Wallet Withdrawal Request Canceled' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {
			return 'Hi,

Your wallet withdrawal request {wallet_withdrawal_amount} submitted on {date} has been cancelled.

Your current wallet balance is {wallet_balance}. Please visit your wallet dashboard for more details {transaction_url}.

Thanks.' ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $withdrawal_id ) {
			// Is enabled?.
			if ( ! $this->is_enabled() ) {
				return ;
			}

			$withdrawal = wal_get_wallet_withdrawal( $withdrawal_id ) ;
			if ( ! is_object( $withdrawal ) ) {
				return ;
			}

			$this->recipient                                    = $withdrawal->get_user_email() ;
			$this->placeholders[ '{wallet_balance}' ]           = wal_convert_price_by_currency( $withdrawal->get_wallet()->get_balance() ) ;
			$this->placeholders[ '{wallet_withdrawal_amount}' ] = wal_convert_price_by_currency( $withdrawal->get_amount()) ;
			$this->placeholders[ '{transaction_url}' ]          = '<a href="' . esc_url( wal_dashboard_menu_endpoint_url( 'wallet_withdrawal_transactions' ) ) . '" class="wal-email-withdrawal-transactions-link">' . __( 'Transaction Page', 'wallet-for-woocommerce' ) . '</a>' ;
			$this->placeholders[ '{date}' ]                     = $withdrawal->get_formatted_created_date() ;

			if ( $this->get_recipient() ) {
				$this->send_email( $this->get_recipient(), $this->get_subject(), $this->get_formatted_message() ) ;
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Wallet Withdrawal Request Cancelled', 'wallet-for-woocommerce' ),
				'id'    => 'wal_wallet_withdrawal_request_cancelled_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable', 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox to send an email to the user when the site admin has cancelled the withdrawal request.', 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message', 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_wallet_withdrawal_request_cancelled_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
