<?php

/**
 * Customer - Wallet Funds Debit Manually.
 *
 * @since 2.9.1
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Customer_Wallet_Funds_Debit_Manually_Notification')) {

	/**
	 * Class.
	 *
	 * @since 2.9.1
	 * */
	class WAL_Customer_Wallet_Funds_Debit_Manually_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 *
				 * @since 2.9.1
		 * */
		public function __construct() {

			$this->id = 'customer_wallet_funds_debit_manually';
			$this->type = 'customer';
			$this->title = __('Customer - Wallet Funds Debited Manually', 'wallet-for-woocommerce');
			$this->description = __('This email will be sent to the user when the site admin manually debited the funds from Users Wallet.', 'wallet-for-woocommerce');

			//Triggers for this email.
			add_action(sanitize_key($this->plugin_slug . '_wallet_amount_debited'), array( $this, 'trigger' ), 10, 3 );

			parent::__construct();
		}

		/**
		 * Default Subject.
		 *
		 * @since 2.9.1
		 * 
		 * @return string
		 * */
		public function get_default_subject() {
			return '{site_name} - Wallet Funds Debited';
		}

		/**
		 * Default Message.
		 *
		 * @since 2.9.1
		 * 
		 * @return string
		 * */
		public function get_default_message() {
			return 'Hi {user_name},

{wallet_funds_debited} funds has been debited from your wallet & current wallet balance is {wallet_balance}.

Reason: {wallet_debited_reason}

Please visit your wallet dashboard for more details {transaction_url}.

Thanks.';
		}

		/**
		 * Trigger the sending of this email.
		 *
		 * @since 2.9.1
		 * @param int $transaction_log_id
		 * @param object $transaction_log
		 * 
		 * @return void
		 * */
		public function trigger( $transaction_log_id, $transaction_log, $mode ) {
			// Is enabled?.
			if (!$this->is_enabled()) {
				return;
			}

			if (!is_a($transaction_log, 'WAL_Transaction_Log')) {
				$transaction_log = wal_get_transaction_log($transaction_log_id);
			}
						
			if ( 'manual' !== $mode ) {
				return;
			}
			
			$this->recipient = $transaction_log->get_user_email();
			$this->placeholders['{user_name}'] = $transaction_log->get_user_name();
			$this->placeholders['{wallet_funds_debited}'] = wal_price( $transaction_log->get_amount());
			$this->placeholders['{wallet_balance}'] = wal_price($transaction_log->get_wallet()->get_balance());
			$this->placeholders['{wallet_debited_reason}'] = $transaction_log->get_event_message();
			$this->placeholders['{transaction_url}'] = '<a href="' . esc_url(wc_get_page_permalink('wal_dashboard')) . '" class="wal-email-dashboard-transactions-link">' . __('Dashboard', 'wallet-for-woocommerce') . '</a>';

			if ($this->get_recipient()) {
				$this->send_email($this->get_recipient(), $this->get_subject(), $this->get_formatted_message());
			}
		}

		/**
		 * Get the settings array.
		 *
		 * @since 2.9.1
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array();

			//Email section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Wallet Funds Debited Manually', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_funds_debit_manually_email_options',
			);
			$section_fields[] = array(
				'title' => __('Enable/Disable', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'id' => $this->get_option_key('enabled'),
				'desc' => __('Enable this checkbox to send an email to your users when the site admin manually debited the funds from Users Wallet.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Subject', 'wallet-for-woocommerce'),
				'type' => 'text',
				'default' => $this->get_default_subject(),
				'id' => $this->get_option_key('subject'),
			);
			$section_fields[] = array(
				'title' => __('Message', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default' => $this->get_default_message(),
				'id' => $this->get_option_key('message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_funds_debit_manually_email_options',
			);
			//Email section end.

			return $section_fields;
		}
	}

}
