<?php

/**
 * Customer - Wallet Funds Credit.
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Customer_Wallet_Funds_Credit_Notification')) {

	/**
	 * Class.
	 * */
	class WAL_Customer_Wallet_Funds_Credit_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id = 'customer_wallet_funds_credit';
			$this->type = 'customer';
			$this->title = __('Customer - Wallet Funds Credited', 'wallet-for-woocommerce');
			$this->description = __('This email will be sent to the user when funds have been credited to their wallet.', 'wallet-for-woocommerce');

			//Triggers for this email.
			add_action(sanitize_key($this->plugin_slug . '_wallet_amount_credited'), array( $this, 'trigger' ), 10, 3);

			parent::__construct();
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {

			return '{site_name} - Wallet Funds Successfully Credited';
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {

			return 'Hi {user_name},

{wallet_transaction_amount} has been successfully credited to your wallet on {date}. Your current Wallet balance is {wallet_balance}.

Thanks. ';
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $transaction_log_id, $transaction_log, $mode ) {
			// Is enabled?.
			if (!$this->is_enabled()) {
				return;
			}

			if (!is_a($transaction_log, 'WAL_Transaction_Log')) {
				$transaction_log = wal_get_transaction_log($transaction_log_id);
			}
						
			if ( 'manual' === $mode ) {
				return;
			}

			$this->recipient = $transaction_log->get_user_email();
			$this->placeholders['{user_name}'] = $transaction_log->get_user_name();
			$this->placeholders['{wallet_transaction_amount}'] = wal_convert_price_by_currency($transaction_log->get_amount());
			$this->placeholders['{wallet_balance}'] = wal_convert_price_by_currency($transaction_log->get_wallet()->get_balance());
			$this->placeholders['{date}'] = $transaction_log->get_formatted_created_date();
			$this->placeholders['{wal_credit_debit_reason}'] = $transaction_log->get_event_message();

			if ($this->get_recipient()) {
				$this->send_email($this->get_recipient(), $this->get_subject(), $this->get_formatted_message());
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array();

			//Email section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Wallet Funds Credited', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_funds_credit_email_options',
			);
			$section_fields[] = array(
				'title' => __('Enable/Disable', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'id' => $this->get_option_key('enabled'),
				'desc' => __('Enable this checkbox to send an email to your users when funds have been credited to their wallet.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Subject', 'wallet-for-woocommerce'),
				'type' => 'text',
				'default' => $this->get_default_subject(),
				'id' => $this->get_option_key('subject'),
			);
			$section_fields[] = array(
				'title' => __('Message', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default' => $this->get_default_message(),
				'id' => $this->get_option_key('message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_funds_credit_email_options',
			);
			//Email section end.

			return $section_fields;
		}
	}

}
