<?php

/**
 * Customer - Wallet expiry reminder.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Customer_Wallet_Expiry_Reminder_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Customer_Wallet_Expiry_Reminder_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'customer_wallet_expiry_reminder' ;
			$this->type        = 'customer' ;
			$this->title       = __( 'Customer - Wallet Funds Expiry Reminder' , 'wallet-for-woocommerce' ) ;
			$this->description = __( 'This email will be sent to the user when their wallet balance is about to expire.' , 'wallet-for-woocommerce' ) ;

			//Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_wallet_fund_expiry_reminder_emails' ) , array( $this, 'trigger' ) , 10 , 1 ) ;

			parent::__construct() ;
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {

			return '{site_name} - Wallet Balance Expiry Reminder' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {

			return 'Hi {user_name},

Your Wallet Balance {wallet_balance} will be expired on {expiry_date}. Use the available funds on {site_link} before it gets expired.

Thanks.' ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger() {
			// Is enabled?.
			if ( ! $this->is_enabled() ) {
				return ;
			}

			$wallet_ids = $this->get_wallet_ids() ;
			if ( ! wal_check_is_array( $wallet_ids ) ) {
				return ;
			}

			foreach ( $wallet_ids as $wallet_id ) {
				$wallet = wal_get_wallet( $wallet_id ) ;

				if ( ! $wallet->exists() ) {
					continue ;
				}

				$this->recipient                          = $wallet->get_user_email() ;
				$this->placeholders[ '{user_name}' ]      = $wallet->get_user_name() ;
				$this->placeholders[ '{wallet_balance}' ] = wal_convert_price_by_currency( $wallet->get_balance()) ;
				$this->placeholders[ '{expiry_date}' ]    = $wallet->get_formatted_current_expiry_date() ;

				if ( $this->get_recipient() ) {
					$this->send_email( $this->get_recipient() , $this->get_subject() , $this->get_formatted_message() ) ;
				}
			}
		}

		/**
		 * Get the wallet IDs.
		 * 
		 * @return array
		 * */
		public function get_wallet_ids() {
			$threshold = get_option( 'wal_customer_wallet_expiry_reminder_threshold' ) ;
			if ( ! $threshold ) {
				return false ;
			}

			$date_object           = WAL_Date_Time::get_date_time_object( 'now' , true ) ;
			$date_object->modify( '+ ' . $threshold . ' days' ) ;
			$threshold_expiry_date = $date_object->format( 'Y-m-d H:i:s' ) ;

			$args = array(
				'post_type'   => WAL_Register_Post_Types::WALLET_POSTTYPE,
				'post_status' => 'wal_active',
				'fields'      => 'ids',
				'numberposts' => '-1',
				'meta_query'  => array(
					array(
						'key'     => 'wal_current_expiry_date',
						'value'   => $threshold_expiry_date,
						'compare' => '<=',
						'type'    => 'DATETIME',
					),
					array(
						'key'     => 'wal_current_expiry_date',
						'value'   => '',
						'compare' => '!=',
					),
				),
					) ;

			return get_posts( $args ) ;
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Wallet Funds Expired' , 'wallet-for-woocommerce' ),
				'id'    => 'wal_wallet_expired_email_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable' , 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'yes',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox to send an email to your users when their wallet balance is about to expire.' , 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'             => __( 'Enter the Threshold Value' , 'wallet-for-woocommerce' ),
				'type'              => 'number',
				'default'           => '10',
				'custom_attributes' => array( 'min' => 1, 'required' => 'required' ),
				'id'                => $this->get_option_key( 'threshold' ),
				'desc'              => __( "The email will be sent to the user based on the value configuring in this field & the expiry date set on the user's wallet. <br /> Ex: If you set the value '5' in this field, then a user will receive an email 5 days before their expiry date." , 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject' , 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message' , 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_wallet_expired_email_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
