<?php

/**
 * Customer - Wallet expired.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Customer_Wallet_Expired_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Customer_Wallet_Expired_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id                   = 'customer_wallet_expired' ;
			$this->type                 = 'customer' ;
			$this->title                = __( 'Customer - Wallet Funds Expired' , 'wallet-for-woocommerce' ) ;
			$this->description          = __( 'This email will be sent to the user when their wallet balance got expired.' , 'wallet-for-woocommerce' ) ;

			//Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_wallet_fund_expired' ) , array( $this, 'trigger' ) , 10 , 1 ) ;

			parent::__construct() ;
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {

			return '{site_name} - Wallet Balance Expired' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {

			return 'Hi {user_name},

Your Wallet Balance {wallet_balance} got expired on {expiry_date}. Use the {wallet_topup_link} to add funds to your Wallet.

Thanks.' ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $wallet ) {
			// Is enabled?.
			if ( ! $this->is_enabled() ) {
				return ;
			}

			if ( ! is_a( $wallet , 'WAL_Wallet' ) ) {
				$wallet = wal_get_wallet( $wallet ) ;
			}

			$this->recipient                             = $wallet->get_user_email() ;
			$this->placeholders[ '{user_name}' ]         = $wallet->get_user_name() ;
			$this->placeholders[ '{wallet_balance}' ]    = wal_convert_price_by_currency( $wallet->get_balance() ) ;
			$this->placeholders[ '{expiry_date}' ]       = WAL_Date_Time::get_wp_format_datetime( 'now' ) ;
			$this->placeholders[ '{wallet_topup_link}' ] = wc_get_page_permalink( 'wal_topup' ) ;

			if ( $this->get_recipient() ) {
				$this->send_email( $this->get_recipient() , $this->get_subject() , $this->get_formatted_message() ) ;
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Wallet Funds Expired' , 'wallet-for-woocommerce' ),
				'id'    => 'wal_wallet_expired_email_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable' , 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'yes',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox  to send an email to your users when their wallet balance has expired.' , 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject' , 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message' , 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_wallet_expired_email_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
