<?php

/**
 * Customer - Wallet Funds for user daily login.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Customer_User_Daily_Login_Fund_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Customer_User_Daily_Login_Fund_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'customer_user_daily_login_fund' ;
			$this->type        = 'customer' ;
			$this->title       = __( 'Customer - Wallet Funds for Daily Login', 'wallet-for-woocommerce' ) ;
			$this->description = __( 'This email will be sent to the user when they login to your site through WooCommerce Login Form.', 'wallet-for-woocommerce' ) ;

			//Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_user_daily_login_fund_credited' ), array( $this, 'trigger' ), 10, 1 ) ;

			parent::__construct() ;
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {

			return '{site_name} - Funds Credited through Daily Login' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {

			return 'Hi {user_name},

{daily_login_amount} has been successfully credited to your wallet on {date} for login to the site {site_name}. Your current Wallet balance is {wallet_balance}.

Thanks.' ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $transaction_log_id, $transaction_log = false ) {
			// Is enabled?.
			if ( ! $this->is_enabled() ) {
				return ;
			}

			if ( ! is_a( $transaction_log, 'WAL_Transaction_Log' ) ) {
				$transaction_log = wal_get_transaction_log( $transaction_log_id ) ;
			}

			$this->recipient                              = $transaction_log->get_user_email() ;
			$this->placeholders[ '{user_name}' ]          = $transaction_log->get_user_name() ;
			$this->placeholders[ '{daily_login_amount}' ] = wal_convert_price_by_currency( $transaction_log->get_amount()) ;
			$this->placeholders[ '{wallet_balance}' ]     = wal_convert_price_by_currency( $transaction_log->get_wallet()->get_balance()) ;
			$this->placeholders[ '{date}' ]               = $transaction_log->get_formatted_created_date() ;

			if ( $this->get_recipient() ) {
				$this->send_email( $this->get_recipient(), $this->get_subject(), $this->get_formatted_message() ) ;
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Wallet Funds for Daily Login', 'wallet-for-woocommerce' ),
				'id'    => 'wal_user_daily_login_fund_email_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable', 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'yes',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox to send an email to your users when they login to your site through WooCommerce Login Form.', 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message', 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_user_daily_login_fund_email_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
