<?php

/**
 * Customer - Purchased product fund notification
 *
 * @since 3.1.0
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Customer_Purchased_Product_Fund_Notification')) {

	/**
	 * Class.
	 *
	 * @since 3.1.0
	 * */
	class WAL_Customer_Purchased_Product_Fund_Notification extends WAL_Notifications {

		/**
		 * Class constructor.
		 *
		 * @since 3.1.0
		 * */
		public function __construct() {

			$this->id = 'customer_purchased_product_funds';
			$this->type = 'customer';
			$this->title = __('Customer – Wallet Funds for Product Purchase', 'wallet-for-woocommerce');
			$this->description = __('This email will be sent to the users when they earn funds for purchasing the products.', 'wallet-for-woocommerce');

			//Triggers for this email.
			add_action(sanitize_key($this->plugin_slug . '_purchased_product_fund_credited'), array( $this, 'trigger' ), 10, 2);

			parent::__construct();
		}

		/**
		 * Default subject.
		 *
		 * @since 3.1.0
		 * @return string
		 * */
		public function get_default_subject() {
			return '{site_name} - Wallet Funds for Product Purchase';
		}

		/**
		 * Default message.
		 *
		 * @since 3.1.0
		 * @return string
		 * */
		public function get_default_message() {
			return 'Hi {user_name},
                            
You have earned funds for purchasing the product(s) in order{order_id} & your available balance is {wallet_balance}.

{product_details} 

Please visit your wallet dashboard for more details {transaction_url}. 

Thanks.';
		}

		/**
		 * Custom CSS.
		 *
		 * @since 3.1.0
		 * 
		 * @return string
		 */
		public function custom_css() {
			return '.wal-product-purchase-fund-details {
                    width: 100%;
                    border: 1px solid #cccccc;
                    border-collapse: collapse;
                    }

                    .wal-product-purchase-fund-details .wal-product-purchase-fund-details-head {
                    padding: 10px;
                    background: #cccccc;
                    color: #000;
                    text-align:center;
                    }

                    .wal-product-purchase-fund-details .wal-product-purchase-fund-details-data {
                    padding: 10px;
                    border-bottom: 1px solid #cccccc;
                    border-right: 1px solid #cccccc;
                    color: #000;
                    text-align:center;
                    }

                    .wal-product-purchase-fund-details .wal-product-purchase-fund-details-data a {
                    color: #00f;
                    text-decoration: underline;
                    }';
		}

		/**
		 * Trigger the sending of this email.
		 *
		 * @since 3.1.0
		 * */
		public function trigger( $transaction_log_id, $order_id ) {
			// Is enabled?.
			if (!$this->is_enabled()) {
				return;
			}

			if (empty($transaction_log_id)) {
				return;
			}

			$transaction_log = wal_get_transaction_log($transaction_log_id);
			if (!is_object($transaction_log)) {
				return;
			}

			$this->recipient = $transaction_log->get_user_email();
			$this->placeholders['{user_name}'] = $transaction_log->get_user_name();
			$this->placeholders['{order_id}'] = '#' . $order_id;
			$this->placeholders['{wallet_balance}'] = wal_convert_price_by_currency($transaction_log->get_wallet()->get_balance());
			$this->placeholders['{product_details}'] = $this->prepare_purchased_product_email_content($order_id);
			$this->placeholders['{transaction_url}'] = '<a href="' . esc_url(wal_dashboard_menu_endpoint_url('activity')) . '" class="wal-email-product-purchase-action">' . __('Transaction Page', 'wallet-for-woocommerce') . '</a>';

			if ($this->get_recipient()) {
				$this->send_email($this->get_recipient(), $this->get_subject(), $this->get_formatted_message());
			}
		}

		/**
		 * Prepare purchased product email content
		 *
		 * @param int $order_id
		 * 
		 * @since 3.1.0
		 * @return string.
		 */
		public function prepare_purchased_product_email_content( $order_id ) {
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return '';
			}

			if ('2' === $order->get_meta('wal_product_purchase_fund_type')) {
				return '';
			}

			ob_start();
			wal_get_template('modules/product-purchase/product-purchase-fund-details.php', array( 'order' => $order ));
			$contents = ob_get_contents();
			ob_end_clean();

			return $contents;
		}

		/**
		 * Get the settings array.
		 *
		 * @since 3.1.0
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array();

			//Email section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Product Purchase Funds Credited', 'wallet-for-woocommerce'),
				'id' => 'wal_purchased_product_fund_credit_email_options',
			);
			$section_fields[] = array(
				'title' => __('Enable/Disable', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'id' => $this->get_option_key('enabled'),
				'desc' => __('Enable this checkbox to send the email to your users when they earn funds for purchasing the products.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Subject', 'wallet-for-woocommerce'),
				'type' => 'text',
				'default' => $this->get_default_subject(),
				'id' => $this->get_option_key('subject'),
			);
			$section_fields[] = array(
				'title' => __('Message', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default' => $this->get_default_message(),
				'id' => $this->get_option_key('message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_purchased_product_fund_credit_email_options',
			);
			//Email section end.

			return $section_fields;
		}
	}

}
