<?php

/**
 * Customer - Low wallet reminder.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Customer_Low_Wallet_Reminder_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Customer_Low_Wallet_Reminder_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'customer_low_wallet_reminder' ;
			$this->type        = 'customer' ;
			$this->title       = __( 'Customer - Low Wallet Funds Threshold', 'wallet-for-woocommerce' ) ;
			$this->description = __( 'This email will be sent to the user when their wallet funds become low based on the threshold configured.', 'wallet-for-woocommerce' ) ;

			parent::__construct() ;

			// Is enabled?.
			if ( $this->is_enabled() ) {
				//Triggers for this email.
				add_action( sanitize_key( $this->plugin_slug . '_wallet_amount_credited' ), array( $this, 'trigger' ), 10, 2 ) ;
				add_action( sanitize_key( $this->plugin_slug . '_wallet_amount_debited' ), array( $this, 'trigger' ), 10, 2 ) ;
			}
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {

			return '{site_name} - Low Wallet Funds Threshold' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {

			return 'Hi {user_name},

Your Wallet Balance {wallet_balance} is running low. Please make use of the Topup Form {wallet_topup_link} to add funds to your Wallet.

Thanks.' ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $transaction_log_id, $transaction_log ) {
			if ( ! is_a( $transaction_log, 'WAL_Transaction_Log' ) ) {
				$transaction_log = wal_get_transaction_log( $transaction_log_id ) ;
			}

			if ( ! is_object( $transaction_log->get_wallet() ) ) {
				return ;
			}

			// Return if the low wallet threshold is not set.
			// if the wallet balance is not reached the low wallet threshold.
			$threshold = floatval( get_option( 'wal_customer_low_wallet_reminder_threshold' ) ) ;
			if ( ! $threshold || floatval( $transaction_log->get_wallet()->get_balance() ) > $threshold ) {
				return ;
			}

			$this->recipient                             = $transaction_log->get_wallet()->get_user_email() ;
			$this->placeholders[ '{user_name}' ]         = $transaction_log->get_wallet()->get_user_name() ;
			$this->placeholders[ '{wallet_balance}' ]    = wal_convert_price_by_currency( $transaction_log->get_wallet()->get_balance()) ;
			$this->placeholders[ '{wallet_topup_link}' ] = wc_get_page_permalink( 'wal_topup' ) ;

			if ( $this->get_recipient() && $this->send_email( $this->get_recipient(), $this->get_subject(), $this->get_formatted_message() ) ) {
				$transaction_log->get_wallet()->update_meta( 'wal_low_wallet_email_sent', WAL_Date_Time::get_mysql_date_time_format( 'now', true ) ) ;
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Low Wallet Funds', 'wallet-for-woocommerce' ),
				'id'    => 'wal_low_wallet_reminder_email_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable', 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox to send an email to your users when their wallet funds become low based on the threshold configured.', 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Enter the Threshold Value', 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'default'   => '',
				'wal_field' => 'price',
				'id'        => $this->get_option_key( 'threshold' ),
				'desc'      => __( "The email will be sent to the user once their wallet funds match the value configured in this field. <br /> Ex: If you set the value '50' in this field, then a user will receive an email when their wallet funds reduce to 50 or below 50.", 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message', 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_low_wallet_reminder_email_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
