<?php

/**
 * Customer - Wallet Funds for Gift Voucher.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Customer_Gift_Voucher_Redeemed_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Customer_Gift_Voucher_Redeemed_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'customer_gift_voucher_redeemed' ;
			$this->type        = 'customer' ;
			$this->title       = __( 'Customer - Wallet Funds for Gift Voucher' , 'wallet-for-woocommerce' ) ;
			$this->description = __( 'This email will be sent to the user when they redeem the valid Gift Voucher.' , 'wallet-for-woocommerce' ) ;

			//Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_gift_voucher_redeemed' ) , array( $this, 'trigger' ) , 10 , 1 ) ;

			parent::__construct() ;
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {

			return '{site_name} - Funds Credited through Gift Voucher' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {

			return 'Hi {user_name},

{voucher_amount} has been successfully credited to your wallet on {date} for redeeming the Gift Voucher "{voucher_code}". Your current Wallet balance is {wallet_balance}.

Thanks.' ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $gift_voucher ) {
			// Is enabled?.
			if ( ! $this->is_enabled() ) {
				return ;
			}

			if ( ! is_a( $gift_voucher , 'WAL_Gift_Voucher' ) ) {
				$gift_voucher = wal_get_gift_voucher( $gift_voucher ) ;
			}

			$wallet_id = wal_get_wallet_id_by_user_id( $gift_voucher->get_redeemed_by() ) ;

			//Return if the user not having wallet.
			if ( ! $wallet_id ) {
				return ;
			}

			$wallet = wal_get_wallet( $wallet_id ) ;
			if ( ! $wallet->exists() ) {
				return ;
			}

			$this->recipient                          = $gift_voucher->get_user_email() ;
			$this->placeholders[ '{user_name}' ]      = $gift_voucher->get_user_name() ;
			$this->placeholders[ '{voucher_code}' ]   = $gift_voucher->get_code() ;
			$this->placeholders[ '{wallet_balance}' ] = wal_convert_price_by_currency( $wallet->get_balance()) ;
			$this->placeholders[ '{date}' ]           = $gift_voucher->get_formatted_redeemed_date() ;
			$this->placeholders[ '{voucher_amount}' ] = wal_convert_price_by_currency( $gift_voucher->get_amount() ) ;

			if ( $this->get_recipient() ) {
				$this->send_email( $this->get_recipient() , $this->get_subject() , $this->get_formatted_message() ) ;
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Wallet Funds for Gift Voucher' , 'wallet-for-woocommerce' ),
				'id'    => 'wal_wallet_expired_email_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable' , 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'yes',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox to send an email to your users when they redeem the valid Gift Voucher.' , 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject' , 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message' , 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_wallet_expired_email_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
