<?php

/**
 * Customer - Wallet Fund Transfer OTP.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Customer_Fund_Transfer_OTP_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Customer_Fund_Transfer_OTP_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'customer_fund_transfer_otp' ;
			$this->type        = 'customer' ;
			$this->title       = __( 'Customer - OTP for Fund Transfer' , 'wallet-for-woocommerce' ) ;
			$this->description = __( 'This email will send to the sender who tries to transfer their funds.' , 'wallet-for-woocommerce' ) ;

			//Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_processed_fund_transfer' ) , array( $this, 'trigger' ) , 10 , 3 ) ;

			parent::__construct() ;
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {

			return '{site_name} - One Time Password for Wallet Fund Transfer' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {

			return 'Hi {user_name},

{otp} is the One Time Password for Wallet Fund Transfer of {transfer_amount} to {receiver_email} on {site_name}.

Note: This One Time Password is valid only for {otp_validity} minutes.

Thanks.' ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $receiver_id, $amount, $action ) {
			// Is enabled?.
			if ( ! $this->is_enabled() ) {
				return ;
			}

			// Return if the OTP is not enabled.
			if ( ! wal_fund_transfer_otp_enabled() ) {
				return ;
			}

			// Return if the action is confirm.
			if ( 'confirm' == $action ) {
				return ;
			}

			// Return if the user ID is not exists.
			if ( ! $receiver_id || ! WAL_Current_User_Wallet::get_user_id() ) {
				return ;
			}

			$user_data     = get_userdata( WAL_Current_User_Wallet::get_user_id() ) ;
			$receiver_data = get_userdata( $receiver_id ) ;

			// Return if the object is not exists.
			if ( ! is_object( $user_data ) || ! is_object( $receiver_data ) ) {
				return ;
			}

			$validity      = intval( get_option( 'wal_module_fund_transfer_otp_validity' ) ) ;
			$length        = intval( get_option( 'wal_module_fund_transfer_otp_length' ) ) ;
			$generated_otp = wal_generate_otp( $length ) ;

			$this->recipient                           = $user_data->user_email ;
			$this->placeholders[ '{user_name}' ]       = $user_data->display_name ;
			$this->placeholders[ '{otp}' ]             = $generated_otp ;
			$this->placeholders[ '{transfer_amount}' ] = wal_convert_price_by_currency( $amount ) ;
			$this->placeholders[ '{receiver_email}' ]  = $receiver_data->user_email ;
			$this->placeholders[ '{otp_validity}' ]    = $validity ;

			if ( $this->get_recipient() ) {
				$return = $this->send_email( $this->get_recipient() , $this->get_subject() , $this->get_formatted_message() ) ;

				if ( $return ) {
					$date_object   = WAL_Date_Time::get_date_time_object( 'now' , true ) ;
					$date_object->modify( '+ ' . $validity . ' days' ) ;
					$validity_date = $date_object->format( 'Y-m-d H:i:s' ) ;

					//Update the OTP Details in user meta.
					update_post_meta( WAL_Current_User_Wallet::get_user_id() , 'wal_fund_transfer_otp_validity' , $validity_date ) ;
					update_post_meta( WAL_Current_User_Wallet::get_user_id() , 'wal_fund_transfer_otp' , $generated_otp ) ;

					WAL_Dashboard_Handler::add_message( __( 'OTP is sent to your Email ID' , 'wallet-for-woocommerce' ) ) ;
				}
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Wallet Fund Transfer OTP' , 'wallet-for-woocommerce' ),
				'id'    => 'wal_wallet_fund_transfer_otp_email_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable' , 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox to send the email to the sender who tries to transfer their funds.' , 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject' , 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message' , 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_wallet_fund_transfer_otp_email_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
