<?php

/**
 * Customer - Wallet Fund Received.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Customer_Fund_Received_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Customer_Fund_Received_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'customer_fund_received' ;
			$this->type        = 'customer' ;
			$this->title       = __( 'Customer - Wallet Fund Received' , 'wallet-for-woocommerce' ) ;
			$this->description = __( 'This email will be sent to the receiver who receives the funds to their wallet.' , 'wallet-for-woocommerce' ) ;

			//Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_user_fund_transfered' ) , array( $this, 'trigger' ) , 10 , 1 ) ;

			parent::__construct() ;
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {

			return '{site_name} - Fund Received Successfully' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {

			return 'Hi {user_name},

You have successfully received {transfer_amount} from {sender} on {transfer_date}. 

Your current wallet balance is {wallet_balance}. Please visit your wallet dashboard for more details {transaction_url}.

Thanks.' ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $args ) {
			// Is enabled?.
			if ( ! $this->is_enabled() ) {
				return ;
			}

			// Return if the receiver/sender ID is not exists.
			if ( ! $args[ 'receiver_id' ] || ! $args[ 'sender_id' ] ) {
				return ;
			}

			$wallet      = wal_get_wallet_by_user_id( $args[ 'receiver_id' ] ) ;
			$sender_data = get_userdata( $args[ 'sender_id' ] ) ;

			// Return if the receiver object is not exists.
			if ( ! is_object( $sender_data ) || ! $wallet ) {
				return ;
			}

			$this->recipient                           = $wallet->get_user_email() ;
			$this->placeholders[ '{user_name}' ]       = $wallet->get_user_name() ;
			$this->placeholders[ '{transfer_amount}' ] = wal_convert_price_by_currency( $args[ 'amount' ] ) ;
			$this->placeholders[ '{sender}' ]          = $sender_data->display_name ;
			$this->placeholders[ '{transfer_date}' ]   = WAL_Date_Time::get_wp_format_datetime( 'now' ) ;
			$this->placeholders[ '{wallet_balance}' ]  = wal_convert_price_by_currency( $wallet->get_balance() ) ;
			$this->placeholders[ '{transaction_url}' ] = '<a href="' . esc_url( wal_dashboard_menu_endpoint_url( 'transactions' ) ) . '" class="wal-email-fund-transactions-link">' . __( 'Transaction Page' , 'wallet-for-woocommerce' ) . '</a>' ;

			if ( $this->get_recipient() ) {
				$this->send_email( $this->get_recipient() , $this->get_subject() , $this->get_formatted_message() ) ;
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Wallet Fund Received' , 'wallet-for-woocommerce' ),
				'id'    => 'wal_wallet_fund_received_email_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable' , 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox to send an email to the receiver who receives the funds to their wallet.' , 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject' , 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message' , 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_wallet_fund_received_email_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
