<?php

/**
 * Admin - Wallet withdrawal request received.
 * 
 * @since 1.8
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Admin_Wallet_Withdrawal_Request_Received_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Admin_Wallet_Withdrawal_Request_Received_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'admin_wallet_withdrawal_request_received' ;
			$this->type        = 'admin' ;
			$this->title       = __( 'Admin - Wallet Withdrawal Request Received', 'wallet-for-woocommerce' ) ;
			$this->description = __( 'This email will be sent to the admin when a user has submitted the withdrawal request.', 'wallet-for-woocommerce' ) ;

			//Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_wallet_withdrawal_request_submitted' ), array( $this, 'trigger' ), 10, 1 ) ;

			parent::__construct() ;
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {
			return '{site_name} - Wallet Withdrawal Request Received' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {
			return 'Hi,

You have received the wallet withdrawal request from one of your users. Please check the below details,

Username: {user_name}

Available Wallet Balance: {wallet_balance}

Requested Amount: {wallet_withdrawal_requested_amount}

Submitted Date: {date}

Thanks.' ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $withdrawal_id ) {
			// Is enabled?.
			if ( ! $this->is_enabled() ) {
				return ;
			}

			$withdrawal = wal_get_wallet_withdrawal( $withdrawal_id ) ;
			if ( ! is_object( $withdrawal ) ) {
				return ;
			}

			$this->recipient                                              = $this->get_admin_emails() ;
			$this->placeholders[ '{user_name}' ]                          = $withdrawal->get_user_name() ;
			$this->placeholders[ '{wallet_balance}' ]                     = wal_price( $withdrawal->get_wallet()->get_balance()) ;
			$this->placeholders[ '{wallet_withdrawal_requested_amount}' ] = wal_price( $withdrawal->get_amount() ) ;
			$this->placeholders[ '{date}' ]                               = $withdrawal->get_formatted_created_date() ;

			if ( $this->get_recipient() ) {
				$this->send_email( $this->get_recipient(), $this->get_subject(), $this->get_formatted_message() ) ;
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Wallet Withdrawal Request Received', 'wallet-for-woocommerce' ),
				'id'    => 'wal_wallet_withdrawal_request_received_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable', 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox to send an email to the admin when a user has submitted the withdrawal request.', 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message', 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_wallet_withdrawal_request_received_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
