<?php

/**
 * Admin - Wallet Funds Debit.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Admin_Wallet_Funds_Debit_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Admin_Wallet_Funds_Debit_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'admin_wallet_funds_debit' ;
			$this->type        = 'admin' ;
			$this->title       = __( 'Admin - Wallet Funds Debited' , 'wallet-for-woocommerce' ) ;
			$this->description = __( 'This email will be sent to the admin when funds have been debited from the customers wallet.' , 'wallet-for-woocommerce' ) ;

			//Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_wallet_amount_debited' ) , array( $this, 'trigger' ) , 10 , 2 ) ;

			parent::__construct() ;
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {

			return '{site_name} - Wallet Funds Debited' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {

			return 'Hi,

Your user {user_name} has used their Wallet Funds {wallet_transaction_amount} on {date}. Please check the below details,
 
Username: {user_name}

Available Wallet Balance: {wallet_balance}

Thanks.' ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $transaction_log_id, $transaction_log ) {
			// Is enabled?.
			if ( ! $this->is_enabled() ) {
				return ;
			}

			if ( ! is_a( $transaction_log , 'WAL_Transaction_Log' ) ) {
				$transaction_log = wal_get_transaction_log( $transaction_log_id ) ;
			}
						
			$this->recipient                                     = $this->get_admin_emails() ;
			$this->placeholders[ '{user_name}' ]                 = $transaction_log->get_user_name() ;
			$this->placeholders[ '{event}' ]                     = $transaction_log->get_formatted_event_message() ;
			$this->placeholders[ '{date}' ]                      = $transaction_log->get_formatted_created_date() ;
			$this->placeholders[ '{wallet_transaction_amount}' ] = wal_price($transaction_log->get_amount() ) ;
			$this->placeholders[ '{wallet_balance}' ]            = wal_price( $transaction_log->get_wallet()->get_balance() ) ;

			if ( $this->get_recipient() ) {
				$this->send_email( $this->get_recipient() , $this->get_subject() , $this->get_formatted_message() ) ;
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Wallet Funds Debited' , 'wallet-for-woocommerce' ),
				'id'    => 'wal_wallet_funds_debit_email_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable' , 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox to send an email to admin when funds have been debited from the users wallet.' , 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'    => __( 'Recipient(s)' , 'wallet-for-woocommerce' ),
				'type'     => 'textarea',
				'default'  => $this->get_from_address(),
				'id'       => $this->get_option_key( 'recipients' ),
				/* translators: %s: from address */
				'desc'     => sprintf( __( 'Enter recipients (comma separated) for this email. Defaults to %s.' , 'wallet-for-woocommerce' ) , $this->get_from_address() ),
				'desc_tip' => true,
				'value'    => $this->get_admin_emails(),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject' , 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message' , 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_wallet_funds_debit_email_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
