<?php

/**
 * Admin - Wallet expired.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Admin_Wallet_Expired_Notification' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Admin_Wallet_Expired_Notification extends WAL_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'admin_wallet_expired' ;
			$this->type        = 'admin' ;
			$this->title       = __( 'Admin - Wallet Funds Expired' , 'wallet-for-woocommerce' ) ;
			$this->description = __( 'This email will be sent to the admin when the wallet balance has expired from the users wallet.' , 'wallet-for-woocommerce' ) ;

			//Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_wallet_fund_expired' ) , array( $this, 'trigger' ) , 10 , 1 ) ;

			parent::__construct() ;
		}

		/**
		 * Default Subject.
		 * 
		 * @return string
		 * */
		public function get_default_subject() {

			return '{site_name} - Wallet Balance Expired' ;
		}

		/**
		 * Default Message.
		 * 
		 * @return string
		 * */
		public function get_default_message() {

			return "Hi,

One of your User's {user_name} Wallet Balance got expired on {expiry_date}. They had {wallet_balance} on their wallet at the time of expiry.

Thanks." ;
		}

		/**
		 * Trigger the sending of this email.
		 * */
		public function trigger( $wallet ) {
			// Is enabled?.
			if ( ! $this->is_enabled() ) {
				return ;
			}

			if ( ! is_a( $wallet , 'WAL_Wallet' ) ) {
				$wallet = wal_get_wallet( $wallet ) ;
			}

			$this->recipient                          = $this->get_admin_emails() ;
			$this->placeholders[ '{user_name}' ]      = $wallet->get_user_name() ;
			$this->placeholders[ '{wallet_balance}' ] = wal_price( $wallet->get_balance()) ;
			$this->placeholders[ '{expiry_date}' ]    = $wallet->get_formatted_current_expiry_date() ;

			if ( $this->get_recipient() ) {
				$this->send_email( $this->get_recipient() , $this->get_subject() , $this->get_formatted_message() ) ;
			}
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			$section_fields = array() ;

			//Email section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Wallet Funds Expired' , 'wallet-for-woocommerce' ),
				'id'    => 'wal_wallet_expired_email_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable/Disable' , 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'enabled' ),
				'desc'    => __( 'Enable this checkbox to send an email to admin when the wallet balance has expired from the users wallet.' , 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'    => __( 'Recipient(s)' , 'wallet-for-woocommerce' ),
				'type'     => 'textarea',
				'default'  => $this->get_from_address(),
				'id'       => $this->get_option_key( 'recipients' ),
				/* translators: %s: from address */
				'desc'     => sprintf( __( 'Enter recipients (comma separated) for this email. Defaults to %s.' , 'wallet-for-woocommerce' ) , $this->get_from_address() ),
				'desc_tip' => true,
				'value'    => $this->get_admin_emails(),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Subject' , 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
					) ;
			$section_fields[] = array(
				'title'     => __( 'Message' , 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_wallet_expired_email_options',
					) ;
			//Email section end.

			return $section_fields ;
		}
	}

}
