<?php

/**
 * Wallet Withdrawal Gateway - Paypal.
 * 
 * @since 1.8
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Withdrawal_Paypal_Gateway' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Withdrawal_Paypal_Gateway extends WAL_Withdrawal_Gateway {

		/**
		 * API URL.
		 * 
		 * @var string
		 */
		private static $api_url ;

		/**
		 * Client ID.
		 * 
		 * @var string
		 */
		private static $client_id ;

		/**
		 * Client Secret Key.
		 * 
		 * @var string
		 */
		private static $client_secret_key ;

		/**
		 * Sandbox Payout API URL.
		 * 
		 * @var string
		 */
		const SANDBOX_PAYOUT_API_URL = 'https://api-m.sandbox.paypal.com/v1/payments/payouts' ;

		/**
		 * Live Payout API URL.
		 * 
		 * @var string
		 */
		const LIVE_PAYOUT_API_URL = 'https://api-m.paypal.com/v1/payments/payouts' ;

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'paypal' ;
			$this->description = __( 'Users can withdraw their funds to the PayPal Account.', 'wallet-for-woocommerce' ) ;

			parent::__construct() ;
		}

		/**
		 * Is valid credentials?.
		 * 
		 * @return bool
		 */
		public function is_valid_credentials() {
			$return = false ;
			if ( $this->get_client_id() && $this->get_client_secret_key() ) {
				$return = true ;
			}
			/**
			 * This hook is used to check if it is valid credentials.
			 * 
			 * @since 1.8
			 */
			return apply_filters( $this->get_plugin_slug() . '_module_withdrawal_' . $this->get_id() . '_is_valid_credentials', $return, $this ) ;
		}

		/**
		 * Default credentials.
		 * 
		 * @return array
		 * */
		public function default_credentials() {
			$default_args = array(
				'paypal_email' => '',
					) ;

			/**
			 * This hook is used to alter the current payment gateway default credentials.
			 * 
			 * @since 1.8
			 */
			return apply_filters( 'wal_wallet_withdrawal_payment_gateway_default_credentials_' . $this->get_id(), $default_args ) ;
		}

		/**
		 * Get the credential labels.
		 * 
		 * @return array
		 * */
		public function get_credential_labels() {
			$labels = array(
				'paypal_email' => wal_get_withdrawal_paypal_gateway_email_label(),
					) ;

			/**
			 * This hook is used to alter the current payment gateway default credentials.
			 * 
			 * @since 1.8
			 */
			return apply_filters( 'wal_wallet_withdrawal_payment_gateway_credential_labels' . $this->get_id(), $labels, $this ) ;
		}

		/**
		 * Get the fee amount.
		 * 
		 * @return float
		 * */
		public function get_fee_amount( $amount ) {
			if ( 'yes' !== get_option( 'wal_module_withdrawal_paypal_fee_enabled' ) ) {
				return 0 ;
			}

			$type = get_option( 'wal_module_withdrawal_paypal_fee_type' ) ;
			switch ( $type ) {
				case '2':
					$pecentage = get_option( 'wal_module_withdrawal_paypal_fee_amount' ) ;
					$amount    = ( $pecentage / 100 ) * $amount ;
					break ;
				default:
					$amount    = get_option( 'wal_module_withdrawal_paypal_fee_amount' ) ;
					break ;
			}

			return $amount ;
		}

		/**
		 * Get the settings.
		 * 
		 * @return array
		 */
		public function get_settings_array() {
			$section_fields = array() ;

			// Withdrawal paypal gateway section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'General Settings', 'wallet-for-woocommerce' ),
				'id'    => 'wal_withdrawal_paypal_gateway_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Enable', 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_enable_key(),
				'desc'    => __( 'By enabling this checkbox, you can allow your users to withdraw their funds to the PayPal Account.', 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Title', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => 'PayPal',
				'id'      => $this->get_option_key( 'title' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Sandbox Mode', 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'sandbox_enabled' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Sandbox Client ID', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => '',
				'id'      => $this->get_option_key( 'sandbox_client_id' ),
				'class'   => 'wal-wallet-withdrawal-paypal-gateway-sandbox-credentials',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Sandbox Client Secret Key', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => '',
				'id'      => $this->get_option_key( 'sandbox_client_secret_key' ),
				'class'   => 'wal-wallet-withdrawal-paypal-gateway-sandbox-credentials',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Client ID', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => '',
				'id'      => $this->get_option_key( 'client_id' ),
				'class'   => 'wal-wallet-withdrawal-paypal-gateway-live-credentials',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Client Secret Key', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => '',
				'id'      => $this->get_option_key( 'client_secret_key' ),
				'class'   => 'wal-wallet-withdrawal-paypal-gateway-live-credentials',
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_withdrawal_paypal_gateway_options',
					) ;
			// Withdrawal paypal gateway section end.
			// Withdrawal paypal gateway fee section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Withdrawal Fee Settings', 'wallet-for-woocommerce' ),
				'id'    => 'wal_withdrawal_paypal_gateway_fee_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Charge Withdrawal Fee', 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'fee_enabled' ),
				'class'   => 'wal-wallet-withdrawal-gateway-fee',
				'desc'    => __( 'By enabling this checkbox, you can charge a withdrawal fee from your users.', 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'    => __( 'Fee Type', 'wallet-for-woocommerce' ),
				'type'     => 'select',
				'default'  => '1',
				'id'       => $this->get_option_key( 'fee_type' ),
				'class'    => 'wal-wallet-withdrawal-gateway-fee-field',
				'options'  => array(
					'1' => __( 'Fixed', 'wallet-for-woocommerce' ),
					'2' => __( 'Percentage', 'wallet-for-woocommerce' ),
				),
				'desc'     => __( 'Choose whether the withdrawal fee should be a fixed fee or a percentage of the withdrawal amount.', 'wallet-for-woocommerce' ),
				'desc_tip' => true,
					) ;
			$section_fields[] = array(
				'title'     => __( 'Set the Fee Value', 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'price',
				'default'   => '',
				'id'        => $this->get_option_key( 'fee_amount' ),
				'class'     => 'wal-wallet-withdrawal-gateway-fee-field',
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_withdrawal_paypal_gateway_fee_options',
					) ;
			// Withdrawal paypal gateway fee section end.
			// Withdrawal paypal gateway localization section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Localization', 'wallet-for-woocommerce' ),
				'id'    => 'wal_withdrawal_paypal_gateway_localization_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Paypal Email Label', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => 'Paypal Email',
				'id'      => $this->get_option_key( 'email_label' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_withdrawal_paypal_gateway_localization_options',
					) ;
			// Withdrawal paypal gateway localization section end.
			// Withdrawal paypal gateway message section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Messages', 'wallet-for-woocommerce' ),
				'id'    => 'wal_withdrawal_paypal_gateway_message_options',
					) ;
			$section_fields[] = array(
				'title'   => __( "Message to display when a user didn't enter the PayPal ID", 'wallet-for-woocommerce' ),
				'type'    => 'textarea',
				'default' => 'Please enter your PayPal ID',
				'id'      => $this->get_option_key( 'email_empty_message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_withdrawal_paypal_gateway_message_options',
					) ;
			// Withdrawal paypal gateway message section end.

			return $section_fields ;
		}

		/**
		 * Prepare the post data.
		 * 
		 * @return array
		 * */
		public function prepare_post_data( &$post_data ) {
			$paypal_email = isset( $post_data[ 'wal_paypal_gateway_email' ] ) ? wc_clean( wp_unslash( $post_data[ 'wal_paypal_gateway_email' ] ) ) : '' ;
			if ( empty( $paypal_email ) ) {
				throw new Exception( wp_kses_post(get_option( 'wal_module_withdrawal_paypal_email_empty_message' )) ) ;
			}

			return array(
				'paypal_email' => $paypal_email,
					) ;
		}

		/**
		 * Process payment.
		 * */
		public function process_payment( $withdrawal ) {
			if ( ! is_object( $withdrawal ) || ! $withdrawal->exists() ) {
				return false ;
			}

			if ( $withdrawal->get_total() <= 0 ) {
				return false ;
			}

			$credentials = $this->get_customer_credentials( $withdrawal->get_wallet_id() ) ;
			if ( ! isset( $credentials[ 'paypal_email' ] ) ) {
				return false ;
			}

			$receiver_email = sanitize_email( $credentials[ 'paypal_email' ] ) ;
			if ( empty( $receiver_email ) ) {
				return false ;
			}

			$request_body = array(
				'sender_batch_header' => array(
					'sender_batch_id' => $withdrawal->get_id(),
					'email_subject'   => wc_clean( __( 'Payout Received Successful', 'wallet-for-woocommerce' ) ),
					'email_message'   => wc_clean( __( 'You have received a payout! Thanks!!', 'wallet-for-woocommerce' ) ),
				),
				'items'               => array(
					array(
						'recipient_type' => 'EMAIL',
						'receiver'       => $receiver_email,
						'note'           => wc_clean( __( 'Payout received.', 'wallet-for-woocommerce' ) ),
						'sender_item_id' => $withdrawal->get_id(),
						'amount'         => array(
							'value'    => $withdrawal->get_total(),
							'currency' => get_woocommerce_currency(),
						),
					),
				),
					) ;

			return $this->request_api( $request_body ) ;
		}

		/**
		 * Get the headers.
		 * 
		 * @return array
		 */
		private function get_headers() {
			return array(
				'Content-Type'  => 'application/json',
				'Authorization' => 'Basic ' . base64_encode( $this->get_client_id() . ':' . $this->get_client_secret_key() ),
					) ;
		}

		/**
		 * Is sandbox mode?.
		 * 
		 * @return bool
		 */
		private function is_sandbox_mode() {
			return 'yes' === get_option( 'wal_module_withdrawal_paypal_sandbox_enabled' ) ;
		}

		/**
		 * Get the API URL.
		 * 
		 * @return string
		 */
		private function get_api_url() {
			if ( self::$api_url ) {
				return self::$api_url ;
			}

			self::$api_url = $this->is_sandbox_mode() ? self::SANDBOX_PAYOUT_API_URL : self::LIVE_PAYOUT_API_URL ;

			return self::$api_url ;
		}

		/**
		 * Get the Client ID.
		 * 
		 * @return string
		 */
		private function get_client_id() {
			if ( self::$client_id ) {
				return self::$client_id ;
			}

			self::$client_id = $this->is_sandbox_mode() ? get_option( 'wal_module_withdrawal_paypal_sandbox_client_id' ) : get_option( 'wal_module_withdrawal_paypal_client_id' ) ;

			return self::$client_id ;
		}

		/**
		 * Get the Client ID.
		 * 
		 * @return string
		 */
		private function get_client_secret_key() {
			if ( self::$client_secret_key ) {
				return self::$client_secret_key ;
			}

			self::$client_secret_key = $this->is_sandbox_mode() ? get_option( 'wal_module_withdrawal_paypal_sandbox_client_secret_key' ) : get_option( 'wal_module_withdrawal_paypal_client_secret_key' ) ;

			return self::$client_secret_key ;
		}

		/**
		 * Request the API.
		 * 
		 * @return array/WP_Error
		 */
		private function request_api( $request_body ) {
			$response = wp_safe_remote_post(
					$this->get_api_url(),
					array(
						'method'  => 'POST',
						'headers' => $this->get_headers(),
						'body'    => wp_json_encode( $request_body ),
						'timeout' => 60,
					)
					) ;

			$response_code = wp_remote_retrieve_response_code( $response ) ;
			$response_body = json_decode( wp_remote_retrieve_body( $response ) ) ;

			if ( is_wp_error( $response ) || ( 201 === $response_code && 'created' !== strtolower( $response[ 'response' ][ 'message' ] ) ) ) {
				$err_message = isset( $response_body->message ) ? $response_body->message : __( 'There was a problem connecting to the PayPal API endpoint.', 'wallet-for-woocommerce' ) ;

				return new WP_Error( 'wal-paypal-payouts-error', $err_message ) ;
			}

			return $response_body ;
		}
	}

}
