<?php

/**
 * Wallet Withdrawal Gateway - Bank.
 * 
 * @since 1.8
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Withdrawal_Bank_Gateway' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Withdrawal_Bank_Gateway extends WAL_Withdrawal_Gateway {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id          = 'bank' ;
			$this->description = __( 'Users can withdraw their funds to the Bank Account.', 'wallet-for-woocommerce' ) ;

			parent::__construct() ;
		}

		/**
		 * Default credentials.
		 * 
		 * @return array
		 * */
		public function default_credentials() {
			$default_args = array(
				'account_name'   => '',
				'account_number' => '',
				'ifsc_code'      => '',
				'bank_name'      => '',
				'routing_number' => '',
				'iban_number'    => '',
				'swift_code'     => '',
					) ;

			/**
			 * This hook is used to alter the current payment gateway default credentials.
			 * 
			 * @since 1.8
			 */
			return apply_filters( 'wal_wallet_withdrawal_payment_gateway_default_credentials_' . $this->get_id(), $default_args, $this ) ;
		}

		/**
		 * Get the credential labels.
		 * 
		 * @return array
		 * */
		public function get_credential_labels() {
			$labels = array(
				'account_name'   => wal_get_withdrawal_bank_gateway_account_name_label(),
				'account_number' => wal_get_withdrawal_bank_gateway_account_number_label(),
				'ifsc_code'      => wal_get_withdrawal_bank_gateway_ifsc_code_label(),
				'bank_name'      => wal_get_withdrawal_bank_gateway_bank_name_label(),
				'routing_number' => wal_get_withdrawal_bank_gateway_routing_number_label(),
				'iban_number'    => wal_get_withdrawal_bank_gateway_iban_number_label(),
				'swift_code'     => wal_get_withdrawal_bank_gateway_swift_code_label(),
					) ;

			/**
			 * This hook is used to alter the current payment gateway default credentials.
			 * 
			 * @since 1.8
			 */
			return apply_filters( 'wal_wallet_withdrawal_payment_gateway_credential_labels' . $this->get_id(), $labels, $this ) ;
		}

		/**
		 * Get the fee amount.
		 * 
		 * @return float
		 * */
		public function get_fee_amount( $amount ) {
			if ( 'yes' !== get_option( 'wal_module_withdrawal_bank_fee_enabled' ) ) {
				return 0 ;
			}

			$type = get_option( 'wal_module_withdrawal_bank_fee_type' ) ;
			switch ( $type ) {
				case '2':
					$pecentage = get_option( 'wal_module_withdrawal_bank_fee_amount' ) ;
					$amount    = ( $pecentage / 100 ) * $amount ;
					break ;
				default:
					$amount    = get_option( 'wal_module_withdrawal_bank_fee_amount' ) ;
					break ;
			}

			return floatval( $amount ) ;
		}

		/**
		 * Get the settings.
		 * 
		 * @return array
		 */
		public function get_settings_array() {
			$section_fields = array() ;

			// Withdrawal bank gateway section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'General Settings', 'wallet-for-woocommerce' ),
				'id'    => 'wal_withdrawal_bank_gateway_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Bank Transfer', 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_enable_key(),
				'desc'    => __( 'By enabling this checkbox, you can allow your users to withdraw their funds to the Bank Account.', 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Title', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => 'Bank Transfer',
				'id'      => $this->get_option_key( 'title' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_withdrawal_bank_gateway_options',
					) ;
			// Withdrawal bank gateway section end.
			// Withdrawal bank gateway fee section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Withdrawal Fee Settings', 'wallet-for-woocommerce' ),
				'id'    => 'wal_withdrawal_bank_gateway_fee_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Charge Withdrawal Fee', 'wallet-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'fee_enabled' ),
				'class'   => 'wal-wallet-withdrawal-gateway-fee',
				'desc'    => __( 'By enabling this checkbox, you can charge a withdrawal fee from your users.', 'wallet-for-woocommerce' ),
					) ;
			$section_fields[] = array(
				'title'    => __( 'Fee Type', 'wallet-for-woocommerce' ),
				'type'     => 'select',
				'default'  => '1',
				'id'       => $this->get_option_key( 'fee_type' ),
				'class'    => 'wal-wallet-withdrawal-gateway-fee-field',
				'options'  => array(
					'1' => __( 'Fixed', 'wallet-for-woocommerce' ),
					'2' => __( 'Percentage', 'wallet-for-woocommerce' ),
				),
				'desc'     => __( 'Choose whether the withdrawal fee should be a fixed fee or a percentage of the withdrawal amount.', 'wallet-for-woocommerce' ),
				'desc_tip' => true,
					) ;
			$section_fields[] = array(
				'title'     => __( 'Set the Fee Value', 'wallet-for-woocommerce' ),
				'type'      => 'wal_custom_fields',
				'wal_field' => 'price',
				'default'   => '',
				'id'        => $this->get_option_key( 'fee_amount' ),
				'class'     => 'wal-wallet-withdrawal-gateway-fee-field',
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_withdrawal_bank_gateway_fee_options',
					) ;
			// Withdrawal bank gateway fee section end.
			// Withdrawal bank gateway localization section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Localization', 'wallet-for-woocommerce' ),
				'id'    => 'wal_withdrawal_bank_gateway_localization_options',
					) ;
			$section_fields[] = array(
				'title'   => __( 'Account Name Label', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => 'Account Name',
				'id'      => $this->get_option_key( 'account_name_label' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Account Number Label', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => 'Account Number',
				'id'      => $this->get_option_key( 'account_number_label' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'IFSC Code Label', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => 'IFSC Code',
				'id'      => $this->get_option_key( 'ifsc_code_label' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Bank Name Label', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => 'Bank Name',
				'id'      => $this->get_option_key( 'bank_name_label' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Routing Number Label', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => 'Routing Number',
				'id'      => $this->get_option_key( 'routing_number_label' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'IBAN Number Label', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => 'IBAN Number',
				'id'      => $this->get_option_key( 'iban_number_label' ),
					) ;
			$section_fields[] = array(
				'title'   => __( 'Swift Code Label', 'wallet-for-woocommerce' ),
				'type'    => 'text',
				'default' => 'Swift Code',
				'id'      => $this->get_option_key( 'swift_code_label' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_withdrawal_bank_gateway_localization_options',
					) ;
			// Withdrawal bank gateway localization section end.
			// Withdrawal bank gateway message section start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Messages', 'wallet-for-woocommerce' ),
				'id'    => 'wal_withdrawal_bank_gateway_message_options',
					) ;
			$section_fields[] = array(
				'title'   => __( "Message to display when a user didn't enter the Account Name", 'wallet-for-woocommerce' ),
				'type'    => 'textarea',
				'default' => 'Please enter the account name',
				'id'      => $this->get_option_key( 'account_name_empty_message' ),
					) ;
			$section_fields[] = array(
				'title'   => __( "Message to display when a user didn't enter the Account Number", 'wallet-for-woocommerce' ),
				'type'    => 'textarea',
				'default' => 'Please enter the account number',
				'id'      => $this->get_option_key( 'account_number_empty_message' ),
					) ;
			$section_fields[] = array(
				'title'   => __( "Message to display when a user didn't enter the IFSC Code", 'wallet-for-woocommerce' ),
				'type'    => 'textarea',
				'default' => 'Please enter the IFSC Code',
				'id'      => $this->get_option_key( 'account_ifsc_empty_message' ),
					) ;
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'wal_withdrawal_bank_gateway_message_options',
					) ;
			// Withdrawal bank gateway message section end.

			return $section_fields ;
		}

		/**
		 * Prepare the post data.
		 * 
		 * @return array
		 * */
		public function prepare_post_data( &$post_data ) {
			$account_name = isset( $post_data[ 'wal_bank_gateway_account_name' ] ) ? wc_clean( wp_unslash( $post_data[ 'wal_bank_gateway_account_name' ] ) ) : '' ;
			if ( empty( $account_name ) ) {
				throw new Exception( wp_kses_post(get_option( 'wal_module_withdrawal_bank_account_name_empty_message' )) ) ;
			}

			$account_number = isset( $post_data[ 'wal_bank_gateway_account_number' ] ) ? wc_clean( wp_unslash( $post_data[ 'wal_bank_gateway_account_number' ] ) ) : '' ;
			if ( empty( $account_number ) ) {
				throw new Exception( wp_kses_post(get_option( 'wal_module_withdrawal_bank_account_number_empty_message' )) ) ;
			}

			$ifsc_code      = isset( $post_data[ 'wal_bank_gateway_ifsc_code' ] ) ? wc_clean( wp_unslash( $post_data[ 'wal_bank_gateway_ifsc_code' ] ) ) : '' ;
			if ( empty( $ifsc_code ) ) {
				throw new Exception( wp_kses_post(get_option( 'wal_module_withdrawal_bank_account_ifsc_empty_message' )) ) ;
			}

			$bank_name      = isset( $post_data[ 'wal_bank_gateway_bank_name' ] ) ? wc_clean( wp_unslash( $post_data[ 'wal_bank_gateway_bank_name' ] ) ) : '' ;
			$routing_number = isset( $post_data[ 'wal_bank_gateway_routing_number' ] ) ? wc_clean( wp_unslash( $post_data[ 'wal_bank_gateway_routing_number' ] ) ) : '' ;
			$iban_number    = isset( $post_data[ 'wal_bank_gateway_iban_number' ] ) ? wc_clean( wp_unslash( $post_data[ 'wal_bank_gateway_iban_number' ] ) ) : '' ;
			$swift_code     = isset( $post_data[ 'wal_bank_gateway_swift_code' ] ) ? wc_clean( wp_unslash( $post_data[ 'wal_bank_gateway_swift_code' ] ) ) : '' ;

			return array(
				'account_name'   => $account_name,
				'account_number' => $account_number,
				'ifsc_code'      => $ifsc_code,
				'bank_name'      => $bank_name,
				'routing_number' => $routing_number,
				'iban_number'    => $iban_number,
				'swift_code'     => $swift_code,
					) ;
		}
	}

}
