<?php

/**
 * Abstract Withdrawal Gateway Class.
 * 
 * @since 1.8
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Withdrawal_Gateway' ) ) {

	/**
	 * Class.
	 */
	abstract class WAL_Withdrawal_Gateway {

		/**
		 * ID.
		 * 
		 * @var string
		 */
		protected $id ;

		/**
		 * Title.
		 * 
		 * @var string
		 */
		protected $title ;

		/**
		 * Description.
		 * 
		 * @var string
		 */
		protected $description ;

		/**
		 * Plugin slug.
		 * 
		 * @var string
		 */
		protected $plugin_slug = 'wal' ;

		/**
		 * Class Constructor.
		 */
		public function __construct() {
			$this->title = get_option( $this->get_option_key( 'title' ) ) ;

			$this->process_actions() ;
		}

		/**
		 * Get the id.
		 * 
		 * @return string
		 */
		public function get_id() {
			return $this->id ;
		}

		/**
		 * Get the title.
		 * 
		 * @return string
		 */
		public function get_title() {
			return $this->title ;
		}

		/**
		 * Get the description.
		 * 
		 * @return string
		 */
		public function get_description() {
			return $this->description ;
		}

		/**
		 * Get the plugin slug.
		 * 
		 * @return string
		 */
		public function get_plugin_slug() {
			return $this->plugin_slug ;
		}

		/**
		 * Is valid gateway?.
		 * 
		 * @return bool
		 */
		public function is_valid_gateway() {
			/**
			 * This hook is used to check if it is valid gateway.
			 * 
			 * @since 1.8
			 */
			return apply_filters( $this->get_plugin_slug() . '_module_withdrawal_' . $this->get_id() . '_is_valid_gateway', $this->is_enabled() && $this->is_valid_credentials(), $this ) ;
		}

		/**
		 * Is valid credentials?.
		 * 
		 * @return bool
		 */
		public function is_valid_credentials() {
			/**
			 * This hook is used to check if it is valid credentials.
			 * 
			 * @since 1.8
			 */
			return apply_filters( $this->get_plugin_slug() . '_module_withdrawal_' . $this->get_id() . '_is_valid_credentials', true, $this ) ;
		}

		/**
		 * Is module enabled?.
		 * 
		 * @return bool
		 */
		public function is_enabled() {
			/**
			 * This hook is used to alter the current module enable status.
			 * 
			 * @since 1.8
			 */
			return apply_filters( $this->get_plugin_slug() . '_module_withdrawal_' . $this->get_id() . '_enabled', 'yes' === get_option( $this->get_enable_key() ), $this ) ;
		}

		/**
		 * Get the enable key.
		 * 
		 * @return string
		 */
		public function get_enable_key() {
			return $this->get_option_key( 'enabled' ) ;
		}

		/**
		 * Get the option key.
		 * 
		 * @return string
		 */
		public function get_option_key( $key ) {
			return sanitize_key( $this->get_plugin_slug() . '_module_withdrawal_' . $this->get_id() . '_' . $key ) ;
		}

		/**
		 * Get the settings array.
		 * 
		 * @return array
		 * */
		public function get_settings_array() {
			return array() ;
		}

		/**
		 * Output the settings fields.
		 * */
		public function output_fields() {
			WC_Admin_Settings::output_fields( $this->get_settings_array() ) ;
		}

		/**
		 * Save the settings fields.
		 * */
		public function save_fields() {
			WC_Admin_Settings::save_fields( $this->get_settings_array() ) ;
		}

		/**
		 * Reset the settings fields.
		 * */
		public function reset_fields() {
			WAL_Settings::reset_fields( $this->get_settings_array() ) ;
		}

		/**
		 * Process actions.
		 */
		protected function process_actions() {
			if ( ! $this->is_enabled() ) {
				return ;
			}

			$this->actions() ;

			if ( ! is_admin() || defined( 'DOING_AJAX' ) ) {
				$this->frontend_action() ;

				add_action( 'wal_wallet_withdrawal_after_fields', array( $this, 'render_credential_fields' ) ) ;
			}
		}

		/**
		 * Actions.
		 */
		protected function actions() {
		}

		/**
		 * Front end actions.
		 */
		protected function frontend_action() {
		}

		/**
		 * Render credential fields.
		 */
		public function render_credential_fields() {
			wal_get_template( 'modules/wallet-withdrawal/' . $this->get_id() . '-gateway-fields.php', $this->get_customer_credentials( WAL_Current_User_Wallet::get_wallet_id() ) ) ;
		}

		/**
		 * Prepare the post data.
		 * 
		 * @return array
		 * */
		public function prepare_post_data( &$post_data ) {
			return array() ;
		}

		/**
		 * Get the fee amount.
		 * 
		 * @return float
		 * */
		public function get_fee_amount( $amount ) {
			return 0 ;
		}

		/**
		 * Get the customer credentials.
		 * 
		 * @return array
		 */
		public function get_customer_credentials( $wallet_id ) {
			$payment_gateway_credentials = get_post_meta( $wallet_id, 'wal_withdrawal_payment_gateway_credentials', true ) ;
			$credentials                 = isset( $payment_gateway_credentials[ $this->get_id() ] ) ? $payment_gateway_credentials[ $this->get_id() ] : array() ;
			$credentials                 = wp_parse_args( $credentials, $this->default_credentials() ) ;

			/**
			 * This hook is used to alter the current payment gateway customer credentials.
			 * 
			 * @since 1.8
			 */
			return apply_filters( 'wal_wallet_withdrawal_payment_gateway_default_credentials_' . $this->get_id(), $credentials, $wallet_id ) ;
		}

		/**
		 * Default credentials.
		 * 
		 * @return array
		 * */
		public function default_credentials() {
			/**
			 * This hook is used to alter the current payment gateway default credentials.
			 * 
			 * @since 1.8
			 */
			return apply_filters( 'wal_wallet_withdrawal_payment_gateway_default_credentials_' . $this->get_id(), array() ) ;
		}

		/**
		 * Process payment.
		 * */
		public function process_payment( $withdrawal ) {
			return true ;
		}
	}

}
