<?php

/**
 * Module - Wallet Withdrawal.
 * 
 * @since 1.8
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Wallet_Withdrawal_Module')) {

	/**
	 * Class.
	 * */
	class WAL_Wallet_Withdrawal_Module extends WAL_Modules {

		/**
		 * Gateways.
		 * 
		 * @var array
		 */
		private $gateways = array();

		/**
		 * Available Gateways.
		 * 
		 * @var array
		 */
		private $available_gateways = array();

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id = 'wallet_withdrawal';
			$this->title = __('Wallet Withdrawal', 'wallet-for-woocommerce');
			$this->description = __('Users can withdraw their funds to the preferred payment method by submitting a request.', 'wallet-for-woocommerce');
			$this->tooltip = __('By enabling this checkbox, you can allow your users to withdraw their funds to the preferred payment method by submitting a request.', 'wallet-for-woocommerce');

			// Save the gateway settings.
			add_action(sanitize_key($this->get_plugin_slug() . '_after_module_' . $this->get_id() . '_settings_saved'), array( $this, 'save_gateway_settings' ));
			// Reset the gateway settings.
			add_action(sanitize_key($this->get_plugin_slug() . '_after_module_' . $this->get_id() . '_settings_reset'), array( $this, 'reset_gateway_settings' ));

			parent::__construct();
		}

		/**
		 * Get the payment gateways.
		 * 
		 * @return array
		 */
		public function payment_gateways() {
			if ($this->gateways) {
				return $this->gateways;
			}

			if (!class_exists('WAL_Withdrawal_Gateway')) {
				include WAL_PLUGIN_PATH . '/inc/modules/withdrawal-gateways/abstract-wal-withdrawal-gateway.php';
			}

			$default_gateway_classes = array(
				'bank' => 'WAL_Withdrawal_Bank_Gateway',
				'paypal' => 'WAL_Withdrawal_Paypal_Gateway',
			);

			foreach ($default_gateway_classes as $file_name => $gateway_class) {

				// Include withdrawal payment gateway file.
				include 'withdrawal-gateways/class-' . $file_name . '.php';

				$gateway = new $gateway_class();

				// Add withdrawal payment gateway object.
				$this->gateways[$gateway->get_id()] = $gateway;

				if ($gateway->is_valid_credentials()) {
					$this->available_gateways[$gateway->get_id()] = $gateway;
				}
			}

			return $this->gateways;
		}

		/**
		 * Get the available payment gateways.
		 * 
		 * @return array
		 */
		public function available_payment_gateways() {
			if ($this->available_gateways) {
				return $this->available_gateways;
			}

			$this->payment_gateways();

			return $this->available_gateways;
		}

		/**
		 * Get the payment gateway by id.
		 * 
		 * @return object/bool
		 * */
		public function get_payment_gateway_by_id( $gateway_id ) {
			$gateways = $this->payment_gateways();

			return isset($gateways[$gateway_id]) ? $gateways[$gateway_id] : false;
		}

		/**
		 * Get the setting tabs.
		 * 
		 * @return array
		 */
		public function setting_tabs() {
			$tabs = array(
				'general' => __('General', 'wallet-for-woocommerce'),
				'payment_method' => __('Payment Methods', 'wallet-for-woocommerce'),
				'localization' => __('Localization', 'wallet-for-woocommerce'),
				'messages' => __('Messages', 'wallet-for-woocommerce'),
			);
			/**
			 * This hook is used to alter the current module tabs.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->get_plugin_slug() . '_get_module_tabs_' . $this->get_id(), $tabs);
		}

		/**
		 * Get settings for general section array.
		 * 
		 * @return array
		 */
		protected function general_section_array() {
			$section_fields = array();
			$wp_user_roles = wal_get_wp_user_roles();

			// General settings section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('General Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_withdrawal_general_options',
			);
			$section_fields[] = array(
				'title' => __('Minimum Amount for Wallet Withdrawal ', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_min_amount'),
				'desc' => __('The minimum amount which the user can request for withdrawal', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Amount for Wallet Withdrawal', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_max_amount'),
				'desc' => __('The maximum amount which the user can request for withdrawal', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Number of Withdrawal Request Per Day', 'wallet-for-woocommerce'),
				'type' => 'number',
				'default' => '',
				'autoload' => false,
				'custom_attributes' => array( 'min' => 1 ),
				'id' => $this->get_option_key('wallet_withdrawal_count_per_day'),
				'desc' => __('You can set users can give maximum number of withdrawal request per day', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_withdrawal_general_options',
			);
			// General settings section end.
			// wallet withdrawal display section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Display Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_withdrawal_display_options',
			);
			$section_fields[] = array(
				'title' => __('Wallet Withdrawal Log Pagination Size', 'wallet-for-woocommerce'),
				'type' => 'number',
				'default' => '5',
				'autoload' => false,
				'custom_attributes' => array( 'min' => 1 ),
				'id' => $this->get_option_key('wallet_withdrawal_transactions_pagination_count'),
				'desc' => __('You can set the number of entries to display in the transaction log[Wallet Dashboard -> Wallet Withdrawal -> Transactions].', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_withdrawal_display_options',
			);
			// wallet withdrawal display section end.
			// Restriction section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Restriction Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_withdrawal_restriction_options',
			);
			$section_fields[] = array(
				'title' => __('User(s)/User Role(s) Selection', 'wallet-for-woocommerce'),
				'type' => 'select',
				'default' => '1',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_user_restriction_type'),
				'class' => 'wal-user-restriction-type',
				'options' => array(
					'1' => __('All Users', 'wallet-for-woocommerce'),
					'2' => __('Include Users', 'wallet-for-woocommerce'),
					'3' => __('Exclude Users', 'wallet-for-woocommerce'),
					'4' => __('Include User Roles', 'wallet-for-woocommerce'),
					'5' => __('Exclude User Roles', 'wallet-for-woocommerce'),
				),
			);
			$section_fields[] = array(
				'title' => __('Select the User(s) to Include', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'ajaxmultiselect',
				'list_type' => 'customers',
				'action' => 'wal_json_search_customers',
				'default' => array(),
				'autoload' => false,
				'placeholder' => __('Select a User', 'wallet-for-woocommerce'),
				'class' => 'wal-user-restriction-field wal-include-users-field',
				'id' => $this->get_option_key('wallet_withdrawal_include_users'),
			);
			$section_fields[] = array(
				'title' => __('Select the User(s) to Exclude', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'ajaxmultiselect',
				'list_type' => 'customers',
				'class' => 'wal-user-restriction-field wal-exclude-users-field',
				'action' => 'wal_json_search_customers',
				'default' => array(),
				'autoload' => false,
				'placeholder' => __('Select a User', 'wallet-for-woocommerce'),
				'id' => $this->get_option_key('wallet_withdrawal_exclude_users'),
			);
			$section_fields[] = array(
				'title' => __('Select the User role(s) to Include', 'wallet-for-woocommerce'),
				'type' => 'multiselect',
				'default' => array(),
				'autoload' => false,
				'custom_attributes' => array( 'placeholder' => __('Select a User Role', 'wallet-for-woocommerce') ),
				'class' => 'wal_select2 wal-user-restriction-field wal-include-user-roles-field',
				'id' => $this->get_option_key('wallet_withdrawal_include_user_roles'),
				'options' => $wp_user_roles,
			);
			$section_fields[] = array(
				'title' => __('Select the User role(s) to Exclude', 'wallet-for-woocommerce'),
				'type' => 'multiselect',
				'default' => array(),
				'autoload' => false,
				'custom_attributes' => array( 'placeholder' => __('Select a User Role', 'wallet-for-woocommerce') ),
				'class' => 'wal_select2 wal-user-restriction-field wal-exclude-user-roles-field',
				'id' => $this->get_option_key('wallet_withdrawal_exclude_user_roles'),
				'options' => $wp_user_roles,
			);

			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_withdrawal_restriction_options',
			);
			// Restriction section end.
			// OTP section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('OTP Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_withdrawal_otp_options',
			);
			$section_fields[] = array(
				'title' => __('Enable OTP', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_otp_enabled'),
				'desc' => __('By enabling this checkbox, you can allow your users to receive an OTP via email while submitting the withdrawal request.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('OTP Length', 'wallet-for-woocommerce'),
				'type' => 'number',
				'default' => '6',
				'autoload' => false,
				'class' => 'wal-wallet-withdrawal-otp-field',
				'custom_attributes' => array( 'min' => 1 ),
				'id' => $this->get_option_key('wallet_withdrawal_otp_length'),
			);
			$section_fields[] = array(
				'title' => __('OTP Validity in Minutes', 'wallet-for-woocommerce'),
				'type' => 'number',
				'default' => '5',
				'autoload' => false,
				'class' => 'wal-wallet-withdrawal-otp-field',
				'custom_attributes' => array( 'min' => 1 ),
				'id' => $this->get_option_key('wallet_withdrawal_otp_validity'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_withdrawal_otp_options',
			);
			// OTP section end.

			return $section_fields;
		}

		/**
		 * Get the settings for localization section array.
		 * 
		 * @return array
		 */
		protected function localization_section_array() {
			$section_fields = array();

			// Dashboard endpoints section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Endpoints', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_withdrawal_dashboard_endpoints_options',
			);
			$section_fields[] = array(
				'title' => __('Withdrawal Endpoint', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'wallet_withdrawal',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_details_menu_endpoint'),
			);
			$section_fields[] = array(
				'title' => __('Withdrawal Transactions Endpoint', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'wallet_withdrawal_transactions',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_transaction_menu_endpoint'),
			);
			$section_fields[] = array(
				'title' => __('Withdrawal Request Endpoint', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'wallet_withdrawal_request',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_menu_endpoint'),
			);
			$section_fields[] = array(
				'title' => __('Withdrawal Payment Settings Endpoint', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'wallet_withdrawal_settings',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_settings_menu_endpoint'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_withdrawal_dashboard_endpoints_options',
			);
			// Dashboard endpoints section end.
			// Dashboard menu labels section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Menu Labels', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_withdrawal_menu_labels_options',
			);
			$section_fields[] = array(
				'title' => __('Wallet Withdrawal Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Wallet Withdrawal',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_details_menu_label'),
			);
			$section_fields[] = array(
				'title' => __('Withdrawal Transactions Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Transactions',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_transaction_menu_label'),
			);
			$section_fields[] = array(
				'title' => __('Withdrawal Request Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Withdrawal Request',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_menu_label'),
			);
			$section_fields[] = array(
				'title' => __('Withdrawal Payment Settings Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Payments',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_settings_menu_label'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_withdrawal_menu_labels_options',
			);
			// Dashboard menu labels section end.
			// Dashboard transaction labels section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Transaction Labels', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_withdrawal_transaction_labels_options',
			);
			$section_fields[] = array(
				'title' => __('Amount Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Amount',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_transaction_amount_label'),
			);
			$section_fields[] = array(
				'title' => __('Fee Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Fee',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_transaction_fee_label'),
			);
			$section_fields[] = array(
				'title' => __('Payment Method Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Payment Method',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_transaction_payment_method_label'),
			);
			$section_fields[] = array(
				'title' => __('Status Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Status',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_transaction_status_label'),
			);
			$section_fields[] = array(
				'title' => __('Created Date Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Created Date',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_transaction_created_date_label'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_withdrawal_transaction_labels_options',
			);
			// Dashboard transaction labels section end.
			// Dashboard overview labels section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Overview Labels', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_withdrawal_overview_labels_options',
			);
			$section_fields[] = array(
				'title' => __('Total Withdrawn Amount Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Total Withdrawn Amount',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_withdrew_total_label'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_withdrawal_overview_labels_options',
			);
			// Dashboard overview labels section end.
			// Fund withdraw form labels section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Withdrawal Form Labels', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_withdraw_form_labels_options',
			);
			$section_fields[] = array(
				'title' => __('Available Wallet Balance Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Available Wallet Balance',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_balance_label'),
			);
			$section_fields[] = array(
				'title' => __('Amount to Withdraw Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Amount to Withdraw',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_amount_label'),
			);
			$section_fields[] = array(
				'title' => __('Amount to Withdraw Placeholder', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Enter the Amount',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_amount_placeholder_label'),
			);
			$section_fields[] = array(
				'title' => __('Payment Method Selection', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Payment Method',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_payment_method_label'),
			);
			$section_fields[] = array(
				'title' => __('Payment Method Default Selection Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Select the Payment Method',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_payment_method_option_label'),
			);
			$section_fields[] = array(
				'title' => __('Submit Button Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Submit',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_button_label'),
			);
			$section_fields[] = array(
				'title' => __('Fee Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Fee',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_fee_label'),
			);
			$section_fields[] = array(
				'title' => __('OTP Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Verify OTP',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_otp_label'),
			);
			$section_fields[] = array(
				'title' => __('Confirm Button Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Confirm',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_request_confirm_button_label'),
			);
			$section_fields[] = array(
				'title' => __('Save Settings Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Save Settings',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_save_settings_label'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_withdraw_form_labels_options',
			);
			// Fund withdraw form labels section end.

			return $section_fields;
		}

		/**
		 * Get the settings for messages section array.
		 * 
		 * @return array
		 */
		protected function messages_section_array() {
			$section_fields = array();

			// Log section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Transaction Log', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_withdrawal_log_options',
			);
			$section_fields[] = array(
				'title' => __('Wallet Withdrawal Log - Submitted', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Successfully submitted the Wallet Funds Withdrawal Request',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_debit_log_message'),
			);
			$section_fields[] = array(
				'title' => __('Wallet Withdrawal Log - Cancelled', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Withdrawal Request has been Cancelled',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_credit_log_message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_withdrawal_log_options',
			);
			// Log section end.
			// Wallet Withdraw section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Wallet Withdrawal Messages', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_withdrawal_message_options',
			);
			$section_fields[] = array(
				'title' => __("Message to display when a User didn't enter the Funds", 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter amount to withdraw',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_amount_empty_message'),
			);
			$section_fields[] = array(
				'title' => __("Message to Display when a User didn't Select the Payment Method", 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please select a payment method',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_payment_method_empty_message'),
			);
			$section_fields[] = array(
				'title' => __('Minimum Amount Required to Withdraw Funds in a Single Transaction', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter an amount more than {minimum_amount}',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_minimum_amount_message'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Amount that can be Withdrawn in a Single Transaction', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter an amount less than {maximum_amount}',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_maximum_amount_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when a User has given the Maximum Number of Withdrawal Request Per Day', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'You are allowed to withdraw your funds a maximum of {withdraw_count} time(s) per day. Since you have reached the count, you cannot withdraw your funds anymore today.',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_maximum_count_per_day_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when a User tries to Withdraw more than the Available Funds on their Wallet', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Insufficient balance to withdraw funds',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_insufficient_wallet_fund_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when the User(s)/User Role(s) is Restricted to Withdraw Funds ', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Currently, you are restricted to withdraw your funds.',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_user_restricted_message'),
			);
			$section_fields[] = array(
				'title' => __('Withdrawal Request Success Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Withdrawal Request Submitted Successfully',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_success_message'),
			);
			$section_fields[] = array(
				'title' => __('Withdrawal Settings Success Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Payment Method Saved Successfully',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_settings_saved_success_message'),
			);
			$section_fields[] = array(
				'title' => __('Invalid OTP Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'The OTP which you have entered is incorrect. Please enter the correct OTP.',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_invalid_otp_message'),
			);
			$section_fields[] = array(
				'title' => __('Expired OTP Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'The OTP which you have entered got expired.',
				'autoload' => false,
				'id' => $this->get_option_key('wallet_withdrawal_otp_expired_message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_withdrawal_message_options',
			);

			// Fund Transfer section end.
			return $section_fields;
		}

		/**
		 * Output the tab content.
		 */
		public function output_tab_content() {
			global $current_tab, $current_section;
			if ('payment_method' !== $current_tab) {
				return;
			}

			if (!$current_section) {
				include_once WAL_ABSPATH . 'inc/modules/views/html-withdrawal-payment-gateways.php' ;
			} else {
				$payment_gateway = $this->get_payment_gateway_by_id($current_section);
				if (!is_object($payment_gateway)) {
					return;
				}

				$payment_gateway->output_fields();
			}
		}

		/**
		 * Save the gateway settings.
		 */
		public function save_gateway_settings() {
			global $current_tab, $current_section;
			if ('payment_method' !== $current_tab) {
				return;
			}

			if ($current_section) {
				$gateway = $this->get_payment_gateway_by_id($current_section);
				if (!$gateway) {
					return;
				}

				$gateway->save_fields();
			} else {
				$gateways = $this->payment_gateways();
				foreach ($gateways as $gateway) {
					// Enable/ disable the gateway.
					$value = ( isset($_REQUEST[$gateway->get_enable_key()]) ) ? 'yes' : 'no';

					update_option($gateway->get_enable_key(), $value);
				}

				$this->gateways = null;
			}
		}

		/**
		 * Reset the gateway settings.
		 */
		public function reset_gateway_settings() {
			global $current_tab, $current_section;
			if ('payment_method' !== $current_tab) {
				return;
			}

			if ($current_section) {
				$gateway = $this->get_payment_gateway_by_id($current_section);
				if (!$gateway) {
					return;
				}

				$gateway->reset_fields();
			} else {
				$gateways = $this->payment_gateways();
				foreach ($gateways as $gateway) {
					update_option($gateway->get_enable_key(), 'no');
				}

				$this->gateways = null;
			}
		}

		/**
		 * Include files.
		 */
		protected function include_files() {
			include_once WAL_ABSPATH . 'inc/entity/class-wal-wallet-withdrawal.php' ;
		}

		/**
		 * Enqueue admin JS files.
		 */
		protected function admin_external_js_files() {
			// Wallet withdrawal.
			wp_enqueue_script('wal-wallet-withdrawal', WAL_PLUGIN_URL . '/assets/js/admin/wallet-withdrawal.js', array( 'jquery', 'jquery-blockui' ), WAL_VERSION);

			wp_localize_script(
					'wal-wallet-withdrawal', 'wal_wallet_withdrawal_params', array(
				'approve_confirm_msg' => __('Are you sure you want to approve the request?', 'wallet-for-woocommerce'),
				'cancel_confirm_msg' => __('Are you sure you want to cancel the request?', 'wallet-for-woocommerce'),
				'wallet_withdrawal_nonce' => wp_create_nonce('wal-wallet-withdrawal'),
					)
			);
		}

		/**
		 * Enqueue front end JS files.
		 * 
		 * @since 2.8.0
		 */
		protected function frontend_external_js_files() {
			// Wallet withdrawal.
			wp_enqueue_script('wal-wallet-withdrawal', WAL_PLUGIN_URL . '/assets/js/frontend/wallet-withdrawal.js', array( 'jquery', 'jquery-blockui' ), WAL_VERSION, WAL_Fronend_Assets::$in_footer);

			wp_localize_script(
					'wal-wallet-withdrawal', 'wal_wallet_withdrawal_params', array(
				'wallet_withdrawal_nonce' => wp_create_nonce('wal-wallet-withdrawal'),
					)
			);
		}

		/**
		 * Actions.
		 */
		protected function actions() {
			//Add the email classes.
			add_filter('wal_wallet_email_classes', array( $this, 'email_classes' ), 10, 1);
			//Transaction logs for a selected page.
			add_action('wp_ajax_wal_wallet_withdrawal_transactions_pagination', array( $this, 'transactions_pagination' ));
			//Set Wallet withdrawal edit post.
			add_action('wp_ajax_wal_set_wallet_withdrawal_edit_post', array( $this, 'set_wallet_withdrawal_edit_post' ));
			//Get wallet withdrawal cancelled reason.
			add_action('wp_ajax_wal_get_wallet_withdrawal_cancelled_reason', array( $this, 'get_wallet_withdrawal_cancelled_reason' ));
			//Cancel wallet withdrawal
			add_action('wp_ajax_wal_cancel_wallet_withdrawal', array( $this, 'cancel_wallet_withdrawal' ));
			//Get wallet withdrawal popup content.
			add_action('wp_ajax_wal_get_wallet_withdrawal_cancel_popup_content', array( $this, 'get_wallet_withdrawal_cancel_popup_content' ));
		}

		/**
		 * Front end actions.
		 */
		protected function frontend_action() {
			//Add the wallet endpoints.
			add_filter('wal_wallet_endpoints', array( $this, 'add_endpoints' ), 10, 1);
			//Add the dashboard menus.
			add_filter('wal_dashboard_menus', array( $this, 'add_dashboard_menus' ), 10, 1);
			//Add the hook for wallet withdrawal transactions menu content.
			add_action('wal_dashboard_menu_content_wallet_withdrawal_transactions', array( $this, 'wallet_withdrawal_transactions_menu_content' ));
			//Add the hook for wallet withdrawal request menu content.
			add_action('wal_dashboard_menu_content_wallet_withdrawal_request', array( $this, 'wallet_withdrawal_request_menu_content' ));
			//Add the wallet withdrawal overview data.
			add_filter('wal_wallet_overview_data', array( $this, 'add_overview_data' ), 10, 1);
			// Process the wallet withdrawal.
			add_action('wp_loaded', array( $this, 'process_wallet_withdrawal' ));
			//Get user withdrawal cancelled reason.
			add_action('wp_ajax_wal_get_user_withdrawal_cancelled_reason', array( $this, 'get_user_withdrawal_cancelled_reason' ));
		}

		/**
		 * Add the custom email classes.
		 * 
		 * @return array
		 */
		public function email_classes( $classes ) {
			$classes = ( wal_check_is_array($classes) ) ? $classes : array();

			$classes['admin-wallet-withdrawal-request-received'] = 'WAL_Admin_Wallet_Withdrawal_Request_Received_Notification';
			$classes['customer-wallet-withdrawal-request-submitted'] = 'WAL_Customer_Wallet_Withdrawal_Request_Submitted_Notification';
			$classes['customer-wallet-withdrawal-request-approved'] = 'WAL_Customer_Wallet_Withdrawal_Request_Approved_Notification';
			$classes['customer-wallet-withdrawal-request-cancelled'] = 'WAL_Customer_Wallet_Withdrawal_Request_Cancelled_Notification';
			$classes['customer-wallet-withdrawal-otp'] = 'WAL_Customer_Wallet_Withdrawal_OTP_Notification';

			return $classes;
		}

		/**
		 * Transactions for a selected page.
		 * 
		 * @return void
		 */
		public function transactions_pagination() {
			check_ajax_referer('wal-pagination-nonce', 'wal_security');

			try {

				if (!isset($_POST) || !isset($_POST['page_number'])) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				// Sanitize post values.
				$current_page = !empty($_POST['page_number']) ? absint($_POST['page_number']) : 0; // @codingStandardsIgnoreLine.

				$per_page = wal_wallet_withdrawal_pagination_count();
				$offset = ( $current_page - 1 ) * $per_page;

				// Get the transaction logs based on per page count.
				$withdrawal_ids = wal_user_wallet_withdrawals();
				$withdrawal_ids = array_slice($withdrawal_ids, $offset, $per_page);

				// Get the transaction logs content.
				$html = wal_get_template_html(
						'modules/wallet-withdrawal/transactions.php', array(
					'withdrawal_ids' => $withdrawal_ids,
						)
				);

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Add the custom wallet endpoints.
		 * 
		 * @return array
		 */
		public function add_endpoints( $endpoints ) {
			$endpoints = ( wal_check_is_array($endpoints) ) ? $endpoints : array();

			$endpoints['wallet_withdrawal_details'] = wal_get_wallet_withdrawal_details_menu_endpoint();
			$endpoints['wallet_withdrawal_transactions'] = wal_get_wallet_withdrawal_transaction_menu_endpoint();
			$endpoints['wallet_withdrawal_request'] = wal_get_wallet_withdrawal_request_menu_endpoint();

			return $endpoints;
		}

		/**
		 * Add the custom dashboard menus.
		 * 
		 * @return array
		 */
		public function add_dashboard_menus( $menus ) {
			$menus = ( wal_check_is_array($menus) ) ? $menus : array();

			$submenus = array(
				'wallet_withdrawal_transactions' => array(
					'name' => wal_get_wallet_withdrawal_transaction_menu_label(),
					'endpoint' => wal_get_wallet_withdrawal_transaction_menu_endpoint(),
				),
				'wallet_withdrawal_request' => array(
					'name' => wal_get_wallet_withdrawal_request_menu_label(),
					'endpoint' => wal_get_wallet_withdrawal_request_menu_endpoint(),
				),
			);

			$menus['wallet_withdrawal_details'] = array(
				'name' => wal_get_wallet_withdrawal_details_menu_label(),
				'endpoint' => wal_get_wallet_withdrawal_details_menu_endpoint(),
				'submenus' => $submenus,
			);

			return $menus;
		}

		/**
		 * Add the fund transfer overview data.
		 * 
		 * @return array
		 */
		public function add_overview_data( $data ) {
			$data['fund_withdrew_total'] = array(
				'label' => wal_get_fund_withdrew_total_label(),
				'value' => wal_convert_price_by_currency(WAL_Current_User_Wallet::get_withdrew_total()),
			);

			return $data;
		}

		/**
		 * Render the transactions menu content in the dashboard.
		 */
		public function wallet_withdrawal_transactions_menu_content() {
			$withdrawal_ids = wal_user_wallet_withdrawals();
			$per_page = intval(wal_wallet_withdrawal_pagination_count());

			$current_page = 1;
			$offset = ( $per_page * $current_page ) - $per_page;
			$page_count = ceil(count($withdrawal_ids) / $per_page);

			$table_args = array(
				'withdrawal_ids' => array_slice($withdrawal_ids, $offset, $per_page),
				'pagination' => array(
					'page_count' => $page_count,
					'current_page' => $current_page,
					'prev_page_count' => ( ( $current_page - 1 ) == 0 ) ? ( $current_page ) : ( $current_page - 1 ),
					'next_page_count' => ( ( $current_page + 1 ) <= ( $page_count ) ) ? ( $current_page + 1 ) : ( $current_page ),
			),
			);

			wal_get_template('modules/wallet-withdrawal/transactions-wrapper.php', $table_args);
		}

		/**
		 * Render the wallet withdrawal request menu content in the dashboard.
		 */
		public function wallet_withdrawal_request_menu_content() {
			global $wal_confirm_action;
			$is_valid_user = $this->is_valid_user();
			if ($wal_confirm_action && $is_valid_user) {
				$payment_gateway_id = isset($_REQUEST['wal_withdrawal_payment_gateway']) ? wc_clean(wp_unslash($_REQUEST['wal_withdrawal_payment_gateway'])) : '';
				$amount = isset($_REQUEST['wal_withdrawal_request_amount']) ? wc_clean(wp_unslash($_REQUEST['wal_withdrawal_request_amount'])) : '';
				$payment_gateway = $this->get_payment_gateway_by_id($payment_gateway_id);
				$args = array(
					'gateway_id' => $payment_gateway_id,
					'amount' => $amount,
					'fee' => $payment_gateway->get_fee_amount($amount),
					'credential_details' => wal_format_payment_gateway_credentials($payment_gateway->get_customer_credentials(WAL_Current_User_Wallet::get_wallet_id()), $payment_gateway->get_credential_labels()),
				);
				wal_get_template('modules/wallet-withdrawal/withdrawal-request-confirmation.php', $args);
			} else {

				if (!$is_valid_user) {
					WAL_Dashboard_Handler::add_error(get_option('wal_module_wallet_withdrawal_user_restricted_message'));
				}

				wal_get_template('modules/wallet-withdrawal/withdrawal-request-form.php', array( 'show' => $is_valid_user ));
			}
		}

		/**
		 * Process the wallet withdrawal.
		 */
		public function process_wallet_withdrawal() {
			$nonce_value = isset($_POST['wal-wallet-withdrawal-nonce']) ? wc_clean(wp_unslash($_POST['wal-wallet-withdrawal-nonce'])) : null;
			if (!isset($_POST['wal-action']) || empty($_POST['wal-action']) || !wp_verify_nonce($nonce_value, 'wal-wallet-withdrawal')) {
				return;
			}

			try {   
				$transfer_action = isset($_POST['wal-action']) ? wc_clean(wp_unslash($_POST['wal-action'])) : '';           
				$payment_gateway = isset($_POST['wal_withdrawal_payment_gateway']) ? wc_clean(wp_unslash($_POST['wal_withdrawal_payment_gateway'])) : '';
				if (!$payment_gateway) {
					throw new Exception(wp_kses_post(get_option('wal_module_wallet_withdrawal_payment_method_empty_message')));
				}

				$gateway = $this->get_payment_gateway_by_id($payment_gateway);
				if (!$gateway) {
					throw new Exception(esc_html__('Cannot process action', 'wallet-for-woocommerce'));
				}

				// Check if the current user is valid.
				if ('wal_active' != WAL_Current_User_Wallet::get_status()) {
					throw new Exception(get_option('wal_module_wallet_withdrawal_user_restricted_message'));
				}

				if ( 'wallet-withdrawal' == $transfer_action) {
					// Prepare the post data.
					$payment_gateway_credentials = WAL_Current_User_Wallet::get_wallet()->get_withdrawal_payment_gateway_credentials();
					$payment_gateway_credentials[$gateway->get_id()] = $gateway->prepare_post_data($_POST);
					$args = array(
						'wal_withdrawal_payment_gateway' => $payment_gateway,
						'wal_withdrawal_payment_gateway_credentials' => $payment_gateway_credentials,
					);
	
					//Update the wallet data.
					wal_update_wallet(WAL_Current_User_Wallet::get_wallet_id(), $args);
				}
				
				$amount = isset($_POST['wal_withdrawal_request_amount']) ? wc_clean(wp_unslash($_POST['wal_withdrawal_request_amount'])) : '';
				$amount = wal_convert_price( $amount , true);           
				// Validate the wallet withdrawal form
				$this->validate_wallet_withdrawal_form($payment_gateway, $amount);

				global $wal_confirm_action;

				$wal_confirm_action = true;

				// $payment_gateway = $this->get_payment_gateway_by_id($payment_gateway);
				if ('confirm' == $transfer_action) {
					// Verify the OTP.
					$this->verify_otp();

					$args = array(
						'wal_user_id' => WAL_Current_User_Wallet::get_user_id(),
						'wal_payment_method' => $payment_gateway,
						'wal_amount' => (float) $amount,
						'wal_fee' => $gateway->get_fee_amount($amount),
						'wal_payment_method_credentials' => $gateway->get_customer_credentials(WAL_Current_User_Wallet::get_wallet_id()),
					);
										
					$withdrawal_id = wal_create_new_wallet_withdrawal($args, array( 'post_parent' => WAL_Current_User_Wallet::get_wallet_id() ));
					if (!$withdrawal_id) {
						throw new Exception(esc_html__('Some issue to proceed the action.', 'wallet-for-woocommerce'));
					}

					$transaction_id = wal_wallet_withdrawal_debit_fund($withdrawal_id);
					if (!$transaction_id) {
						throw new Exception(esc_html__('Some issue to proceed the action.', 'wallet-for-woocommerce'));
					}

					WAL_Dashboard_Handler::add_message(get_option('wal_module_wallet_withdrawal_success_message'));

					$wal_confirm_action = false;

					$keys = array(
						'wal-wallet-withdrawal-nonce',
						'wal_withdrawal_payment_gateway',
						'wal_withdrawal_request_amount',
					);

					wal_unset_global_variable_values($keys);

					//Reset wallet.
					WAL_Current_User_Wallet::reset();
					/**
					 * This hook is used to do extra action after wallet withdraw request submitted.
					 * 
					 * @since 1.8
					 */
					do_action('wal_wallet_withdrawal_request_submitted', $withdrawal_id);
				} else {
					/**
					 * This hook is used to do extra action after wallet withdrawal processed.
					 * 
					 * @since 1.8
					 */
					do_action('wal_processed_wallet_withdrawal', WAL_Current_User_Wallet::get_wallet_id(), $amount);
				}
			} catch (Exception $ex) {

				WAL_Dashboard_Handler::add_error($ex->getMessage());
			}
		}

		/**
		 * Validate the wallet withdrawal form.
		 */
		public function validate_wallet_withdrawal_form( $payment_gateway_id, $amount ) {
			//Validate the amount.
			if (empty($amount)) {
				throw new Exception(wp_kses_post(get_option('wal_module_wallet_withdrawal_amount_empty_message')));
			}

			// Validate the payment gateway selected.
			if (!$payment_gateway_id) {
				throw new Exception(wp_kses_post(get_option('wal_module_wallet_withdrawal_payment_method_empty_message')));
			}

			// Validate the payment gateway is valid.
			$payment_gateway = $this->get_payment_gateway_by_id($payment_gateway_id);
			if (!$payment_gateway) {
				throw new Exception(esc_html__('Cannot process action', 'wallet-for-woocommerce'));
			}

			$fee = $payment_gateway->get_fee_amount($amount);

			//Validate the user.
			if (!$this->is_valid_user()) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_transfer_user_restricted_message')));
			}

			// Validate the withdraw amount is less than minimum amount.
			$minimum_amount = floatval(get_option('wal_module_wallet_withdrawal_min_amount'));
			if ($minimum_amount && $minimum_amount > $amount) {
				throw new exception(wp_kses_post(str_replace('{minimum_amount}', wal_convert_price_by_currency($minimum_amount), get_option('wal_module_wallet_withdrawal_minimum_amount_message'))));
			}

			// Validate the withdraw amount is greater than maximum amount.
			$maximum_amount = floatval(get_option('wal_module_wallet_withdrawal_max_amount'));
			if ($maximum_amount && $maximum_amount < $amount) {
				throw new exception(wp_kses_post(str_replace('{maximum_amount}', wal_convert_price_by_currency($maximum_amount), get_option('wal_module_wallet_withdrawal_maximum_amount_message'))));
			}
						
			// Validate the withdraw amount greater than user balance.
			if ((float) WAL_Current_User_Wallet::get_balance() < (float) ( $amount + $fee )) {
				throw new Exception(wp_kses_post(get_option('wal_module_wallet_withdrawal_insufficient_wallet_fund_message')));
			}

			//Validate the wallet withdraw count has reached the per day count.
			$max_count_per_day = floatval(get_option('wal_module_wallet_withdrawal_count_per_day'));
			if ($max_count_per_day && $max_count_per_day <= wal_get_user_wallet_withdrew_count()) {
				throw new exception(wp_kses_post(str_replace('{withdraw_count}', $max_count_per_day, get_option('wal_module_wallet_withdrawal_maximum_count_per_day_message'))));
			}
		}

		/**
		 * Validate the OTP.
		 */
		public function verify_otp() {
			// Return if the OTP is not enabled.
			if (!wal_wallet_withdrawal_otp_enabled()) {
				return;
			}

			$otp = isset($_REQUEST['wal_withdrawal_request_verify_otp']) ? wc_clean(wp_unslash($_REQUEST['wal_withdrawal_request_verify_otp'])) : '';
			if (!$otp) {
				throw new Exception(wp_kses_post(get_option('wal_module_wallet_withdrawal_invalid_otp_message')));
			}

			// Validate the OTP expiry.
			$validity = get_post_meta(WAL_Current_User_Wallet::get_user_id(), 'wal_wallet_withdrawal_otp_validity', true);
			if (WAL_Date_Time::get_date_time_object('now', true) >= WAL_Date_Time::get_date_time_object($validity)) {
				throw new Exception(wp_kses_post(get_option('wal_module_wallet_withdrawal_otp_expired_message')));
			}

			// Validate the entered OTP.
			$saved_otp = get_post_meta(WAL_Current_User_Wallet::get_user_id(), 'wal_wallet_withdrawal_otp', true);
			if ($otp != $saved_otp) {
				throw new Exception(wp_kses_post(get_option('wal_module_wallet_withdrawal_invalid_otp_message')));
			}
		}

		/**
		 * Is a valid user?.
		 * 
		 * @return bool
		 */
		public function is_valid_user() {
			$return = true;

			if ($this->validate_user()) {
				$return = false;
			} elseif ('wal_active' != WAL_Current_User_Wallet::get_status()) {
				$return = false;
			}
			/**
			 * This hook is used to validate the wallet withdrawal for user.
			 * 
			 * @since 1.8
			 */
			return apply_filters('wal_is_valid_wallet_withdrawal_user', $return);
		}

		/**
		 * Validate the user.
		 * 
		 * @return bool
		 */
		protected function validate_user() {
			$return = false;
			$user_type = get_option('wal_module_wallet_withdrawal_user_restriction_type');

			switch ($user_type) {
				case '2':
					$include_users = array_filter((array) get_option('wal_module_wallet_withdrawal_include_users'), 'wal_array_filter');
					if (!in_array(WAL_Current_User_Wallet::get_user_id(), $include_users)) {
						$return = true;
					}

					break;
				case '3':
					$exclude_users = array_filter((array) get_option('wal_module_wallet_withdrawal_exclude_users'), 'wal_array_filter');
					if (in_array(WAL_Current_User_Wallet::get_user_id(), $exclude_users)) {
						$return = true;
					}
					break;
				case '4':
					$return = true;
					$include_user_roles = array_filter((array) get_option('wal_module_wallet_withdrawal_include_user_roles'), 'wal_array_filter');

					// Loggedin user restriction
					if (wal_check_is_array(WAL_Current_User_Wallet::get_user()->roles)) {
						foreach (WAL_Current_User_Wallet::get_user()->roles as $role) {
							if (in_array($role, $include_user_roles)) {
								$return = false;
							}
						}
					}

					break;
				case '5':
					$exclude_user_roles = array_filter((array) get_option('wal_module_wallet_withdrawal_exclude_user_roles'), 'wal_array_filter');

					// Loggedin user restriction
					if (wal_check_is_array(WAL_Current_User_Wallet::get_user()->roles)) {
						foreach (WAL_Current_User_Wallet::get_user()->roles as $role) {
							if (in_array($role, $exclude_user_roles)) {
								$return = true;
							}
						}
					}

					break;
			}
			/**
			 * This hook is used to validate the wallet withdrawal for user.
			 * 
			 * @since 1.0
			 */
			return apply_filters('wal_validate_user_wallet_withdrawal', $return);
		}

		/**
		 * Get user withdrawal cancelled reason.
		 *
		 * @since 2.8.0
		 * 
		 * @return void
		 */
		public static function get_user_withdrawal_cancelled_reason() {
			check_ajax_referer('wal-wallet-withdrawal', 'wal_security');

			try {
				if (!isset($_REQUEST['withdrawal_id']) || empty($_REQUEST['withdrawal_id'])) {
					throw new exception(esc_html__('Invalid Request.', 'wallet-for-woocommerce'));
				}

				$wallet_withdrawal = wal_get_wallet_withdrawal(absint($_REQUEST['withdrawal_id']));
				if (!$wallet_withdrawal->exists()) {
					throw new exception(esc_html__('Invalid request', 'wallet-for-woocommerce'));
				}

				ob_start();
				wal_get_template('modules/wallet-withdrawal/cancelled-reason.php', array( 'wallet_withdrawal' => $wallet_withdrawal ));
				$html = ob_get_contents();
				ob_end_clean();

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Get wallet withdrawal cancel popup content.
		 *
		 * @since 2.8.0
		 * 
		 * @return void
		 */
		public static function get_wallet_withdrawal_cancel_popup_content() {
			check_ajax_referer('wal-wallet-withdrawal', 'wal_security');

			try {
				if (!isset($_REQUEST['withdrawal_id']) || empty($_REQUEST['withdrawal_id'])) {
					throw new exception(esc_html__('Invalid Request.', 'wallet-for-woocommerce'));
				}

				$wallet_withdrawal = wal_get_wallet_withdrawal(absint($_REQUEST['withdrawal_id']));
				if (!$wallet_withdrawal->exists()) {
					throw new exception(esc_html__('Invalid request', 'wallet-for-woocommerce'));
				}

				ob_start();
				// Render the wallet withdrawal cancel popup.
				include WAL_PLUGIN_PATH . '/inc/modules/views/html-wallet-withdrawal-cancel-popup.php';
				$html = ob_get_contents();
				ob_end_clean();

				wp_set_post_lock($wallet_withdrawal->get_post());

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Cancel the wallet withdrawal. 
		 * 
		 * @since 2.8.0
		 * @throws exception
		 */
		public static function cancel_wallet_withdrawal() {
			check_ajax_referer('wal-wallet-withdrawal', 'wal_security');

			try {
				if (!isset($_REQUEST['withdrawal_id']) || empty($_REQUEST['withdrawal_id'])) {
					throw new exception(esc_html__('Invalid Request.', 'wallet-for-woocommerce'));
				}

				$withdrawal = wal_get_wallet_withdrawal(absint($_REQUEST['withdrawal_id']));
				if (!$withdrawal->exists()) {
					throw new exception(esc_html__('Invalid request', 'wallet-for-woocommerce'));
				}

				$reason = isset($_REQUEST['reason']) ? sanitize_textarea_field($_REQUEST['reason']) : '';

				// Cancel the wallet withdrawal.
				wal_cancel_wallet_withdrawal($withdrawal->get_id(), $reason);

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Get the wallet withdrawal cancelled reason.
		 * 
		 * @since 2.8.0
		 * @throws exception
		 */
		public static function get_wallet_withdrawal_cancelled_reason() {
			check_ajax_referer('wal-wallet-withdrawal', 'wal_security');

			try {
				if (!isset($_REQUEST['withdrawal_id']) || empty($_REQUEST['withdrawal_id'])) {
					throw new exception(__('Invalid Request.', 'wallet-for-woocommerce'));
				}

				$withdrawal_id = absint($_REQUEST['withdrawal_id']);
				$wallet_withdraw = wal_get_wallet_withdrawal($withdrawal_id);
				if (!$wallet_withdraw->exists()) {
					throw new exception(esc_html__('Invalid request', 'wallet-for-woocommerce'));
				}

				ob_start();
				// Render the withdrawal cancelled reason.
				include_once WAL_PLUGIN_PATH . '/inc/modules/views/html-wallet-withdrawal-cancelled-reason.php';
				$html = ob_get_contents();
				ob_end_clean();

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Set wallet withdrawal edit post.
		 * 
		 * @since 2.8.0
		 * @throws exception
		 */
		public static function set_wallet_withdrawal_edit_post() {
			check_ajax_referer('wal-wallet-withdrawal', 'wal_security');

			try {
				if (!isset($_REQUEST['withdrawal_id']) || empty($_REQUEST['withdrawal_id'])) {
					throw new exception(__('Invalid Request.', 'wallet-for-woocommerce'));
				}

				$post_id = absint($_REQUEST['withdrawal_id']);

				wp_set_post_lock($post_id);

				wp_send_json_success(array( 'link' => get_edit_post_link($post_id) ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}
	}

}
