<?php

/**
 * Module - Wallet Actions.
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Wallet_Actions_Module')) {

	/**
	 * Class.
	 * */
	class WAL_Wallet_Actions_Module extends WAL_Modules {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id = 'wallet_actions';
			$this->title = __('Wallet Actions', 'wallet-for-woocommerce');
			$this->description = __('Users can collect the funds in their wallet by performing actions.', 'wallet-for-woocommerce');
			$this->tooltip = __('By enabling this checkbox, you can allow your users to collect the funds to their wallet by performing actions.', 'wallet-for-woocommerce');

			parent::__construct();
		}

		/**
		 * Get the setting tabs.
		 * 
		 * @return array
		 */
		public function setting_tabs() {
			$tabs = array(
				'actions' => __('Actions', 'wallet-for-woocommerce'),
				'messages' => __('Messages', 'wallet-for-woocommerce'),
			);
			/**
			 * This hook is used to alter the current module tabs.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->get_plugin_slug() . '_get_module_tabs_' . $this->get_id(), $tabs);
		}

		/**
		 * Get settings for actions section array.
		 * 
		 * @return array
		 */
		protected function actions_section_array() {
			$section_fields = array();

			// Signup reward actions section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Account Signup Reward Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_signup_reward_action_options',
			);
			$section_fields[] = array(
				'title' => __('Account Signup Reward', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'autoload' => false,
				'id' => $this->get_option_key('signup_reward_action_enabled'),
				'desc' => __('By enabling this checkbox, you can award funds to your users wallet for registering on your site.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Enter the Reward Value', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('signup_reward_amount'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_signup_reward_action_options',
			);
			// Signup reward actions section end.
			// Daily login reward action section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Daily Login Reward Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_daily_login_reward_action_options',
			);
			$section_fields[] = array(
				'title' => __('Daily Login Reward', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'autoload' => false,
				'id' => $this->get_option_key('daily_login_reward_action_enabled'),
				'desc' => __('By enabling this checkbox, you can award funds to your users wallet for login to the site.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Enter the Reward Value', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('daily_login_reward_amount'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_signup_reward_action_options',
			);
			// Daily login reward action section end.
			// Product review reward action section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Product Review Reward Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_product_review_reward_action_options',
			);
			$section_fields[] = array(
				'title' => __('Product Review Reward', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'autoload' => false,
				'id' => $this->get_option_key('product_review_reward_action_enabled'),
				'desc' => __('By enabling this checkbox, you can award funds to your users wallet for reviewing the products.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Enter the Reward Value', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('product_review_reward_amount'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_product_review_reward_action_options',
			);
			// Product review reward action section end.

			return $section_fields;
		}

		/**
		 * Get the settings for messages section array.
		 * 
		 * @return array
		 */
		protected function messages_section_array() {
			$section_fields = array();

			// Log section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Transaction Log', 'wallet-for-woocommerce'),
				'id' => 'wal_wallet_actions_log_options',
			);
			$section_fields[] = array(
				'title' => __('Account Signup Log', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds successfully credited for registering on the site',
				'autoload' => false,
				'id' => $this->get_option_key('signup_reward_log_message'),
			);
			$section_fields[] = array(
				'title' => __('Daily Login Log', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds successfully credited for Login to the Site on {login_date}',
				'autoload' => false,
				'id' => $this->get_option_key('daily_login_reward_log_message'),
			);
			$section_fields[] = array(
				'title' => __('Product Review Log', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds successfully credited for reviewing the product "{product_name}"',
				'autoload' => false,
				'id' => $this->get_option_key('product_review_reward_log_message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_wallet_actions_log_options',
			);
			// Log section end.

			return $section_fields;
		}

		/**
		 * Actions.
		 */
		protected function actions() {
			//Handle the Signup reward.
			add_action('user_register', array( $this, 'signup_reward' ));
			//Handle the daily login.
			add_action('wp_login', array( $this, 'daily_login_reward' ), 10, 2);
			//Reward the fund to the user for product review when the comment is approved.
			add_action('comment_post', array( $this, 'comment_approved_product_review_reward' ), 10, 3);
			//Reward the fund to the user for product review when the comment status changed to approved.
			add_action('transition_comment_status', array( $this, 'reward_product_review_comment_status_changed' ), 10, 3);
			//Add the email classes.
			add_filter('wal_wallet_email_classes', array( $this, 'email_classes' ), 10, 1);
		}

		/**
		 * Add the custom email classes.
		 * 
		 * @return array
		 */
		public function email_classes( $classes ) {
			$classes = ( wal_check_is_array($classes) ) ? $classes : array();

			$classes['customer-wallet-funds-account-signup'] = 'WAL_Customer_Wallet_Funds_Account_Signup_Notification';
			$classes['customer-user-daily-login-fund'] = 'WAL_Customer_User_Daily_Login_Fund_Notification';
			$classes['customer-user-product-review-fund'] = 'WAL_Customer_User_Product_Review_Fund_Notification';

			return $classes;
		}

		/**
		 * Credit the Signup reward to wallet.
		 */
		public function signup_reward( $user_id ) {
			// Return if the signup reward action is not enabled.
			if ('yes' != get_option('wal_module_signup_reward_action_enabled')) {
				return;
			}

			// Return if the signup reward amount is not configured.
			$amount = get_option('wal_module_signup_reward_amount');
			if (!$amount) {
				return;
			}

			// Return if the reward is already awarded.
			if (get_user_meta($user_id, 'wal_signup_reward_id', true)) {
				return;
			}

			$args = array(
				'user_id' => $user_id,
				'amount' => $amount,
				'event_id' => 15,
				'event_message' => get_option('wal_module_signup_reward_log_message'),
			);

			$transaction_log_id = wal_credit_wallet_fund($args);
			if ($transaction_log_id) {
				update_user_meta($user_id, 'wal_signup_reward_id', $transaction_log_id);
				/**
				 * This hook is used to do extra action after account signup fund credited.
				 * 
				 * @since 1.0
				 */
				do_action('wal_account_signup_fund_credited', $transaction_log_id, $user_id);
			}
		}

		/**
		 * Credit the daily login reward to wallet.
		 */
		public function daily_login_reward( $user_login, $user ) {
			if (!is_object($user)) {
				return;
			}

			// Return if the daily login reward action is not enabled.
			if ('yes' != get_option('wal_module_daily_login_reward_action_enabled')) {
				return;
			}

			// Return if the daily login reward amount is not configured.
			$amount = get_option('wal_module_daily_login_reward_amount');
			if (!$amount) {
				return;
			}

			$current_date = WAL_Date_Time::get_date_mysql_format('now');
			// Return if the reward is already awarded for today.
			if (get_user_meta($user->ID, 'wal_daily_login_reward_date', true) == $current_date) {
				return;
			}

			$args = array(
				'user_id' => $user->ID,
				'amount' => $amount,
				'event_id' => 16,
				'event_message' => str_replace('{login_date}', WAL_Date_Time::get_wp_format_datetime('now', 'date'), get_option('wal_module_daily_login_reward_log_message')),
			);

			$transaction_log_id = wal_credit_wallet_fund($args);
			if ($transaction_log_id) {
				update_user_meta($user->ID, 'wal_daily_login_reward_date', $current_date);
				/**
				 * This hook is used to do extra action after user daily login fund credited.
				 * 
				 * @since 1.0
				 */
				do_action('wal_user_daily_login_fund_credited', $transaction_log_id, $user);
			}
		}

		/**
		 * Reward the fund to the user for product review when the comment is approved.
		 */
		public function comment_approved_product_review_reward( $comment_ID, $comment_approved, $commentdata ) {
			// Return if the comment is not approved.
			if (!$comment_ID || '1' != $comment_approved) {
				return;
			}

			self::product_review_reward($comment_ID, $commentdata['comment_post_ID'], $commentdata['user_id']);
		}

		/**
		 * Reward the fund to the user for product review when the comment status changed to approved.
		 */
		public function reward_product_review_comment_status_changed( $new_status, $old_status, $comment ) {
			if (!is_object($comment) || 'approved' != $new_status) {
				return;
			}

			self::product_review_reward($comment->comment_ID, $comment->comment_post_ID, $comment->user_id);
		}

		/**
		 * Credit the product review reward to wallet.
		 */
		public function product_review_reward( $comment_id, $product_id, $user_id ) {

			// Return if the post is not a poduct.
			if ('product' != get_post_type($product_id)) {
				return;
			}

			// Return if the product review reward action is not enabled.
			if ('yes' != get_option('wal_module_product_review_reward_action_enabled')) {
				return;
			}

			// Return if the product review reward amount is not configured.
			$amount = get_option('wal_module_product_review_reward_amount');
			if (!$amount) {
				return;
			}

			// Return if the reward is already awarded for today.
			if (get_comment_meta($comment_id, 'wal_product_review_reward_id', true) || get_post_meta($product_id, 'wal_product_review_reward_id_' . $user_id, true)) {
				return;
			}

			$product = wc_get_product($product_id);

			$args = array(
				'user_id' => $user_id,
				'amount' => $amount,
				'event_id' => 17,
				'event_message' => str_replace('{product_name}', $product->get_name(), get_option('wal_module_product_review_reward_log_message')),
			);

			$transaction_log_id = wal_credit_wallet_fund($args);
			if ($transaction_log_id) {

				update_comment_meta($comment_id, 'wal_product_review_reward_id', $transaction_log_id);

				update_post_meta($product_id, 'wal_product_review_reward_id_' . $user_id, $transaction_log_id);
				/**
				 * This hook is used to do extra action after user product review fund credited.
				 * 
				 * @since 1.0
				 */
				do_action('wal_user_product_review_fund_credited', $transaction_log_id, $comment_id, $product);
			}
		}
	}

}
