<?php

/**
 * Module - Product Purchase.
 *
 * @since 3.1.0
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Product_Purchase_Module')) {

	/**
	 * Class.
	 *
	 * @since 3.1.0
	 * */
	class WAL_Product_Purchase_Module extends WAL_Modules {

		/**
		 * Class Constructor.
		 *
		 * @since 3.1.0
		 * */
		public function __construct() {

			$this->id = 'product_purchase';
			$this->title = __('Product Purchase', 'wallet-for-woocommerce');
			$this->description = __('Users can collect the funds to their wallet for purchasing the products.', 'wallet-for-woocommerce');
			$this->tooltip = __('By enabling this checkbox, users can collect the funds to their wallet for purchasing the products.', 'wallet-for-woocommerce');

			parent::__construct();
		}

		/**
		 * Get the setting tabs.
		 *
		 * @since 3.1.0
		 * @return array
		 */
		public function setting_tabs() {
			/**
			 * This hook is used to alter the current module tabs.
			 *
			 * @param string $id
			 * @param array $tabs
			 * @since 3.1.0
			 */
			return apply_filters($this->get_plugin_slug() . '_get_module_tabs_' . $this->get_id(),
					array(
						'product_purchase' => __('Product Purchase', 'wallet-for-woocommerce'),
						'localization' => __('Localization', 'wallet-for-woocommerce'),
						'messages' => __('Messages', 'wallet-for-woocommerce'),
			));
		}

		/**
		 * Get settings for product purchase section array.
		 *
		 * @since 3.1.0
		 * @return array
		 */
		protected function product_purchase_section_array() {
			$section_fields = array();

			// Product purchase section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Product Purchase Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_product_purchase_options',
			);
			$section_fields[] = array(
				'title' => __('Product Purchase Funds based on', 'wallet-for-woocommerce'),
				'type' => 'select',
				'default' => '1',
				'autoload' => false,
				'id' => $this->get_option_key('product_purchase_mode'),
				'class' => 'wal-product-purchase-fund-type',
				'options' => array(
					'1' => __('Product(s) / Product Category', 'wallet-for-woocommerce'),
					'4' => __('Category Total Range', 'wallet-for-woocommerce'),
					'2' => __('Order Total', 'wallet-for-woocommerce'),
					'3' => __('Order Total Range', 'wallet-for-woocommerce'),
				),
			);
			$section_fields[] = array(
				'title' => __('Fund Type', 'wallet-for-woocommerce'),
				'type' => 'select',
				'default' => '1',
				'autoload' => false,
				'id' => $this->get_option_key('product_purchase_fund_type'),
				'class' => 'wal-product-purchase-fund-type',
				'options' => array(
					'1' => __('Percentage of Product Price', 'wallet-for-woocommerce'),
					'2' => __('Fixed Value', 'wallet-for-woocommerce'),
				),
			);
			$section_fields[] = array(
				'title' => __('Fund Type', 'wallet-for-woocommerce'),
				'type' => 'select',
				'default' => '1',
				'autoload' => false,
				'id' => $this->get_option_key('product_purchase_order_fund_type'),
				'class' => 'wal-product-purchase-fund-type',
				'options' => array(
					'1' => __('Percentage of Order Total', 'wallet-for-woocommerce'),
					'2' => __('Fixed Value', 'wallet-for-woocommerce'),
				),
			);
			$section_fields[] = array(
				'title' => __('Enter the Value', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-product-purchase-field wal-product-purchase-fund',
				'id' => $this->get_option_key('product_purchase_amount'),
			);
			$section_fields[] = array(
				'title' => __('Minimum Order Total', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-product-purchase-field wal-order-total-fund',
				'id' => $this->get_option_key('product_purchase_order_total_minimum_amount'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Order Total', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-product-purchase-field wal-order-total-fund',
				'id' => $this->get_option_key('product_purchase_order_total_maximum_amount'),
			);
			$section_fields[] = array(
				'title' => __('Rule Priority', 'wallet-for-woocommerce'),
				'type' => 'select',
				'default' => '1',
				'autoload' => false,
				'id' => $this->get_option_key('product_purchase_order_total_range_priority'),
				'class' => 'wal-product-purchase-range',
				'options' => array(
					'1' => __('First Matched Rule', 'wallet-for-woocommerce'),
					'2' => __('Last Matched Rule', 'wallet-for-woocommerce'),
					'3' => __('Minimum Value Matched Rule', 'wallet-for-woocommerce'),
					'4' => __('Maximum Value Matched Rule', 'wallet-for-woocommerce'),
				),
			);
			$section_fields[] = array(
				'title' => __('Rule Priority', 'wallet-for-woocommerce'),
				'type' => 'select',
				'default' => '1',
				'autoload' => false,
				'id' => $this->get_option_key('category_total_range_priority'),
				'class' => 'wal-product-purchase-range',
				'options' => array(
					'1' => __('First Matched Rule', 'wallet-for-woocommerce'),
					'2' => __('Last Matched Rule', 'wallet-for-woocommerce'),
					'3' => __('Minimum Value Matched Rule', 'wallet-for-woocommerce'),
					'4' => __('Maximum Value Matched Rule', 'wallet-for-woocommerce'),
				),
			);
			$section_fields[] = array(
				'type' => 'wal_custom_fields',
				'wal_field' => 'product_purchase_range',
				'id' => $this->get_option_key('product_purchase_order_total_range_rules'),
				'default' => array(),
				'class' => 'wal-product-purchase-range',
			);
			$section_fields[] = array(
				'type' => 'wal_custom_fields',
				'wal_field' => 'category_total_range',
				'id' => $this->get_option_key('category_total_range_rules'),
				'default' => array(),
				'class' => 'wal-category-total-range',
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_product_purchase_options',
			);
			// Product purchase section end
			// Order status section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Order Status Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_product_purchase_order_status_options',
			);
			$section_fields[] = array(
				'title' => __('Funds will be Added to Wallet when the Order Status Reaches', 'wallet-for-woocommerce'),
				'type' => 'multiselect',
				'default' => array( 'completed' ),
				'autoload' => false,
				'class' => 'wal_select2',
				'options' => wal_get_wc_order_statuses(false),
				'id' => $this->get_option_key('product_purchase_paid_order_statuses'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_product_purchase_order_status_options',
			);
			// Order status section end.
			// Restriction section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Restriction Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_product_purchase_restriction_options',
			);
			$section_fields[] = array(
				'title' => __('Shipping Method Selection', 'wallet-for-woocommerce'),
				'type' => 'multiselect',
				'default' => array(),
				'autoload' => false,
				'class' => 'wal_select2',
				'options' => wal_get_shipping_methods(),
				'id' => $this->get_option_key('product_purchase_allowed_shipping_method'),
				'desc_tip' => true,
				'desc' => __('If any shipping method is selected, then product purchase funds will be awarded to users only if they choose any of the shipping method(s) selected in this option & make a purchase. Leave empty if you dont want to restrict based on shipping method.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_product_purchase_restriction_options',
			);
			// Restriction section end.

			return $section_fields;
		}

		/**
		 * Get the settings for localization section array.
		 *
		 * @since 3.1.0
		 * @return array
		 */
		protected function localization_section_array() {
			$section_fields = array();

			// Product Level Transaction Log section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Product Level Transaction Log', 'wallet-for-woocommerce'),
				'id' => 'wal_product_purchase_transaction_log_options',
			);
			$section_fields[] = array(
				'title' => __('Product Purchase - Product/Category[Earned]', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds credited for purchasing the product {product_name} in order {order_id}',
				'autoload' => false,
				'id' => $this->get_option_key('purchased_product_fund_credit_log'),
			);
			$section_fields[] = array(
				'title' => __('Product Purchase - Product/Category[Revised]', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds revised for purchasing the product {product_name} in order {order_id}',
				'autoload' => false,
				'id' => $this->get_option_key('purchased_product_fund_debit_log'),
			);
			$section_fields[] = array(
				'title' => __('Product Purchase - Order Total[Earned] ', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds credited for completing the order {order_id}',
				'autoload' => false,
				'id' => $this->get_option_key('purchased_order_fund_credit_log'),
			);
			$section_fields[] = array(
				'title' => __('Product Purchase - Order Total[Revised] ', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds revised towards the order {order_id}',
				'autoload' => false,
				'id' => $this->get_option_key('purchased_order_fund_debit_log'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_product_purchase_transaction_log_options',
			);
			// Product Level Transaction Log section end.
			
			return $section_fields;
		}

		/**
		 * Get the settings for messages section array.
		 *
		 * @since 3.1.0
		 * @return array
		 */
		protected function messages_section_array() {
			$section_fields = array();

			// Product Purchase Message section start - Product.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Product Purchase - Product/Category', 'wallet-for-woocommerce'),
				'id' => 'wal_purchase_product_message_options',
			);
			$section_fields[] = array(
				'title' => __('Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Get {product_purchase_funds} funds by purchasing {product_name}.',
				'autoload' => false,
				'id' => $this->get_option_key('product_purchase_fund_message'),
				'desc_tip' => true,
				'desc' => __('This message will be displayed in Product Page, Cart Page & Checkout Page.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Shop & Category Page Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Earn {product_funds} Funds',
				'autoload' => false,
				'id' => $this->get_option_key('earn_fund_message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_purchase_product_message_options',
			);
			// Product Purchase Message section end - Product.
			// Product Purchase Message section start - Order.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Product Purchase - Order Total', 'wallet-for-woocommerce'),
				'id' => 'wal_purchase_order_message_options',
			);
			$section_fields[] = array(
				'title' => __('Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Get {order_funds} funds by completing this order.',
				'autoload' => false,
				'id' => $this->get_option_key('product_purchase_order_fund_message'),
				'desc_tip' => true,
				'desc' => __('This message will be displayed in Cart Page & Checkout Page', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Minimum Order Total Error Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Minimum Order Total of {minimum_fund_value} required to earn funds in this order',
				'autoload' => false,
				'id' => $this->get_option_key('product_purchase_required_minimum_fund_message'),
				'desc_tip' => true,
				'desc' => __('This message will be displayed in Cart Page & Checkout Page', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Order Total Error Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'You are eligible to earn funds only when the order total is below {maximum_fund_value} ',
				'autoload' => false,
				'id' => $this->get_option_key('product_purchase_required_maximum_fund_message'),
				'desc_tip' => true,
				'desc' => __('This message will be displayed in Cart Page & Checkout Page', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_purchase_order_message_options',
			);
			// Product Purchase Message section end - Order.
			// Product Purchase Message section start - Order.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Product Purchase - Category Total Range', 'wallet-for-woocommerce'),
				'id' => 'wal_purchase_order_message_options',
			);
			$section_fields[] = array(
				'title' => __('Shop and Category Page Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Get {category_funds_value} funds for purchasing products from {category_name}. Category Total Range: {min_value} to {max_value}.',
				'autoload' => false,
				'id' => $this->get_option_key('category_total_range_min_to_max_message_in_shop_and_category'),
				'desc_tip' => true,
			);
			$section_fields[] = array(
				'title' => __('Shop and Category Page Message - Min Value', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Get {category_funds_value} funds for purchasing products from {category_name}. Category Total Range: minimum {min_value} and above.',
				'autoload' => false,
				'id' => $this->get_option_key('category_total_range_min_message_in_shop_and_category'),
				'desc_tip' => true,
			);
			$section_fields[] = array(
				'title' => __('Shop and Category Page Message - Max Value', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Get {category_funds_value} funds for purchasing products from {category_name}. Category Total Range: maximum of {max_value}.',
				'autoload' => false,
				'id' => $this->get_option_key('category_total_range_max_message_in_shop_and_category'),
				'desc_tip' => true,
			);
			$section_fields[] = array(
				'title' => __('Cart and Checkout Page', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'You will get {category_funds_value} for purchasing products from {category_name}.',
				'autoload' => false,
				'id' => $this->get_option_key('category_total_range_message_in_cart_and_checkout'),
				'desc_tip' => true,
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_purchase_order_message_options',
			);
			// Product Purchase Message section end - Order.
			// Product Purchase Message section start - Shipping method restriction.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Product Purchase - Shipping Method Restriction', 'wallet-for-woocommerce'),
				'id' => 'wal_purchase_shipping_method_restriction_message_options',
			);
			$section_fields[] = array(
				'title' => __('Message to Display in Cart & Checkout', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'You can get funds for this purchase only if you choose any of the {shipping method(s)}',
				'autoload' => false,
				'id' => $this->get_option_key('product_purchase_valid_shipping_method_message'),
			);
			$section_fields[] = array(
				'title' => __('Popup Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Shipping Method List',
				'autoload' => false,
				'id' => $this->get_option_key('product_purchase_shipping_method_lists_label'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_purchase_shipping_method_restriction_message_options',
			);
			// Product Purchase Message section end - Shipping method restriction.

			return $section_fields;
		}

		/**
		 * Include files.
		 *
		 * @since 3.1.0
		 */
		protected function include_files() {
			include_once WAL_ABSPATH . 'inc/class-wal-purchased-order-fund-handler.php';
			include_once WAL_ABSPATH . 'inc/class-wal-purchased-product-fund-handler.php';
			include_once WAL_ABSPATH . 'inc/class-wal-order-total-range-fund-handler.php';
			include_once WAL_ABSPATH . 'inc/class-wal-category-total-range-fund-handler.php';
		}

		/**
		 * Actions.
		 *
		 * @since 3.1.0
		 */
		protected function actions() {
			// Maybe create new order product purchase commission
			add_action('woocommerce_new_order', array( $this, 'maybe_create_product_purchase_commission' ), 10, 1);
			//Maybe update product purchase order meta via shortcode checkout.
			add_action('woocommerce_checkout_update_order_meta', array( $this, 'maybe_update_product_purchase_order_meta' ), 10, 1);
			//Maybe update product purchase order meta via block checkout.
			add_action('woocommerce_store_api_checkout_order_processed', array( $this, 'maybe_update_product_purchase_order_meta' ), 10, 1);
			//Add the email classes.
			add_filter('wal_wallet_email_classes', array( $this, 'email_classes' ), 10, 1);
			// May be credit the product purchase fund in wallet based on order statuses.
			$product_purchase_order_statuses = get_option('wal_module_product_purchase_paid_order_statuses');
			if (wal_check_is_array($product_purchase_order_statuses)) {
				foreach ($product_purchase_order_statuses as $order_status) {
					add_action("woocommerce_order_status_{$order_status}", array( $this, 'maybe_credit_product_purchase_fund_form_order' ), 1);
				}
			}

			/**
			 * This hook is used to alter the product purchase refund hooks.
			 *
			 * @param array $product_purchase_refund_hooks            
			 * @since 3.1.0
			 */
			$product_purchase_refund_hooks = apply_filters('wal_product_purchase_refund_hooks', array(
				'woocommerce_order_status_refunded',
				'woocommerce_order_status_cancelled',
				'woocommerce_order_status_failed',
					)
			);

			if (wal_check_is_array($product_purchase_refund_hooks)) {
				foreach ($product_purchase_refund_hooks as $hook) {
					// May be debit the product purchase fund in wallet based on order statuses.
					add_action($hook, array( $this, 'maybe_debit_product_purchase_fund_from_order' ), 1);
				}
			}
		}

		/**
		 * Admin actions.
		 * 
		 * @since 3.1.0
		 */
		protected function admin_action() {
			if ('1' !== get_option('wal_module_product_purchase_mode')) {
				return;
			}
			// Render the wallet simple product setting in general panel.
			add_action('woocommerce_product_options_general_product_data', array( $this, 'render_simple_product_settings' ));
			// Render the wallet variation product setting in variation panel.
			add_action('woocommerce_product_after_variable_attributes', array( $this, 'render_variation_product_settings' ), 10, 3);
			// Save the simple product settings.
			add_action('woocommerce_process_product_meta', array( $this, 'save_simple_product_settings' ), 10, 2);
			// Save the variation product Settings.
			add_action('woocommerce_save_product_variation', array( $this, 'save_variation_product_settings' ), 10, 2);
			// Render new category settings 
			add_action('product_cat_add_form_fields', array( $this, 'render_new_category_settings' ));
			// Render edit category settings 
			add_action('product_cat_edit_form_fields', array( $this, 'render_edit_category_settings' ), 10, 2);
			// Save add category settings
			add_action('created_term', array( $this, 'save_new_category_settings' ), 10, 3);
			// Save edit category settings
			add_action('edit_term', array( $this, 'save_new_category_settings' ), 10, 3);
		}

		/**
		 * Front end actions.
		 * 
		 * @since 3.1.0
		 */
		protected function frontend_action() {
			// Add notice in the shop page.
			add_action( 'woocommerce_shop_loop_header', array( $this, 'render_product_awarding_fund_message_in_shop_page' ) );
			// Render product awarding fund message in single product page
			add_action('woocommerce_before_single_product', array( $this, 'render_product_awarding_fund_message_in_product_page' ), 4);
			// Render product awarding fund message in cart page
			add_action('woocommerce_before_cart', array( $this, 'render_product_awarding_fund_message_in_cart_page' ), 5);
			add_action('woocommerce_before_checkout_form', array( $this, 'render_product_awarding_fund_message_in_cart_page' ), 5);
			// Render product awarding fund message in variable product page
			add_action('woocommerce_available_variation', array( $this, 'render_product_awarding_fund_message_in_variable_product' ), 10, 3);
			// Render variation notice in single product page.
			add_action('woocommerce_before_single_product_summary', array( $this, 'render_variation_notice_single_product_page' ));
			// Render earn fund notice in loop page.
			add_action('woocommerce_shop_loop_item_title', array( $this, 'render_loop_earn_funds_notice' ), 10);
			// Register store notices for cart/checkout blocks.
			add_filter('wal_store_api_cart_notices', array( $this, 'register_store_api_cart_notices' ));
		}

		/**
		 * Render loop earn funds notice 
		 *
		 * @since 3.3.0
		 * @global object $product
		 * @return string.
		 */
		public function render_loop_earn_funds_notice() {
			global $product;
			//Return if not product object 
			if (!is_object($product) || 'simple' != $product->get_type()) {
				return;
			}

			//Return if not product/product category mode
			if ('2' == get_option('wal_module_product_purchase_mode')) {
				return;
			}

			$fund = WAL_Purchased_Product_Fund_Handler::get_fund($product->get_id(), 0, 1, $product->get_price());
			if (!$fund) {
				return;
			}

			wal_get_template('loop/earn-fund-message.php', array( 'fund' => $fund ));
		}

		/**
		 * Enqueue admin end JS files.
		 * 
		 * @since 3.1.0
		 */
		protected function admin_external_js_files() {
			// Product purchase.   
			wp_enqueue_script('wal-product-purchase', WAL_PLUGIN_URL . '/assets/js/admin/product-purchase.js', array( 'jquery', 'jquery-blockui' ), WAL_VERSION);
		}

		/**
		 * Enqueue front end JS files.
		 *
		 * @since 3.1.0
		 */
		protected function frontend_external_js_files() {
			// Frontend.
			wp_enqueue_script('wal-product-purchase', WAL_PLUGIN_URL . '/assets/js/frontend/product-purchase.js', array( 'jquery', 'jquery-blockui' ), WAL_VERSION, WAL_Fronend_Assets::$in_footer);

			wp_localize_script(
					'wal-product-purchase', 'wal_product_purchase_frontend_params', array(
				'wal_product_variations_fund_nonce' => wp_create_nonce('wal-product-variations-fund-nonce'),
				'ajaxurl' => WAL_ADMIN_AJAX_URL,
					)
			);
		}

		/**
		 * Maybe create product purchase commission
		 * 
		 * @since 3.3.0
		 * @param int $order_id
		 * @return void
		 */
		public function maybe_create_product_purchase_commission( $order_id ) {
			$order = wc_get_order($order_id);
			// Return if not order object
			if (!is_object($order)) {
				return;
			}

			$this->maybe_update_product_purchase_order_meta($order_id);
		}

		/**
		 * Maybe update product purchase order meta
		 *
		 * @param int $order_id
		 * @since 3.1.0         
		 */
		public function maybe_update_product_purchase_order_meta( $order_id ) {
			$order_id = is_object($order_id) ? $order_id->get_id() : $order_id;
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}

			$bool = false;
			switch (get_option('wal_module_product_purchase_mode')) {
				case '2':
					$purchased_order_fund = WAL_Purchased_Order_Fund_Handler::get_fund($order->get_total());
					if ($purchased_order_fund) {
						$order->update_meta_data('wal_purchased_product_fund', $purchased_order_fund);
						$bool = true;
					}
					break;
				case '3':
					$purchased_order_range_fund = WAL_Order_Total_Range_Fund_Handler::get_fund($order->get_total());
					if ($purchased_order_range_fund) {
						$order->update_meta_data('wal_purchased_product_fund', $purchased_order_range_fund);
						$bool = true;
					}
					break;
				case '4':
					$category_total_range_fund = WAL_Category_Total_Range_Fund_Handler::get_fund($order->get_items());
					if ($category_total_range_fund) {
						$matched_key = WAL_Category_Total_Range_Fund_Handler::get_matched_rule_key();
						$previous_key = get_user_meta($order->get_user_id(), 'wal_awarded_rule_ids', true);
						$merged_keys = wal_check_is_array( $previous_key ) ? array_merge( $previous_key, array( $matched_key )) : array( $matched_key );
						update_user_meta($order->get_user_id(), 'wal_awarded_rule_ids', $merged_keys);
						$order->update_meta_data('wal_purchased_product_fund', $category_total_range_fund);
						$bool = true;
					}
					break;
				default:
					$purchased_product_details = $this->prepare_product_purchase_fund_from_order($order);
					if (wal_check_is_array($purchased_product_details)) {
						$order->update_meta_data('wal_product_purchase_details', $purchased_product_details);
						$order->update_meta_data('wal_purchased_product_fund', array_sum($purchased_product_details));
						$bool = true;
					}
					break;
			}

			if ($bool) {
				$order->update_meta_data('wal_product_purchase_fund_type', get_option('wal_module_product_purchase_mode'));
			}

			$order->save();
		}

		/**
		 * Prepare product purchase fund from order
		 *    
		 * @param object $order  
		 * @since 3.1.0
		 * @return array
		 */
		public function prepare_product_purchase_fund_from_order( $order ) {
			// Return not if an order object
			if (!is_object($order)) {
				return array();
			}

			//Return not if an order array
			if (!wal_check_is_array($order->get_items())) {
				return array();
			}

			foreach ($order->get_items() as $item) {
				$product_id = !empty($item['variation_id']) ? $item['variation_id'] : $item['product_id'];
				$product = wc_get_product($product_id);
				if (!is_object($product)) {
					continue;
				}

				if ('incl' === get_option('woocommerce_tax_display_cart')) {
					$regular_price = floatval($item['line_subtotal_tax']) + floatval($item['line_subtotal']);
				} else {
					$regular_price = floatval($item['line_subtotal']);
				}

				$fund[$product_id] = WAL_Purchased_Product_Fund_Handler::get_fund($item['product_id'], $item['variation_id'], $item['qty'], $regular_price);
			}

			return wal_check_is_array($fund) ? $fund : array();
		}

		/**
		 * Render simple product settings.
		 *
		 * @since 3.1.0
		 * @global object $post
		 * @global int $thepostid
		 * @global object $product_object
		 */
		public function render_simple_product_settings() {
			global $post, $thepostid, $product_object;
			include_once WAL_ABSPATH . 'inc/admin/menu/views/html-simple-product-settings.php';
		}

		/**
		 * Render variation product settings
		 * 
		 * @param int $loop
		 * @param array $variation_data
		 * @param object $variations
		 * 
		 * @since 3.1.0
		 * @return void
		 */
		public function render_variation_product_settings( $loop, $variation_data, $variations ) {
			// Return if not variation object
			if (!is_object($variations)) {
				return;
			}

			$variation_id = isset($variations->ID) ? absint($variations->ID) : 0;
			$variation_object = wc_get_product($variation_id);
			if (!is_object($variation_object) || 'variation' != $variation_object->get_type()) {
				return;
			}

			include WAL_ABSPATH . 'inc/admin/menu/views/html-product-variation-settings.php';
		}

		/**
		 * Render new category settings
		 *
		 * @since 3.1.0
		 */
		public function render_new_category_settings() {
			include_once WAL_ABSPATH . 'inc/admin/menu/views/html-add-category-level-settings.php';
		}

		/**
		 * Render edit category settings
		 * 
		 * @param object $term 
		 * @param object $taxonomy
		 * @since 3.1.0
		 */
		public function render_edit_category_settings( $term, $taxonomy ) {
			include_once WAL_ABSPATH . 'inc/admin/menu/views/html-edit-category-level-settings.php';
		}

		/**
		 * Save new category settings
		 * 
		 * @param int $term_id
		 * @param int $taxonamy_id
		 * @param object $taxonomy
		 * 
		 * @since 3.1.0
		 */
		public function save_new_category_settings( $term_id, $taxonamy_id, $taxonomy ) {
			$meta_data = array(
				'_wal_fund_type' => isset($_REQUEST['_wal_fund_type']) ? wc_clean(wp_unslash($_REQUEST['_wal_fund_type'])) : '',
				'_wal_fund_value' => isset($_REQUEST['_wal_fund_value']) ? wc_clean(wp_unslash($_REQUEST['_wal_fund_value'])) : '',
				'_wal_block_fund' => isset($_REQUEST['_wal_block_fund']) ? wc_clean(wp_unslash($_REQUEST['_wal_block_fund'])) : 'no',
			);

			foreach ($meta_data as $key => $value) {
				update_term_meta($term_id, $key, $value);
			}

			/**
			 * This hook is used to do extra action after category product setting saved.
			 *
			 * @param int $term_id
			 * @param int $taxonamy_id
			 * @param object $taxonomy
			 *
			 * @since 3.1.0
			 */
			do_action('wal_category_product_settings_saved', $term_id, $taxonamy_id, $taxonomy);
		}

		/**
		 * Save simple product settings
		 *
		 * @param int $post_id
		 * @param object $post
		 * 
		 * @since 3.1.0
		 */
		public function save_simple_product_settings( $post_id, $post ) {
			$meta_data = array(
				'_wal_fund_type' => isset($_REQUEST['_wal_fund_type']) ? wc_clean(wp_unslash($_REQUEST['_wal_fund_type'])) : '1',
				'_wal_fund_value' => isset($_REQUEST['_wal_fund_value']) ? wc_clean(wp_unslash($_REQUEST['_wal_fund_value'])) : '',
				'_wal_block_fund' => isset($_REQUEST['_wal_block_fund']) ? wc_clean(wp_unslash($_REQUEST['_wal_block_fund'])) : 'no',
			);

			foreach ($meta_data as $key => $value) {
				update_post_meta($post_id, $key, $value);
			}

			/**
			 * This hook is used to do extra action after simple product setting saved.
			 *
			 * @param int $post_id
			 * @param object $post
			 *
			 * @since 3.1.0
			 */
			do_action('wal_simple_product_settings_saved', $post_id, $post);
		}

		/**
		 * Save variation product settings
		 *
		 * @param int $variation_id
		 * @param int $i
		 * 
		 * @since 3.1.0
		 */
		public function save_variation_product_settings( $variation_id, $i ) {
			$meta_data = array(
				'_wal_fund_type' => isset($_REQUEST['_wal_fund_type'][$i]) ? wc_clean(wp_unslash($_REQUEST['_wal_fund_type'][$i])) : '1',
				'_wal_fund_value' => isset($_REQUEST['_wal_fund_value'][$i]) ? wc_clean(wp_unslash($_REQUEST['_wal_fund_value'][$i])) : '',
				'_wal_block_fund' => isset($_REQUEST['_wal_block_fund'][$i]) ? wc_clean(wp_unslash($_REQUEST['_wal_block_fund'][$i])) : 'no',
			);

			foreach ($meta_data as $key => $value) {
				update_post_meta($variation_id, $key, $value);
			}

			/**
			 * This hook is used to do extra action after variation product settings saved.
			 *
			 * @param int $variation_id
			 * @param int $i
			 *
			 * @since 3.1.0
			 */
			do_action('wal_variation_product_settings_saved', $variation_id, $i);
		}

		/**
		 * Render purchased product fund message product page.
		 * 
		 * @since 3.1.0
		 * @return bool/error message
		 */
		public function render_product_awarding_fund_message_in_product_page() {
			//Return not single product page.
			if (!is_single()) {
				return;
			}

			//Return if product purchase order total type
			if ('1' !== get_option('wal_module_product_purchase_mode')) {
				return;
			}

			global $post;
			// Return if current post not object 
			if (!is_object($post) || !isset($post->ID)) {
				return;
			}

			$product = wc_get_product($post->ID);
			// Return if not product object 
			if (!is_object($product)) {
				return;
			}

			//Return if the current product type is variable
			if ('variable' === $product->get_type()) {
				return;
			}

			$fund = WAL_Purchased_Product_Fund_Handler::get_fund($product->get_id(), 0, 1, $product->get_price());
			// Return if fund is empty
			if (empty($fund)) {
				return;
			}

			$message = wal_prepare_product_awarding_fund_message($product->get_id(), $fund);
			wal_add_wc_notice($message, 'notice');
		}

		/**
		 * Render product awarding fund message in cart page.
		 * 
		 * @since 3.1.0
		 * @return bool/error message
		 */
		public function render_product_awarding_fund_message_in_cart_page() {
			if (is_single()) {
				return;
			}

			$messages = self::get_cart_product_fund_awarding_messages();
			if (!wal_check_is_array($messages)) {
				return;
			}

			wal_add_wc_notice(implode('< /br>', $messages), 'notice');
		}

		/**
		 * Render product awarding fund message in shop/category page.
		 * 
		 * @since 3.1.0
		 * @return bool/error message
		 */
		public function render_product_awarding_fund_message_in_shop_page() {
			$messages = self::get_category_fund_awarding_messages();
			if (!wal_check_is_array($messages)) {
				return;
			}

			wal_get_template( 'info-notice.php', array( 'notice_msg' => implode('< /br>', $messages), 'class_name' => '' ) );
		}

		/**
		 * Get the cart product fund awarding messages.
		 * 
		 * @since 3.7.0
		 * @return array
		 */
		public function get_category_fund_awarding_messages() {
			$messages = array();
			if ( '4' != get_option('wal_module_product_purchase_mode')) {
				return $messages;
			}
			$category_total_range_rules = get_option('wal_module_category_total_range_rules', array());
			//Return if purchase category range rules empty
			if (!wal_check_is_array($category_total_range_rules)) {
				return $messages;
			}

			$min_max_message = get_option('wal_module_category_total_range_min_to_max_message_in_shop_and_category');
			$min_message = get_option('wal_module_category_total_range_min_message_in_shop_and_category');
			$max_message = get_option('wal_module_category_total_range_max_message_in_shop_and_category');
			foreach ($category_total_range_rules as $key => $category_total_range_rule) {
				$values_to_find = array( '{category_funds_value}', '{category_name}', '{min_value}', '{max_value}' );
				$values_to_replace = array( wal_price($category_total_range_rule['fund_value']), '<b>"' . $this->get_category_name( $category_total_range_rule['included_category'] ) . '"</b>', wal_price($category_total_range_rule['min_category_total']), wal_price($category_total_range_rule['max_category_total']) );
				if ( !empty($category_total_range_rule['min_category_total']) && !empty($category_total_range_rule['max_category_total'])) {
					$message_to_replace = $min_max_message;
				} else if ( !empty($category_total_range_rule['min_category_total']) && empty($category_total_range_rule['max_category_total'])) {
					$message_to_replace = $min_message;
				} else if ( empty($category_total_range_rule['min_category_total']) && !empty($category_total_range_rule['max_category_total'])) {
					$message_to_replace = $max_message;
				}

				$messages[] = str_replace( $values_to_find, $values_to_replace, $message_to_replace );
			}

			return $messages;
		}

		/**
		 * Get the cart product fund awarding messages.
		 * 
		 * @since 3.7.0
		 * @return array
		 */
		public function get_category_name( $category_ids ) {
			$category_name = array();
			if ( !wal_check_is_array($category_ids) ) {
				return $category_name;
			}

			foreach ($category_ids as $category_id) {
				$category_name[] = wal_get_category_name($category_id);
			}

			return implode(',', $category_name);
		}

		/**
		 * Get the cart product fund awarding messages.
		 * 
		 * @since 3.7.0
		 * @return array
		 */
		public function get_cart_product_fund_awarding_messages() {
			// Return if a cart object is not initialized.
			if (!is_object(WC()->cart)) {
				return array();
			}

			$cart_items = WC()->cart->get_cart();
			if (!wal_check_is_array($cart_items)) {
				return array();
			}

			$messages = array();
			switch (get_option('wal_module_product_purchase_mode')) {
				case '3':
					$fund = WAL_Order_Total_Range_Fund_Handler::get_fund(WC()->cart->total);
					if (!empty($fund)) {
						$messages[] = wal_prepare_order_awarding_fund_message($fund);
						if (wal_check_is_array( wal_get_valid_shipping_methods())) {
							$messages[] = wal_valid_shipping_method_lists_message();
						}
					}
					break;
				case '4':
					$fund = WAL_Category_Total_Range_Fund_Handler::get_fund(WC()->cart->get_cart());
					$category_names = WAL_Category_Total_Range_Fund_Handler::get_category_name();
					if (!empty($fund)) {
						$messages[] = wal_prepare_category_awarding_fund_message($fund, $category_names);
						if (wal_check_is_array( wal_get_valid_shipping_methods())) {
							$messages[] = wal_valid_shipping_method_lists_message();
						}
					}
					break;
				case '2':
					$fund = WAL_Purchased_Order_Fund_Handler::get_fund(WC()->cart->total);
					if ($this->wal_validate_order_awarding_fund(WC()->cart->total)) {
						$messages[] = wal_prepare_order_awarding_fund_message($fund);
						if (wal_check_is_array( wal_get_valid_shipping_methods())) {
							$messages[] = wal_valid_shipping_method_lists_message();
						}
					}
					break;
				default:
					foreach ($cart_items as $value) {
						//Return if not is product object
						if (!is_object($value['data'])) {
							continue;
						}

						$product_id = !empty($value['variation_id']) ? $value['variation_id'] : $value['product_id'];
						$regular_price = isset($value['line_subtotal']) ? $value['line_subtotal'] : 0;
						if ('incl' === get_option('woocommerce_tax_display_cart')) {
							$line_subtotal_tax = isset($value['line_subtotal_tax']) ? $value['line_subtotal_tax'] : 0;
							$regular_price = $regular_price + $line_subtotal_tax;
						}

						$fund = WAL_Purchased_Product_Fund_Handler::get_fund($value['product_id'], $value['variation_id'], $value['quantity'], $regular_price);
						if ($fund) {
							$messages[] = wal_prepare_product_awarding_fund_message($product_id, $fund);
							if (wal_check_is_array( wal_get_valid_shipping_methods())) {
								$messages[] = wal_valid_shipping_method_lists_message();
							}
						}
					}
					break;
			}

			return $messages;
		}

		/**
		 * Prepare order awarding fund validation message.
		 *
		 * @param float $order_total
		 * @since 3.1.0
		 * 
		 * @return string.
		 * */
		public function wal_validate_order_awarding_fund( $order_total ) {
			$message = '';
			$minimum_order_total = get_option('wal_module_product_purchase_order_total_minimum_amount');
			$maximum_order_total = get_option('wal_module_product_purchase_order_total_maximum_amount');

			if (!empty($minimum_order_total) && $minimum_order_total > $order_total) {
				$message = str_replace('{minimum_fund_value}', wal_convert_price_by_currency($minimum_order_total), get_option('wal_module_product_purchase_required_minimum_fund_message'));
				wal_add_wc_notice($message, 'error');
				return false;
			}

			if (!empty($maximum_order_total) && $maximum_order_total < $order_total) {
				$message = str_replace('{maximum_fund_value}', wal_convert_price_by_currency($maximum_order_total), get_option('wal_module_product_purchase_required_maximum_fund_message'));
				wal_add_wc_notice($message, 'error');
				return false;
			}

			/**
			 * This hook is used to alter the wallet prepare order awarding fund validation message.
			 *           
			 * @since 3.1.0
			 */
			return apply_filters('wal_validate_order_awarding_fund', true);
		}

		/**
		 * Render variation notice for single product page
		 *
		 * @global object $product
		 * @since 3.1.0
		 */
		public function render_variation_notice_single_product_page() {
			global $product;
			// Return if not a variable product
			if ('variable' !== $product->get_type()) {
				return;
			}
			// Return if not a order total
			if ('2' === get_option('wal_module_product_purchase_mode')) {
				return;
			}

			wal_get_template('modules/product-purchase/variation-notice.php');
		}

		/**
		 * Render product awarding fund message in variable product page
		 *
		 * @since 3.1.0
		 * @param array $array
		 * @param string $class
		 * @param object $variation
		 *
		 * @return array
		 * */
		public function render_product_awarding_fund_message_in_variable_product( $array, $class, $variation ) {
			// Return if the array is empty.
			if (!wal_check_is_array($array)) {
				return $array;
			}

			// Return if the variation is not object.
			if (!is_object($variation)) {
				return $array;
			}

			// Return if variable product gift card display type is not 2.
			if ('2' === get_option('wal_module_product_purchase_mode')) {
				return $array;
			}

			$fund = WAL_Purchased_Product_Fund_Handler::get_fund(0, $variation->get_id(), 1, $variation->get_price());
			if (!$fund) {
				return $array;
			}

			// Get variation notice html.
			$variation_notice_html = wal_prepare_product_awarding_fund_message($variation->get_id(), $fund);
			if (!$variation_notice_html) {
				return $array;
			}

			$array['wal_variation_notice'] = $variation_notice_html;

			return $array;
		}

		/**
		 * May be credit the product purchase fund from order
		 *
		 * @since 3.1.0
		 * @param int $order_id
		 * 
		 * @return void
		 * */
		public function maybe_credit_product_purchase_fund_form_order( $order_id ) {
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}

			if ( wal_check_is_array( wal_get_valid_shipping_methods() ) ) {

				$shipping_method_instance_ids = array_intersect(wal_get_shipping_method_instance_ids_from_order( $order ), wal_get_valid_shipping_methods());

				// Return if the product purchase shipping method instance IDs is empty.
				if ( ! wal_check_is_array( $shipping_method_instance_ids ) ) {
					return;
				}
			}

			// Return if the product purchase fund is already credited.
			if ('yes' === $order->get_meta('wal_product_purchase_fund_credited')) {
				return;
			}

			if ('1' === $order->get_meta('wal_product_purchase_fund_type')) {
				$this->credit_purchased_product_fund($order);
			} else {
				$this->credit_purchased_order_fund($order);
			}

			// Update the meta.
			$order->update_meta_data('wal_product_purchase_fund_credited', 'yes');
			$order->save();
		}

		/**
		 * Credit purchased product fund
		 * 
		 * @param object $order        
		 * @since 3.1.0
		 * @return void
		 */
		public function credit_purchased_product_fund( $order ) {
			$product_purchase_funds = $order->get_meta('wal_product_purchase_details');
			$transaction_log_id = '';
			// Return if the product purchase fund details is empty.
			if (!wal_check_is_array($product_purchase_funds)) {
				return;
			}

			foreach ($product_purchase_funds as $product_purchase_id => $product_purchase_fund) {
				if (empty($product_purchase_fund)) {
					continue;
				}

				$event_message = wal_prepare_purchased_product_event_message($order, $product_purchase_id, get_option('wal_module_purchased_product_fund_credit_log'));
				$args = $this->prepare_product_purchase_fund_args($order, $product_purchase_fund, $event_message, 23);
				$transaction_log_id = wal_credit_wallet_fund($args);

				// Add the note.
				/* translators: %1s- Amount, %2s - User name */
				$note = sprintf(__('The product purchase fund amount of %1$s credited to user %2$s', 'wallet-for-woocommerce'), number_format($product_purchase_fund, 2), $order->get_user()->display_name);
				$order->add_order_note($note);
				$order->save();
			}

			/**
			 * This hook is used to do extra action after purchased product fund credited.
			 *
			 * @param int $transaction_log_id
			 * @param int $order_id
			 * 
			 * @since 3.1.0
			 */
			do_action('wal_purchased_product_fund_credited', $transaction_log_id, $order->get_id());
		}

		/**
		 * Credit purchased order fund
		 * 
		 * @param object $order        
		 * @since 3.1.0
		 * @return void
		 */
		public function credit_purchased_order_fund( $order ) {
			$product_purchase_fund = $order->get_meta('wal_purchased_product_fund');
			// Return if the product purchase fund is empty.
			if (!$product_purchase_fund) {
				return;
			}

			$event_message = wal_prepare_purchased_order_event_message($order, get_option('wal_module_purchased_order_fund_credit_log'));
			$args = $this->prepare_product_purchase_fund_args($order, $product_purchase_fund, $event_message, 23);
			$transaction_log_id = wal_credit_wallet_fund($args);

			// Add the note.
			/* translators: %1s- Amount, %2s - User name */
			$note = sprintf(__('The product purchase fund amount of %1$s credited to user %2$s', 'wallet-for-woocommerce'), number_format($product_purchase_fund, 2), $order->get_user()->display_name);
			$order->add_order_note($note);
			$order->save();
			/**
			 * This hook is used to do extra action after purchased order fund credited.
			 *
			 * @param int $transaction_log_id
			 * @param int $order_id
			 * 
			 * @since 3.1.0
			 */
			do_action('wal_purchased_order_fund_credited', $transaction_log_id, $order->get_id());
		}

		/**
		 * May be debit the product purchase fund from order.
		 * 
		 * @param int $order_id
		 * @since 3.1.0
		 * 
		 * @return void
		 * */
		public function maybe_debit_product_purchase_fund_from_order( $order_id ) {
			$order = wc_get_order($order_id);
			if (!is_object($order)) {
				return;
			}

			if ('yes' === $order->get_meta('wal_product_purchase_fund_debited')) {
				return;
			}

			if ('1' === $order->get_meta('wal_product_purchase_fund_type')) {
				$this->debit_purchased_product_fund($order);
			} else {
				$this->debit_purchased_order_fund($order);
			}

			// Update the meta.
			$order->update_meta_data('wal_product_purchase_fund_debited', 'yes');
			$order->save();
		}

		/**
		 * Debit purchase product fund
		 * 
		 * @param object $order      
		 * @since 3.1.0
		 * @return void
		 */
		public function debit_purchased_product_fund( $order ) {
			$product_purchase_funds = $order->get_meta('wal_product_purchase_details');
			// Return if the product purchase fund is empty.
			if (!wal_check_is_array($product_purchase_funds)) {
				return;
			}

			foreach ($product_purchase_funds as $product_purchase_id => $product_purchase_fund) {
				if (empty($product_purchase_fund)) {
					continue;
				}
				$event_message = wal_prepare_purchased_product_event_message($order, $product_purchase_id, get_option('wal_module_purchased_product_fund_debit_log'));
				$args = $this->prepare_product_purchase_fund_args($order, $product_purchase_fund, $event_message, 24);
				$transaction_log_id = wal_debit_wallet_fund($args);

				// Add the note.
				/* translators: %1s- Amount, %2s - User name */
				$note = sprintf(__('The product purchase fund amount of %1$s debit to user %2$s', 'wallet-for-woocommerce'), $product_purchase_fund, $order->get_user()->display_name);
				$order->add_order_note($note);
				$order->save();
				/**
				 * This hook is used to do extra action after purchased product fund debited.
				 *
				 * @param int $transaction_log_id
				 * @param int $order_id
				 * 
				 * @since 3.1.0
				 */
				do_action('wal_purchased_product_fund_debited', $transaction_log_id, $order->get_id());
			}
		}

		/**
		 * Debit purchased order fund
		 * 
		 * @param object $order      
		 * @since 3.1.0
		 * @return void
		 */
		public function debit_purchased_order_fund( $order ) {
			$product_purchase_fund = $order->get_meta('wal_purchased_product_fund');
			// Return if the product purchase fund is empty.
			if (!$product_purchase_fund) {
				return;
			}

			$event_message = wal_prepare_purchased_order_event_message($order, get_option('wal_module_purchased_order_fund_debit_log'));
			$args = $this->prepare_product_purchase_fund_args($order, $product_purchase_fund, $event_message, 24);
			$transaction_log_id = wal_debit_wallet_fund($args);

			// Add the note.
			/* translators: %1s- Amount, %2s - User name */
			$note = sprintf(__('The product purchase fund amount of %1$s debit to user %2$s', 'wallet-for-woocommerce'), $product_purchase_fund, $order->get_user()->display_name);
			$order->add_order_note($note);
			$order->save();
			/**
			 * This hook is used to do extra action after purchased order fund debited.
			 *
			 * @param int $transaction_log_id
			 * @param int $order_id
			 * 
			 * @since 3.1.0
			 */
			do_action('wal_purchased_order_fund_debited', $transaction_log_id, $order->get_id());
		}

		/**
		 * Prepare product purchase fund args
		 *
		 * @param object $order
		 * @param float $fund
		 * @param string $event_message
		 * @param string $event_id
		 * 
		 * @since 3.1.0
		 * @return array
		 */
		public function prepare_product_purchase_fund_args( $order, $fund, $event_message, $event_id ) {
			/**
			 * This hook is used to alert the prepare product purchase args.
			 *
			 * @param object $order
			 * @param float fund
			 * @param string $event_message
			 * @since 3.1.0
			 */
			return apply_filters('wal_prepare_product_purchase_args', array(
				'user_id' => $order->get_customer_id(),
				'order_id' => $order->get_id(),
				'amount' => wal_convert_price($fund, true),
				'event_id' => $event_id,
				'event_message' => $event_message,
				'currency' => $order->get_currency(),
				'update_topup_total' => true,
			));
		}

		/**
		 * Add the custom email classes.
		 *
		 * @param array $classes
		 * @since 3.1.0
		 * 
		 * @return array
		 */
		public function email_classes( $classes ) {
			$classes = ( wal_check_is_array($classes) ) ? $classes : array();

			$classes['customer-purchased-product-fund'] = 'WAL_Customer_Purchased_Product_Fund_Notification';
			$classes['customer-purchased-order-fund'] = 'WAL_Customer_Purchased_Order_Fund_Notification';

			return $classes;
		}

		/**
		 * Register the store API notices for cart / checkout blocks.
		 * 
		 * @since 3.7.0
		 * @param array $notices
		 * @return array
		 */
		public function register_store_api_cart_notices( $notices ) {
			$notices = wal_check_is_array($notices) ? $notices : array();
			$messages = self::get_cart_product_fund_awarding_messages();
			if (!wal_check_is_array($messages)) {
				return $notices;
			}

			foreach ($messages as $key => $message) {
				$notices['wal-product-purchase-notice-' . $key] = array(
					'type' => 'info',
					'content' => $message,
				);
			}

			return $notices;
		}
	}

}
