<?php

/**
 * Module - Gift Voucher.
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Gift_Voucher_Module')) {

	/**
	 * Class.
	 * */
	class WAL_Gift_Voucher_Module extends WAL_Modules {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id = 'gift_voucher';
			$this->title = __('Gift Voucher', 'wallet-for-woocommerce');
			$this->description = __('Users can add the funds to their wallet by redeeming gift vouchers.', 'wallet-for-woocommerce');
			$this->tooltip = __('By enabling this checkbox, you can allow your users to add the funds to their wallet by redeeming gift vouchers.', 'wallet-for-woocommerce');

			parent::__construct();
		}

		/**
		 * Get the setting tabs.
		 * 
		 * @return array
		 */
		public function setting_tabs() {
			$tabs = array(
				'gift_voucher_generation' => __('Gift Voucher Generation', 'wallet-for-woocommerce'),
				'gift_voucher' => __('Gift Vouchers', 'wallet-for-woocommerce'),
				'settings' => __('Settings', 'wallet-for-woocommerce'),
			);
			/**
			 * This hook is used to alter the current module tabs.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->get_plugin_slug() . '_get_module_tabs_' . $this->get_id(), $tabs);
		}

		/**
		 * Show buttons?.
		 * 
		 * @return string
		 */
		public function show_buttons() {
			global $current_tab;

			switch ($current_tab) {
				case 'settings':
					return true;
					break;
			}

			return false;
		}

		/**
		 * Output the tab content.
		 */
		public function output_tab_content() {
			if (!class_exists('WAL_Gift_Vouchers_List_Table')) {
				require_once WAL_PLUGIN_PATH . '/inc/admin/menu/wp-list-tables/class-wal-gift-vouchers-list-table.php' ;
			}

			$post_table = new WAL_Gift_Vouchers_List_Table();
			$post_table->render();
		}

		/**
		 * Get settings for gift voucher generation section array.
		 * 
		 * @return array
		 */
		protected function gift_voucher_generation_section_array() {
			$section_fields = array();

			// Gift Voucher Generation section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Generate Gift Voucher', 'wallet-for-woocommerce'),
				'id' => 'wal_gift_voucher_generation_options',
			);
			$section_fields[] = array(
				'title' => __('Prefix', 'wallet-for-woocommerce'),
				'type' => 'text',
				'id' => 'prefix',
				'autoload' => false,
			);
			$section_fields[] = array(
				'title' => __('Suffix', 'wallet-for-woocommerce'),
				'type' => 'text',
				'id' => 'suffix',
				'autoload' => false,
			);
			$section_fields[] = array(
				'title' => __('Gift Voucher Type', 'wallet-for-woocommerce'),
				'type' => 'select',
				'id' => 'type',
				'class' => 'wal_module_gift_voucher_type',
				'autoload' => false,
				'options' => array(
					'1' => __('Numeric', 'wallet-for-woocommerce'),
					'2' => __('Alphanumeric', 'wallet-for-woocommerce'),
				),
			);
			$section_fields[] = array(
				'title' => __('Alphabets Excluded from Gift Voucher Creation', 'wallet-for-woocommerce'),
				'type' => 'text',
				'class' => 'wal_module_gift_voucher_excluded_alphabets',
				'id' => 'exclude_alphabets',
				'autoload' => false,
			);
			$section_fields[] = array(
				'title' => __('Gift Voucher Length', 'wallet-for-woocommerce'),
				'type' => 'number',
				'id' => 'length',
				'custom_attributes' => array( 'min' => '1' ),
				'autoload' => false,
			);
			$section_fields[] = array(
				'title' => __('Wallet Funds Per Gift Voucher', 'wallet-for-woocommerce'),
				'type' => 'text',
				'id' => 'amount',
				'autoload' => false,
			);
			$section_fields[] = array(
				'title' => __('Number of Vouchers to be Generated', 'wallet-for-woocommerce'),
				'type' => 'text',
				'id' => 'count',
				'autoload' => false,
			);
			$section_fields[] = array(
				'title' => __('Expiry Date for Gift Vouchers', 'wallet-for-woocommerce'),
				'type' => 'number',
				'id' => 'expiry_date',
				'autoload' => false,
				'custom_attributes' => array( 'min' => '1' ),
			);
			$section_fields[] = array(
				'type' => 'wal_custom_fields',
				'wal_field' => 'button',
				'class' => 'wal-generate-gift-vouchers-btn',
				'autoload' => false,
				'default' => __('Generate Gift Vouchers', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_gift_voucher_generation_options',
			);
			// Gift Voucher Generation section end.

			return $section_fields;
		}

		/**
		 * Get settings for settings section array.
		 * 
		 * @return array
		 */
		protected function settings_section_array() {
			$section_fields = array();

			// General section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('General Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_gift_voucher_general_options',
			);
			$section_fields[] = array(
				'title' => __('Allow Users to Redeem Gift Vouchers', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'autoload' => false,
				'id' => $this->get_option_key('gift_voucher_enabled'),
				'desc' => __('By enabling this checkbox, you can allow your users to request funds from other users on the site.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_gift_voucher_general_options',
			);
			// General section end.
			// Localization section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Localization', 'wallet-for-woocommerce'),
				'id' => 'wal_gift_voucher_localization_options',
			);
			$section_fields[] = array(
				'title' => __('Gift Voucher Endpoint', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'gift_voucher',
				'id' => $this->get_option_key('gift_voucher_menu_endpoint'),
			);
			$section_fields[] = array(
				'title' => __('Gift Voucher Redeeming Menu Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Gift Voucher',
				'autoload' => false,
				'id' => $this->get_option_key('gift_voucher_menu_label'),
			);
			$section_fields[] = array(
				'title' => __('Gift Voucher Redeeming Field Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Redeem Your Gift Voucher',
				'autoload' => false,
				'id' => $this->get_option_key('gift_voucher_field_label'),
			);
			$section_fields[] = array(
				'title' => __('Gift Voucher Redeeming Field Placeholder', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Enter a Gift Voucher',
				'autoload' => false,
				'id' => $this->get_option_key('gift_voucher_field_placeholder_label'),
			);
			$section_fields[] = array(
				'title' => __('Gift Voucher Redeem Button Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Redeem',
				'autoload' => false,
				'id' => $this->get_option_key('gift_voucher_redeem_button_label'),
			);
			$section_fields[] = array(
				'title' => __('Gift Voucher Log', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds successfully credited for redeeming a Gift Voucher{code}',
				'autoload' => false,
				'id' => $this->get_option_key('gift_voucher_credit_log_message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_gift_voucher_localization_options',
			);
			// Localization section end.
			// Messages section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Messages', 'wallet-for-woocommerce'),
				'id' => 'wal_gift_voucher_messages_options',
			);
			$section_fields[] = array(
				'title' => __("Message to Display when a User didn't Enter the Code", 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter a code',
				'autoload' => false,
				'id' => $this->get_option_key('gift_voucher_empty_code_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when a User Enters Invalid/Already Redeemed Code', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter a valid code',
				'autoload' => false,
				'id' => $this->get_option_key('gift_voucher_valid_code_message'),
			);
			$section_fields[] = array(
				'title' => __('Gift Voucher Redeemed Success Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Gift Voucher Redeemed Successfully',
				'autoload' => false,
				'id' => $this->get_option_key('gift_voucher_redeemed_message'),
			);
			$section_fields[] = array(
				'title' => __('Gift Voucher Error Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Currently, your wallet has been disabled. Hence, you are unable to redeem the Gift Voucher.',
				'autoload' => false,
				'id' => $this->get_option_key('gift_voucher_error_message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_gift_voucher_messages_options',
			);
			// Messages section end.

			return $section_fields;
		}

		/**
		 * Include files.
		 */
		protected function include_files() {
			include_once WAL_ABSPATH . 'inc/entity/class-wal-gift-voucher.php' ;
		}

		/**
		 * Actions.
		 */
		protected function actions() {
			//Add the email classes.
			add_filter('wal_wallet_email_classes', array( $this, 'email_classes' ), 10, 1);
			//Generate the gift vouchers.
			add_action('wp_ajax_wal_generate_gift_vouchers', array( $this, 'generate_gift_vouchers' ));

			//Handle the expiry of gift vouchers.
			add_action('wal_handle_expired_cron', array( $this, 'maybe_handle_gift_voucher_expiry' ));
		}

		/**
		 * Admin actions.
		 */
		protected function admin_action() {
			// Export the CSV.
			add_action('admin_init', array( $this, 'export_csv' ));
		}

		/**
		 * Front end actions.
		 */
		protected function frontend_action() {
			//Add the wallet endpoints.
			add_filter('wal_wallet_endpoints', array( $this, 'add_endpoints' ), 10, 1);
			//Add the dashboard menus.
			add_filter('wal_dashboard_menus', array( $this, 'add_dashboard_menus' ), 10, 1);
			//Add the hook for gift voucher menu content.
			add_action('wal_dashboard_menu_content_gift_voucher', array( $this, 'gift_voucher_menu_content' ));
			//Process the gift voucher.
			add_action('wp_loaded', array( $this, 'process_gift_voucher' ));
		}

		/**
		 * Enqueue admin end JS files.
		 */
		protected function admin_external_js_files() {
			// Gift Voucher.   
			wp_enqueue_script('wal-gift-voucher', WAL_PLUGIN_URL . '/assets/js/admin/gift-voucher.js', array( 'jquery', 'jquery-blockui' ), WAL_VERSION);
			wp_localize_script(
					'wal-gift-voucher', 'wal_gift_voucher_params', array(
				'gift_voucher_nonce' => wp_create_nonce('wal-gift-voucher-nonce'),
				'processing_msg' => __('Gift Vouchers are generating. Please wait to complete the action.', 'wallet-for-woocommerce'),
				'confirm_msg' => __('Are you sure you want to proceed?', 'wallet-for-woocommerce'),
					)
			);
		}

		/**
		 * Add the custom email classes.
		 * 
		 * @return array
		 */
		public function email_classes( $classes ) {
			$classes = ( wal_check_is_array($classes) ) ? $classes : array();

			$classes['customer-gift-voucher-redeemed'] = 'WAL_Customer_Gift_Voucher_Redeemed_Notification';

			return $classes;
		}

		/**
		 * Export the CSV.
		 * */
		public function export_csv() {

			if (!isset($_REQUEST['wal_export_csv'])) {
				return;
			}

			$action = wc_clean(wp_unslash($_REQUEST['wal_export_csv']));

			switch ($action) {
				case 'gift_voucher':
					include_once WAL_ABSPATH . 'inc/exports/class-wal-gift-vouchers-export-csv.php';

					$exporter = new WAL_Gift_Vouchers_Export_CSV();
					$exporter->export();
					break;
			}
		}

		/**
		 * Handles the expiry of gift voucher.
		 * 
		 * @return void.
		 * */
		public static function maybe_handle_gift_voucher_expiry() {

			$current_date = WAL_Date_Time::get_mysql_date_time_format('now', true);
			$args = array(
				'post_type' => WAL_Register_Post_Types::GIFT_VOUCHER_POSTTYPE,
				'post_status' => 'wal_unused',
				'fields' => 'ids',
				'numberposts' => '-1',
				'meta_query' => array(
					array(
						'key' => 'wal_expiry_date',
						'value' => $current_date,
						'compare' => '<=',
						'type' => 'DATETIME',
					),
					array(
						'key' => 'wal_expiry_date',
						'value' => '',
						'compare' => '!=',
					),
				),
			);

			$post_ids = get_posts($args);

			// Return if the post ids is not exists.
			if (!wal_check_is_array($post_ids)) {
				return;
			}

			foreach ($post_ids as $post_id) {
				$gift_voucher = wal_get_gift_voucher($post_id);
				if (!$gift_voucher->exists()) {
					continue;
				}

				wal_update_wallet($post_id, array(), array( 'post_status' => 'wal_expired' ));

				/**
				 * This hook is used to do extra action after gift voucher expired.
				 * 
				 * @since 1.0
				 */
				do_action('wal_gift_voucher_expired', $gift_voucher);
			}
		}

		/**
		 * Generate the gift vouchers.
		 * 
		 * @return void
		 */
		public function generate_gift_vouchers() {
			check_ajax_referer('wal-gift-voucher-nonce', 'wal_security');

			try {

				if (!isset($_POST)) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				// Sanitize post values.
				$form_data = !empty($_POST['form_data']) ? wc_clean(wp_unslash($_POST['form_data'])) : array();
				if (!wal_check_is_array($form_data)) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				if (!isset($form_data['length']) || empty($form_data['length'])) {
					throw new exception(__('Please enter the character length for gift vouchers', 'wallet-for-woocommerce'));
				}

				if (!isset($form_data['amount']) || empty($form_data['amount'])) {
					throw new exception(__('Please enter the funds for gift vouchers', 'wallet-for-woocommerce'));
				}

				if (!isset($form_data['count']) || empty($form_data['count'])) {
					throw new exception(__('Please enter the number of gift vouchers to generate', 'wallet-for-woocommerce'));
				}

				$total_count = $form_data['length'] * $form_data['length'];
				if ($total_count < $form_data['count']) {
					/* Translators: %s- Count */
					throw new exception(sprintf(__('Please enter the number of gift voucher to generate less than or equal to %d', 'wallet-for-woocommerce'), $total_count));
				}

				for ($i = 0; $i < $form_data['count']; $i++) {
					$args = array(
						'wal_code' => $this->generate_code($form_data),
						'wal_amount' => $form_data['amount'],
						'wal_currency' => get_woocommerce_currency(),
						'wal_expiry_date' => $this->get_expiry_date($form_data['expiry_date']),
					);

					// Create the gift voucher.
					wal_create_new_gift_voucher($args);
				}

				wp_send_json_success(array( 'msg' => __('Gift Vouchers Generated Successfully', 'wallet-for-woocommerce') ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Get the gift voucher expiry date.
		 *
		 * @return string
		 */
		private function get_expiry_date( $expiry_days ) {
			$expiry_date = '';

			if (!empty($expiry_days)) {
				// Get the current UTC time.
				$date_object = WAL_Date_Time::get_date_time_object('now', true);
				$date_object->modify('+ ' . $expiry_days . ' days');
				$expiry_date = $date_object->format('Y-m-d H:i:s');
			}
			/**
			 * This hook is used to validate the gift voucher expiry day.
			 * 
			 * @since 1.0
			 */
			return apply_filters('wal_gift_voucher_expiry_day', $expiry_date, $expiry_days);
		}

		/**
		 * Generate the code.
		 * 
		 * @return string
		 */
		private function generate_code( $args ) {
			$bool = true;

			while ($bool) {
				$code = wal_generate_random_code($args);
				if (!wal_get_gift_voucher_id_by_code($code)) {
					$bool = false;
				}
			}

			return $code;
		}

		/**
		 * Add the custom wallet endpoints.
		 * 
		 * @return array
		 */
		public function add_endpoints( $endpoints ) {
			$endpoints = ( wal_check_is_array($endpoints) ) ? $endpoints : array();

			$endpoints['gift_voucher'] = wal_get_gift_voucher_menu_endpoint();

			return $endpoints;
		}

		/**
		 * Add the custom dashboard menus.
		 * 
		 * @return array
		 */
		public function add_dashboard_menus( $menus ) {
			$menus = ( wal_check_is_array($menus) ) ? $menus : array();

			$menus['gift_voucher'] = array(
				'name' => wal_get_gift_voucher_menu_label(),
				'endpoint' => wal_get_gift_voucher_menu_endpoint(),
			);

			return $menus;
		}

		/**
		 * Validate the current user.
		 * 
		 * @return boolean
		 */
		public function is_invalid_user() {
			// Return if the wallet is in active.
			if ('wal_inactive' == WAL_Current_User_Wallet::get_status()) {
				return true;
			}

			return false;
		}


		/**
		 * Display the dashboard gift voucher menu content.
		 * 
		 * @return void
		 */
		public function gift_voucher_menu_content() {
			$show = true;
			if ($this->is_invalid_user()) {
				$show = false;
				WAL_Dashboard_Handler::add_error(esc_html(get_option('wal_module_gift_voucher_error_message')));
			}
			wal_get_template('modules/gift-voucher/redeem-form.php', array( 'show' => $show ));
		}

		/**
		 * Process the gift voucher.
		 */
		public function process_gift_voucher() {
			$nonce_value = isset($_POST['wal_redeem_gift_voucher_nonce']) ? wc_clean(wp_unslash($_POST['wal_redeem_gift_voucher_nonce'])) : null;
			if (!isset($_POST['wal_redeem_gift_voucher_action']) || empty($_POST['wal_redeem_gift_voucher_action']) || !wp_verify_nonce($nonce_value, 'wal-redeem-gift-voucher')) {
				return;
			}

			try {

				$code = isset($_POST['wal_redeem_gift_voucher_code']) ? wc_clean(wp_unslash($_POST['wal_redeem_gift_voucher_code'])) : '';

				//Validate the redeem gift voucher code is empty.
				if (!$code) {
					throw new Exception(get_option('wal_module_gift_voucher_empty_code_message'));
				}

				//Validate the gift voucher code.
				$gift_voucher_id = wal_get_gift_voucher_id_by_code($code, 'wal_unused');
				if (empty($gift_voucher_id)) {
					throw new Exception(get_option('wal_module_gift_voucher_valid_code_message'));
				}

				$gift_voucher = wal_get_gift_voucher($gift_voucher_id);
				if (!$gift_voucher->exists()) {
					throw new Exception(get_option('wal_module_gift_voucher_valid_code_message'));
				}

				$log = get_option('wal_module_gift_voucher_credit_log_message');
				$event_message = str_replace('{code}', $code, $log);

				$args = array(
					'user_id' => WAL_Current_User_Wallet::get_user_id(),
					'amount' => $gift_voucher->get_amount(),
					'event_id' => 14,
					'event_message' => $event_message,
				);

				$transaction_log_id = wal_credit_wallet_fund($args);

				// Credit the wallet amount from the gift voucher.
				if (!$transaction_log_id) {
					throw new Exception(__('Something wrong please try again', 'wallet-for-woocommerce'));
				}

				$meta_args = array(
					'wal_redeemed_date' => WAL_Date_Time::get_mysql_date_time_format('now', true),
				);

				$post_args = array(
					'post_parent' => WAL_Current_User_Wallet::get_user_id(),
					'post_status' => 'wal_used',
				);

				//Update the gift voucher.
				wal_update_gift_voucher($gift_voucher_id, $meta_args, $post_args);

				// Unset the form values.
				$keys = array(
					'wal_redeem_gift_voucher_code',
				);

				wal_unset_global_variable_values($keys);

				WAL_Dashboard_Handler::add_message(get_option('wal_module_gift_voucher_redeemed_message'));
				/**
				 * This hook is used to do extra action gift voucher redeemed.
				 * 
				 * @since 1.0
				 */
				do_action('wal_gift_voucher_redeemed', $gift_voucher_id);
			} catch (Exception $ex) {

				WAL_Dashboard_Handler::add_error($ex->getMessage());
			}
		}
	}

}
