<?php

/**
 * Module - Fund Transfer.
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Fund_Transfer_Module')) {

	/**
	 * Class.
	 * */
	class WAL_Fund_Transfer_Module extends WAL_Modules {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id = 'fund_transfer';
			$this->title = __('Fund Transfer & Request', 'wallet-for-woocommerce');
			$this->description = __('Users can transfer their funds & request funds from other users on the site.', 'wallet-for-woocommerce');
			$this->tooltip = __('By enabling this checkbox, users can transfer their funds & request funds from other users on the site.', 'wallet-for-woocommerce');

			parent::__construct();
		}

		/**
		 * Get the setting tabs.
		 * 
		 * @return array
		 */
		public function setting_tabs() {
			$tabs = array(
				'fund_transfer' => __('Fund Transfer', 'wallet-for-woocommerce'),
				'fund_request' => __('Fund Request', 'wallet-for-woocommerce'),
				'advanced' => __('Advanced', 'wallet-for-woocommerce'),
				'localization' => __('Localization', 'wallet-for-woocommerce'),
				'messages' => __('Messages', 'wallet-for-woocommerce'),
			);
			/**
			 * This hook is used to alter the current module tabs.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->get_plugin_slug() . '_get_module_tabs_' . $this->get_id(), $tabs);
		}

		/**
		 * Get settings for fund transfer section array.
		 * 
		 * @return array
		 */
		protected function fund_transfer_section_array() {
			$section_fields = array();

			// Fund Transfer section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Fund Transfer Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_transfer_options',
			);
			$section_fields[] = array(
				'title' => __('Fund Transfer', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'autoload' => false,
				'id' => $this->get_option_key('transfer_enabled'),
				'desc' => __('By enabling this checkbox, you can allow your users to directly transfer their Wallet balance to other users on the site.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Minimum Amount for Wallet Transfer', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-fund-transfer-field',
				'id' => $this->get_option_key('fund_transfer_min_amount'),
				'desc' => __('You can set the minimum amount required to transfer in a single transaction.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Amount for Wallet Transfer', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-fund-transfer-field',
				'id' => $this->get_option_key('fund_transfer_max_amount'),
				'desc' => __('You can set the maximum amount that can be transferred in a single transaction.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Number of Transactions Per Day', 'wallet-for-woocommerce'),
				'type' => 'number',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-fund-transfer-field',
				'custom_attributes' => array( 'min' => 1 ),
				'id' => $this->get_option_key('fund_transfer_count_per_day'),
				'desc' => __('You can set the maximum number of wallet transfer can be performed in a day.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Number of Unique Transfers Per Day', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-fund-transfer-field',
				'id' => $this->get_option_key('fund_transfer_user_count_per_day'),
				'desc' => __('You can set the maximum number of transfers that can be made to one user in a day.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Wallet Transfer Fee', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'autoload' => false,
				'class' => 'wal-fund-transfer-field',
				'id' => $this->get_option_key('fund_transfer_fee_enabled'),
				'desc' => __('By enabling this checkbox, users can be charged a fee for transferring their wallet balance.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Transfer Fee Type', 'wallet-for-woocommerce'),
				'type' => 'select',
				'default' => '1',
				'autoload' => false,
				'class' => 'wal-fund-transfer-field',
				'id' => $this->get_option_key('fund_transfer_fee_type'),
				'options' => array(
					'1' => __('Fixed', 'wallet-for-woocommerce'),
					'2' => __('Percentage', 'wallet-for-woocommerce'),
				),
			);
			$section_fields[] = array(
				'title' => __('Transfer Fee Value', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-fund-transfer-field',
				'id' => $this->get_option_key('fund_transfer_fee_amount'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_transfer_options',
			);
			// Fund transfer section end.

			return $section_fields;
		}

		/**
		 * Get settings for fund request section array.
		 * 
		 * @return array
		 */
		protected function fund_request_section_array() {
			$section_fields = array();

			// Fund request section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Fund Request Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_request_options',
			);
			$section_fields[] = array(
				'title' => __('Fund Request', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_enabled'),
				'desc' => __(' By enabling this checkbox, you can allow your users to request funds from other users on the site.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Minimum Amount for Wallet Request', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-fund-request-field',
				'id' => $this->get_option_key('fund_request_min_amount'),
				'desc' => __('You can set the minimum amount required to give a request in a single transaction.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Amount for Wallet Request', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-fund-request-field',
				'id' => $this->get_option_key('fund_request_max_amount'),
				'desc' => __('You can set the maximum amount that can be requested in a single transaction.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Number of Requests Per Day', 'wallet-for-woocommerce'),
				'type' => 'number',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-fund-request-field',
				'custom_attributes' => array( 'min' => 1 ),
				'id' => $this->get_option_key('fund_request_count_per_day'),
				'desc' => __('You can set the maximum number of wallet request can be submitted in a day.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Number of Unique Requests Per Day', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'price',
				'default' => '',
				'autoload' => false,
				'class' => 'wal-fund-request-field',
				'id' => $this->get_option_key('fund_request_max_amount_per_day'),
				'desc' => __('You can set the maximum number of requests that can be submitted to one user in a day.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_request_options',
			);
			// Fund request section end.

			return $section_fields;
		}

		/**
		 * Get settings for advanced section array.
		 * 
		 * @return array
		 */
		protected function advanced_section_array() {
			$section_fields = array();
			$wp_user_roles = wal_get_wp_user_roles();

			// Fund transfer display section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Display Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_transfer_display_options',
			);
			$section_fields[] = array(
				'title' => __('Pagination Size', 'wallet-for-woocommerce'),
				'type' => 'number',
				'default' => '5',
				'autoload' => false,
				'custom_attributes' => array( 'min' => 1 ),
				'id' => $this->get_option_key('fund_transfer_transactions_pagination_count'),
				'desc' => __('You can set the number of entries to display in the transaction log[Wallet Dashboard -> Fund Transfer & Request -> Transactions].', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_transfer_display_options',
			);
			// Fund transfer display section end.
			// Restriction section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Restriction Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_restriction_options',
			);
			$section_fields[] = array(
				'title' => __('User Selection', 'wallet-for-woocommerce'),
				'type' => 'select',
				'default' => '1',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_user_restriction_type'),
				'class' => 'wal-user-restriction-type',
				'options' => array(
					'1' => __('All Users', 'wallet-for-woocommerce'),
					'2' => __('Include Users', 'wallet-for-woocommerce'),
					'3' => __('Exclude Users', 'wallet-for-woocommerce'),
					'4' => __('Include User Roles', 'wallet-for-woocommerce'),
					'5' => __('Exclude User Roles', 'wallet-for-woocommerce'),
				),
			);
			$section_fields[] = array(
				'title' => __('Select Users', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'ajaxmultiselect',
				'list_type' => 'customers',
				'action' => 'wal_json_search_customers',
				'default' => array(),
				'autoload' => false,
				'placeholder' => __('Select a User', 'wallet-for-woocommerce'),
				'class' => 'wal-user-restriction-field wal-include-users-field',
				'id' => $this->get_option_key('fund_transfer_include_users'),
			);
			$section_fields[] = array(
				'title' => __('Select Users', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'ajaxmultiselect',
				'list_type' => 'customers',
				'class' => 'wal-user-restriction-field wal-exclude-users-field',
				'action' => 'wal_json_search_customers',
				'default' => array(),
				'autoload' => false,
				'placeholder' => __('Select a User', 'wallet-for-woocommerce'),
				'id' => $this->get_option_key('fund_transfer_exclude_users'),
			);
			$section_fields[] = array(
				'title' => __('Select User Roles', 'wallet-for-woocommerce'),
				'type' => 'multiselect',
				'default' => array(),
				'autoload' => false,
				'custom_attributes' => array( 'placeholder' => __('Select a User Role', 'wallet-for-woocommerce') ),
				'class' => 'wal_select2 wal-user-restriction-field wal-include-user-roles-field',
				'id' => $this->get_option_key('fund_transfer_include_user_roles'),
				'options' => $wp_user_roles,
			);
			$section_fields[] = array(
				'title' => __('Select User Roles', 'wallet-for-woocommerce'),
				'type' => 'multiselect',
				'default' => array(),
				'autoload' => false,
				'custom_attributes' => array( 'placeholder' => __('Select a User Role', 'wallet-for-woocommerce') ),
				'class' => 'wal_select2 wal-user-restriction-field wal-exclude-user-roles-field',
				'id' => $this->get_option_key('fund_transfer_exclude_user_roles'),
				'options' => $wp_user_roles,
			);

			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_restriction_options',
			);
			// Restriction section end.
			// OTP section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('OTP Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_otp_options',
			);
			$section_fields[] = array(
				'title' => __('Enable OTP', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_otp_enabled'),
				'desc' => __('By enabling this checkbox, you can allow your users to receive an OTP via email while transferring the funds.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('OTP Length', 'wallet-for-woocommerce'),
				'type' => 'number',
				'default' => '6',
				'autoload' => false,
				'class' => 'wal-fund-transfer-otp-field',
				'custom_attributes' => array( 'min' => 1 ),
				'id' => $this->get_option_key('fund_transfer_otp_length'),
			);
			$section_fields[] = array(
				'title' => __('OTP Validity in Minutes', 'wallet-for-woocommerce'),
				'type' => 'number',
				'default' => '5',
				'autoload' => false,
				'class' => 'wal-fund-transfer-otp-field',
				'custom_attributes' => array( 'min' => 1 ),
				'id' => $this->get_option_key('fund_transfer_otp_validity'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_otp_options',
			);
			// OTP section end.

			return $section_fields;
		}

		/**
		 * Get the settings for localization section array.
		 * 
		 * @return array
		 */
		protected function localization_section_array() {
			$section_fields = array();

			// Dashboard endpoints section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Endpoints', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_tansfer_dashboard_endpoints_options',
			);
			$section_fields[] = array(
				'title' => __('Fund Transfer & Request Endpoint', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'fund_transfer_request',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_details_menu_endpoint'),
			);
			$section_fields[] = array(
				'title' => __('Transactions Endpoint', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'transactions',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_transaction_menu_endpoint'),
			);
			$section_fields[] = array(
				'title' => __('Fund Transfer Endpoint', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'fund_transfer',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_menu_endpoint'),
			);
			$section_fields[] = array(
				'title' => __('Fund Request Endpoint', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'fund_request',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_menu_endpoint'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_tansfer_dashboard_endpoints_options',
			);
			// Dashboard endpoints section end.
			// Dashboard menu labels section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Menu Labels', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_tansfer_menu_labels_options',
			);
			$section_fields[] = array(
				'title' => __('Fund Transfer & Request Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Fund Transfer & Request',
				'autoload' => false,
				'id' => $this->get_option_key('fund_tansfer_details_menu_label'),
			);
			$section_fields[] = array(
				'title' => __('Transactions Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Transactions',
				'autoload' => false,
				'id' => $this->get_option_key('fund_tansfer_transaction_menu_label'),
			);
			$section_fields[] = array(
				'title' => __('Fund Transfer Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Fund Transfer',
				'autoload' => false,
				'id' => $this->get_option_key('fund_tansfer_menu_label'),
			);
			$section_fields[] = array(
				'title' => __('Fund Request Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Fund Request',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_menu_label'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_tansfer_menu_labels_options',
			);
			// Dashboard menu labels section end.
			// Dashboard transaction labels section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Transaction Labels', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_tansfer_transaction_labels_options',
			);
			$section_fields[] = array(
				'title' => __('Username Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Username',
				'autoload' => false,
				'id' => $this->get_option_key('transaction_username_label'),
			);
			$section_fields[] = array(
				'title' => __('Total Amount Transferred Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Total Amount Transferred',
				'autoload' => false,
				'id' => $this->get_option_key('transaction_transfered_total_label'),
			);
			$section_fields[] = array(
				'title' => __('Total Amount Received Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Total Amount Received',
				'autoload' => false,
				'id' => $this->get_option_key('transaction_received_total_label'),
			);
			$section_fields[] = array(
				'title' => __('Last Activity Date Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Last Activity Date',
				'autoload' => false,
				'id' => $this->get_option_key('transaction_last_activity_date_label'),
			);
			$section_fields[] = array(
				'title' => __('Status Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Status',
				'autoload' => false,
				'id' => $this->get_option_key('transaction_status_label'),
			);
			$section_fields[] = array(
				'title' => __('Actions Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Actions',
				'autoload' => false,
				'id' => $this->get_option_key('transaction_actions_label'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_tansfer_transaction_labels_options',
			);
			// Dashboard transaction labels section end.
			// Dashboard overview labels section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Overview Labels', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_tansfer_overview_labels_options',
			);
			$section_fields[] = array(
				'title' => __('Total Transferred Amount Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Total Transferred Amount',
				'autoload' => false,
				'id' => $this->get_option_key('transferred_total_label'),
			);
			$section_fields[] = array(
				'title' => __('Total Received Amount Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Total Received Amount',
				'autoload' => false,
				'id' => $this->get_option_key('received_total_label'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_tansfer_overview_labels_options',
			);
			// Dashboard overview labels section end.
			// Fund transfer form labels section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Fund Transfer Form Labels', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_tansfer_form_labels_options',
			);
			$section_fields[] = array(
				'title' => __('User Selection Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'User Selection',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_user_selection_label'),
			);
			$section_fields[] = array(
				'title' => __('User Selection Placeholder Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Enter the Username/Email',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_user_selection_placeholder_label'),
			);
			$section_fields[] = array(
				'title' => __('Username Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Username',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_user_name_label'),
			);
			$section_fields[] = array(
				'title' => __('Available Wallet Balance Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Available Wallet Balance',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_user_balance_label'),
			);
			$section_fields[] = array(
				'title' => __('Amount to Transfer Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Amount to Transfer',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_amount_label'),
			);
			$section_fields[] = array(
				'title' => __('Amount to Transfer Placeholder Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Enter the Amount',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_amount_placeholder_label'),
			);
			$section_fields[] = array(
				'title' => __('Transfer Fee Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Transfer Fee',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_fee_label'),
			);
			$section_fields[] = array(
				'title' => __('Reason Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Reason',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_reason_label'),
			);
			$section_fields[] = array(
				'title' => __('Reason Placeholder Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Enter the Reason',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_reason_placeholder_label'),
			);
			$section_fields[] = array(
				'title' => __('OTP Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Verify OTP',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_otp_label'),
			);
			$section_fields[] = array(
				'title' => __('Transfer Button Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Transfer',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_button_label'),
			);
			$section_fields[] = array(
				'title' => __('Confirm Button Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Confirm',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_confirm_button_label'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_tansfer_form_labels_options',
			);
			// Fund transfer form labels section end.
			// Fund request form labels section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Fund Request Form Labels', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_request_form_labels_options',
			);
			$section_fields[] = array(
				'title' => __('User Selection Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'User Selection',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_user_selection_label'),
			);
			$section_fields[] = array(
				'title' => __('User Selection Placeholder Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Select the User',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_user_selection_placeholder_label'),
			);
			$section_fields[] = array(
				'title' => __('Username Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Username',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_user_name_label'),
			);
			$section_fields[] = array(
				'title' => __('Available Wallet Balance Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Available Wallet Balance',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_user_balance_label'),
			);
			$section_fields[] = array(
				'title' => __('Amount to Request Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Amount to Request',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_amount_label'),
			);
			$section_fields[] = array(
				'title' => __('Amount to Request Placeholder Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Enter the Amount',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_amount_placeholder_label'),
			);
			$section_fields[] = array(
				'title' => __('Reason Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Reason',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_reason_label'),
			);
			$section_fields[] = array(
				'title' => __('Reason Placeholder Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Enter the Reason',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_reason_placeholder_label'),
			);
			$section_fields[] = array(
				'title' => __('Request Button Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Request',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_button_label'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_request_form_labels_options',
			);
			// Fund request form labels section end.

			return $section_fields;
		}

		/**
		 * Get the settings for messages section array.
		 * 
		 * @return array
		 */
		protected function messages_section_array() {
			$section_fields = array();

			// Log section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Transaction Log', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_tansfer_log_options',
			);
			$section_fields[] = array(
				'title' => __('Fund Transferred Log', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds successfully transferred to {user_name}',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_log_message'),
			);
			$section_fields[] = array(
				'title' => __('Fund Received Log', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds successfully received from {user_name}',
				'autoload' => false,
				'id' => $this->get_option_key('fund_received_log_message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_tansfer_log_options',
			);
			// Log section end.
			// Fund transfer section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Fund Transfer Messages', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_transfer_message_options',
			);
			$section_fields[] = array(
				'title' => __("Message to Display when a Sender didn't Select the Receiver Name", 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please select a receiver name',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_empty_receiver_message'),
			);
			$section_fields[] = array(
				'title' => __("Message to Display when a Sender didn't Enter the Funds", 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter funds to transfer',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_empty_fund_message'),
			);
			$section_fields[] = array(
				'title' => __('Minimum Amount Required to Transfer Funds in a Single Transaction', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter an amount more than {minimum_amount}',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_minimum_amount_message'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Amount that can be Transferred in a Single Transaction', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter an amount less than {maximum_amount}',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_maximum_amount_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when a User has Reached the Maximum Number of Fund Transfers Per Day', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'You are allowed to transfer your funds a maximum of {transfer_count} time(s) per day. Since you have reached the count, you cannot transfer your funds anymore today.',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_maximum_count_per_day_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when a User has Reached the Maximum Number of Unique Fund Transfers Per Day', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'You are allowed to transfer your funds a maximum of {user_count} time(s) to each user per day. Since you have reached the count, you cannot transfer your funds anymore today.',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_maximum_user_count_per_day_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when a User tries to Transfer More than the Available Funds on their Wallet', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Insufficient balance to transfer funds',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_insufficient_wallet_fund_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when the User(s)/User Role(s) is Restricted to Transfer Funds', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Currently, you are restricted to transfer your funds to others.',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_user_restricted_message'),
			);
			$section_fields[] = array(
				'title' => __('Fund Transfer Success Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Fund Transferred Successfully',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_success_message'),
			);
			$section_fields[] = array(
				'title' => __('Invalid OTP Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'The OTP which you have entered is incorrect. Please enter the correct OTP.',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_invalid_otp_message'),
			);
			$section_fields[] = array(
				'title' => __('Expired OTP Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'The OTP which you have entered got expired.',
				'autoload' => false,
				'id' => $this->get_option_key('fund_transfer_otp_expired_message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_transfer_message_options',
			);

			// Fund Transfer section end.
			// Fund request section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Fund Request Messages', 'wallet-for-woocommerce'),
				'id' => 'wal_fund_request_message_options',
			);
			$section_fields[] = array(
				'title' => __("Message to Display when a Receiver didn't Select the Sender Name", 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please select a sender name',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_empty_receiver_message'),
			);
			$section_fields[] = array(
				'title' => __("Message to Display when a Receiver didn't Enter the Funds", 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter funds to request',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_empty_fund_message'),
			);
			$section_fields[] = array(
				'title' => __('Minimum Amount Required to submit the Fund Request in a Single Transaction', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter an amount more than {minimum_amount}',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_minimum_amount_message'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Amount that can be submitted for Fund Request in a Single Transaction', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please enter an amount less than {maximum_amount}',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_maximum_amount_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when a User has Reached the Maximum Number of Fund Requests Per Day', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'You are allowed to submit fund transfer requests for upto a maximum of {request_count} times per day. Since you have reached the count, you cannot submit new fund requests today.',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_maximum_count_per_day_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when a User has reached the Maximum Number of Unique Fund Requests Per Day', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'You are allowed to submit a fund requests a maximum of {user_count} time(s) to each user per day. Since you have reached the count, you cannot submit the fund request anymore today.',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_maximum_user_count_per_day_message'),
			);
			$section_fields[] = array(
				'title' => __('Message to Display when the User(s)/User Role(s) is Restricted to submit Fund Request', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Currently, you are restricted from submitting fund requests',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_user_restricted_message'),
			);
			$section_fields[] = array(
				'title' => __('Fund Request Success Message', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Funds Requested Successfully',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_success_message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_fund_request_message_options',
			);

			// Fund request section end.
			// Alert section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Alert Messages', 'wallet-for-woocommerce'),
				'id' => 'wal_alert_message_options',
			);
			$section_fields[] = array(
				'title' => __('Request', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Are you sure you want to sumbit the request?',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_alert_message'),
			);
			$section_fields[] = array(
				'title' => __('Transfer', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Are you sure you want to accept this request by transferring the funds?',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_accept_alert_message'),
			);
			$section_fields[] = array(
				'title' => __('Cancel', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Are you sure you want to cancel this request?',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_cancel_alert_message'),
			);
			$section_fields[] = array(
				'title' => __('Decline', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Are you sure you want to decline this request?',
				'autoload' => false,
				'id' => $this->get_option_key('fund_request_decline_alert_message'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_alert_message_options',
			);

			// Alert section end.

			return $section_fields;
		}

		/**
		 * Include files.
		 */
		protected function include_files() {
			include_once WAL_ABSPATH . 'inc/entity/class-wal-fund-transfer.php' ;
			include_once WAL_ABSPATH . 'inc/entity/class-wal-fund-transfer-log.php' ;
		}

		/**
		 * Actions.
		 */
		protected function actions() {

			//Add the email classes.
			add_filter('wal_wallet_email_classes', array( $this, 'email_classes' ), 10, 1);
			//Validate User for fund request.
			add_action('wp_ajax_wal_user_validation_for_transfer', array( $this, 'user_validation_for_transfer' ));
			//Validate User for fund request.
			add_action('wp_ajax_wal_user_validation_for_request', array( $this, 'user_validation_for_request' ));
			//Accept the fund request.
			add_action('wp_ajax_wal_accept_fund_request', array( $this, 'accept_fund_request' ));
			//Decline the fund request.
			add_action('wp_ajax_wal_decline_fund_request', array( $this, 'decline_fund_request' ));
			//Cancel the fund request.
			add_action('wp_ajax_wal_cancel_fund_request', array( $this, 'cancel_fund_request' ));
			//Transaction logs for a selected page.
			add_action('wp_ajax_wal_fund_transfer_transaction_logs_pagination', array( $this, 'transaction_logs_pagination' ));
		}

		/**
		 * Front end actions.
		 */
		protected function frontend_action() {
			//Add the wallet endpoints.
			add_filter('wal_wallet_endpoints', array( $this, 'add_endpoints' ), 10, 1);
			//Add the dashboard menus.
			add_filter('wal_dashboard_menus', array( $this, 'add_dashboard_menus' ), 10, 1);
			//Add the hook for transaction menu content.
			add_action('wal_dashboard_menu_content_transactions', array( $this, 'transactions_menu_content' ));
			//Add the hook for fund transfer menu content.
			add_action('wal_dashboard_menu_content_fund_transfer', array( $this, 'fund_transfer_menu_content' ));
			//Add the hook for fund request menu content.
			add_action('wal_dashboard_menu_content_fund_request', array( $this, 'fund_request_menu_content' ));
			//Add the fund transfer overview data.
			add_filter('wal_wallet_overview_data', array( $this, 'add_overview_data' ), 10, 1);
			//Process the fund transfer.
			add_action('wp_loaded', array( $this, 'process_fund_transfer' ));
			//Process the fund request.
			add_action('wp_loaded', array( $this, 'process_fund_request' ));
		}

		/**
		 * Enqueue admin end JS files.
		 */
		protected function admin_external_js_files() {
			// Fund transfer.   
			wp_enqueue_script('wal-fund-transfer', WAL_PLUGIN_URL . '/assets/js/admin/fund-transfer.js', array( 'jquery', 'jquery-blockui' ), WAL_VERSION);
		}

		/**
		 * Enqueue front end JS files.
		 */
		protected function frontend_external_js_files() {
			// Frontend.
			wp_enqueue_script('wal-fund-transfer', WAL_PLUGIN_URL . '/assets/js/frontend/fund-transfer.js', array( 'jquery', 'jquery-blockui', 'accounting' ), WAL_VERSION, WAL_Fronend_Assets::$in_footer);

			wp_localize_script(
					'wal-fund-transfer', 'wal_fund_transfer_params', array(
				'user_nonce' => wp_create_nonce('wal-user-nonce'),
				'fund_transfer_nonce' => wp_create_nonce('wal-fund-transfer-nonce'),
				'accept_alert_msg' => get_option('wal_module_fund_request_accept_alert_message'),
				'decline_alert_msg' => get_option('wal_module_fund_request_decline_alert_message'),
				'cancel_alert_msg' => get_option('wal_module_fund_request_cancel_alert_message'),
				'request_alert_msg' => get_option('wal_module_fund_request_alert_message'),
				'fee_enabled' => get_option('wal_module_fund_transfer_fee_enabled'),
				'fee_type' => get_option('wal_module_fund_transfer_fee_type'),
				'fee_value' => get_option('wal_module_fund_transfer_fee_amount'),
					)
			);
		}

		/**
		 * Add the custom email classes.
		 * 
		 * @return array
		 */
		public function email_classes( $classes ) {
			$classes = ( wal_check_is_array($classes) ) ? $classes : array();

			$classes['customer-fund-transfer-otp'] = 'WAL_Customer_Fund_Transfer_OTP_Notification';
			$classes['customer-fund-transferred'] = 'WAL_Customer_Fund_Transferred_Notification';
			$classes['customer-fund-received'] = 'WAL_Customer_Fund_Received_Notification';
			$classes['customer-fund-request-submitted'] = 'WAL_Customer_Fund_Request_Submitted_Notification';
			$classes['customer-fund-request-received'] = 'WAL_Customer_Fund_Request_Received_Notification';
			$classes['customer-fund-request-cancelled'] = 'WAL_Customer_Fund_Request_Cancelled_Notification';
			$classes['customer-fund-request-approved'] = 'WAL_Customer_Fund_Request_Approved_Notification';
			$classes['customer-fund-request-declined'] = 'WAL_Customer_Fund_Request_Declined_Notification';

			return $classes;
		}

		/**
		 * Add the custom wallet endpoints.
		 * 
		 * @return array
		 */
		public function add_endpoints( $endpoints ) {
			$endpoints = ( wal_check_is_array($endpoints) ) ? $endpoints : array();

			$endpoints['fund_transfer_details'] = wal_get_fund_transfer_details_menu_endpoint();
			$endpoints['transactions'] = wal_get_fund_transfer_transaction_menu_endpoint();
			$endpoints['fund_transfer'] = wal_get_fund_transfer_menu_endpoint();
			$endpoints['fund_request'] = wal_get_fund_request_menu_endpoint();

			return $endpoints;
		}

		/**
		 * Add the custom dashboard menus.
		 * 
		 * @return array
		 */
		public function add_dashboard_menus( $menus ) {
			$menus = ( wal_check_is_array($menus) ) ? $menus : array();

			$submenus = array(
				'transactions' => array(
					'name' => wal_get_fund_transfer_transaction_menu_label(),
					'endpoint' => wal_get_fund_transfer_transaction_menu_endpoint(),
			),
			);

			if (wal_fund_transfer_enabled()) {
				$submenus['fund_transfer'] = array(
					'name' => wal_get_fund_transfer_menu_label(),
					'endpoint' => wal_get_fund_transfer_menu_endpoint(),
				);
			}

			if (wal_fund_request_enabled()) {
				$submenus['fund_request'] = array(
					'name' => wal_get_fund_request_menu_label(),
					'endpoint' => wal_get_fund_request_menu_endpoint(),
				);
			}

			$menus['fund_transfer_details'] = array(
				'name' => wal_get_fund_transfer_details_menu_label(),
				'endpoint' => wal_get_fund_transfer_details_menu_endpoint(),
				'submenus' => $submenus,
			);

			return $menus;
		}

		/**
		 * Add the fund transfer overview data.
		 * 
		 * @return array
		 */
		public function add_overview_data( $data ) {
			$data['fund_transferred_total'] = array(
				'label' => wal_get_fund_transferred_total_label(),
				'value' => wal_convert_price_by_currency(WAL_Current_User_Wallet::get_transferred_total()),
			);

			$data['fund_received_total'] = array(
				'label' => wal_get_fund_received_total_label(),
				'value' => wal_convert_price_by_currency(WAL_Current_User_Wallet::get_received_total()),
			);

			return $data;
		}

		/**
		 * Display the dashboard transaction menu content.
		 * 
		 * @return array
		 */
		public function transactions_menu_content() {
			$transfer_id = isset($_REQUEST['wal_view_transfer']) ? absint($_REQUEST['wal_view_transfer']) : '';
			if (!empty($transfer_id)) {

				$args = array(
					'transaction_log_ids' => wal_get_fund_transfer_log_ids($transfer_id),
					'fund_transfer' => wal_get_fund_transfer($transfer_id),
				);

				wal_get_template('modules/fund-transfer/view-transactions.php', $args);
			} else {
				$fund_transfer_ids = wal_user_fund_transfers();
				$post_per_page = intval(wal_fund_transfers_pagination_count());

				$current_page = 1;
				$offset = ( $post_per_page * $current_page ) - $post_per_page;
				$page_count = ceil(count($fund_transfer_ids) / $post_per_page);

				$table_args = array(
					'fund_transfer_ids' => array_slice($fund_transfer_ids, $offset, $post_per_page),
					'pagination' => array(
						'page_count' => $page_count,
						'current_page' => $current_page,
						'prev_page_count' => ( ( $current_page - 1 ) == 0 ) ? ( $current_page ) : ( $current_page - 1 ),
						'next_page_count' => ( ( $current_page + 1 ) <= ( $page_count ) ) ? ( $current_page + 1 ) : ( $current_page ),
				),
				);

				wal_get_template('modules/fund-transfer/transaction-wrapper.php', $table_args);
			}
		}

		/**
		 * Validate the user.
		 * 
		 * @return void
		 */
		public function user_validation_for_transfer() {
			check_ajax_referer('wal-user-nonce', 'wal_security');

			try {

				if (!isset($_POST)) {
					throw new exception(__('Invalid Username/Email', 'wallet-for-woocommerce'));
				}

				// Sanitize post values.
				$user_name = !empty($_POST['user_name']) ? wp_unslash(wc_clean($_POST['user_name'])) : '';
				if ( ! isset( $user_name ) ) {
					throw new exception(__('Invalid Username/Email', 'wallet-for-woocommerce'));
				}

				$user_ids = get_users(
					array(
						'fields' => 'ID',
						'search' => $user_name,
						'search_columns' => array( 'user_login', 'user_nicename', 'user_email' ),
					)
				);

				if ( ! wal_check_is_array($user_ids) ) {
					throw new exception(__('Invalid Username/Email', 'wallet-for-woocommerce'));
				}

				$user_id = reset($user_ids);
				if ( get_current_user_id() == $user_id ) {
					throw new exception(__('You should not enter your name/email to transfer', 'wallet-for-woocommerce'));
				}

				wp_send_json_success(array( 'user_id' => $user_id ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Validate the user.
		 * 
		 * @return void
		 */
		public function user_validation_for_request() {
			check_ajax_referer('wal-user-nonce', 'wal_security');

			try {

				if (!isset($_POST)) {
					throw new exception(__('Invalid Username/Email', 'wallet-for-woocommerce'));
				}

				// Sanitize post values.
				$user_name = !empty($_POST['user_name']) ? wp_unslash(wc_clean($_POST['user_name'])) : '';
				if ( ! isset( $user_name ) ) {
					throw new exception(__('Invalid Username/Email', 'wallet-for-woocommerce'));
				}

				$user_ids = get_users(
					array(
						'fields' => 'ID',
						'search' => $user_name,
						'search_columns' => array( 'user_login', 'user_nicename', 'user_email' ),
					)
				);

				if ( ! wal_check_is_array($user_ids) ) {
					throw new exception(__('Invalid Username/Email', 'wallet-for-woocommerce'));
				}

				$user_id = reset($user_ids);
				if ( get_current_user_id() == $user_id ) {
					throw new exception(__('You should not enter your name/email to request', 'wallet-for-woocommerce'));
				}

				wp_send_json_success(array( 'user_id' => $user_id ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Accept the fund request.
		 * 
		 * @return void
		 */
		public function accept_fund_request() {
			check_ajax_referer('wal-fund-transfer-nonce', 'wal_security');

			try {

				if (!isset($_POST)) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				// Sanitize post values.
				$transaction_id = !empty($_POST['transaction_id']) ? absint($_POST['transaction_id']) : 0;
				if (!isset($transaction_id)) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				$transaction_log = wal_get_fund_transfer_log($transaction_id);
				if (!$transaction_log->exists()) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				if (!$transaction_log->has_status('wal_request_received')) {
					throw new exception(__('You have already completed the action successfully.', 'wallet-for-woocommerce'));
				}

				$fee = $this->get_fee_amount($transaction_log->get_amount());                               
				// Validate the transfer amount greater than user balance.
				if (WAL_Current_User_Wallet::get_balance() < ( $transaction_log->get_amount() + $fee )) {
					throw new Exception(get_option('wal_module_fund_transfer_insufficient_wallet_fund_message'));
				}
							   
				$args = array(
					'sender_id' => $transaction_log->get_sender_id(),
					'receiver_id' => absint($transaction_log->get_receiver_id()),
					'amount' => $transaction_log->get_amount(),
					'fee' => $fee,
					'sender_log_id' => absint($transaction_log->get_id()),
					'receiver_log_id' => absint($transaction_log->get_receiver_log_id()),
				);
							
				// Accept the fund request.
				wal_accept_fund_request($args);

				$updated_transaction_log = wal_get_fund_transfer_log($transaction_id);

				// Get the fund transfer transaction log content.
				$html = wal_get_template_html(
						'modules/fund-transfer/transaction-details.php', array(
					'transaction_log' => $updated_transaction_log,
						)
				);

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Decline the fund request.
		 * 
		 * @return void
		 */
		public function decline_fund_request() {
			check_ajax_referer('wal-fund-transfer-nonce', 'wal_security');

			try {

				if (!isset($_POST)) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				// Sanitize post values.
				$transaction_id = !empty($_POST['transaction_id']) ? absint($_POST['transaction_id']) : 0;
				if (!isset($transaction_id)) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				$transaction_log = wal_get_fund_transfer_log($transaction_id);
				if (!$transaction_log->exists()) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				if (!$transaction_log->has_status('wal_request_received')) {
					throw new exception(__('You have already completed the action successfully.', 'wallet-for-woocommerce'));
				}

				$args = array(
					'sender_id' => $transaction_log->get_sender_id(),
					'receiver_id' => absint($transaction_log->get_receiver_id()),
					'amount' => $transaction_log->get_amount(),
					'fee' => $this->get_fee_amount($transaction_log->get_amount()),
					'sender_log_id' => absint($transaction_log->get_id()),
					'receiver_log_id' => absint($transaction_log->get_receiver_log_id()),
				);

				// Decline the fund request.
				wal_decline_fund_request($args);

				$updated_transaction_log = wal_get_fund_transfer_log($transaction_id);

				// Get the fund transfer transaction log content.
				$html = wal_get_template_html(
						'modules/fund-transfer/transaction-details.php', array(
					'transaction_log' => $updated_transaction_log,
						)
				);

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Cancel the fund request.
		 * 
		 * @return void
		 */
		public function cancel_fund_request() {
			check_ajax_referer('wal-fund-transfer-nonce', 'wal_security');

			try {

				if (!isset($_POST)) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				// Sanitize post values.
				$transaction_id = !empty($_POST['transaction_id']) ? absint($_POST['transaction_id']) : 0;
				if (!isset($transaction_id)) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				$transaction_log = wal_get_fund_transfer_log($transaction_id);
				if (!$transaction_log->exists()) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				if (!$transaction_log->has_status('wal_requested')) {
					throw new exception(__('You have already completed the action successfully.', 'wallet-for-woocommerce'));
				}

				$args = array(
					'sender_id' => $transaction_log->get_sender_id(),
					'receiver_id' => absint($transaction_log->get_receiver_id()),
					'amount' => $transaction_log->get_amount(),
					'fee' => $this->get_fee_amount($transaction_log->get_amount()),
					'sender_log_id' => absint($transaction_log->get_id()),
					'receiver_log_id' => absint($transaction_log->get_receiver_log_id()),
				);

				// Cancel the fund request.
				wal_cancel_fund_request($args);

				$updated_transaction_log = wal_get_fund_transfer_log($transaction_id);

				// Get the fund transfer transaction log content.
				$html = wal_get_template_html(
						'modules/fund-transfer/transaction-details.php', array(
					'transaction_log' => $updated_transaction_log,
						)
				);

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Transaction logs for a selected page.
		 * 
		 * @return void
		 */
		public function transaction_logs_pagination() {
			check_ajax_referer('wal-pagination-nonce', 'wal_security');

			try {

				if (!isset($_POST) || !isset($_POST['page_number'])) {
					throw new exception(__('Cannot process action', 'wallet-for-woocommerce'));
				}

				// Sanitize post values.
				$current_page = !empty($_POST['page_number']) ? absint($_POST['page_number']) : 0; // @codingStandardsIgnoreLine.

				$per_page = wal_fund_transfers_pagination_count();
				$offset = ( $current_page - 1 ) * $per_page;

				// Get the transaction logs based on per page count.
				$fund_transfer_ids = wal_user_fund_transfers();
				$fund_transfer_ids = array_slice($fund_transfer_ids, $offset, $per_page);

				// Get the transaction logs content.
				$html = wal_get_template_html(
						'modules/fund-transfer/transaction.php', array(
					'fund_transfer_ids' => $fund_transfer_ids,
					'columns' => wal_fund_transfer_transaction_columns(),
						)
				);

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Display the dashboard fund transfer menu content.
		 * 
		 * @return array
		 */
		public function fund_transfer_menu_content() {
			global $wal_confirm_action;

			$valid = $this->is_valid_user();

			if ($wal_confirm_action && $valid) {
				$user_id = isset($_REQUEST['wal_fund_transfer_receiver_id']) ? wc_clean(wp_unslash($_REQUEST['wal_fund_transfer_receiver_id'])) : array();
				$amount = isset($_REQUEST['wal_fund_transfer_amount']) ? wc_clean(wp_unslash($_REQUEST['wal_fund_transfer_amount'])) : 0;
				$reason = isset($_REQUEST['wal_fund_transfer_reason']) ? wc_clean(wp_unslash($_REQUEST['wal_fund_transfer_reason'])) : '';

				$args = array(
					'user_id' => reset($user_id),
					'amount' => $amount,
					'reason' => $reason,
					'fee' => $this->get_fee_amount($amount),
				);

				wal_get_template('modules/fund-transfer/fund-transfer-confirmation.php', $args);
			} else {

				if (!$valid) {
					WAL_Dashboard_Handler::add_error(get_option('wal_module_fund_transfer_user_restricted_message'));
				}
				
				wp_enqueue_script('select2');
				wp_enqueue_style('select2');
				wp_enqueue_script('wal-enhanced');

				$get_user_id = isset($_GET['wal_receiver_id']) ? absint($_GET['wal_receiver_id']) : '';
				$post_user_id = isset($_REQUEST['wal_fund_transfer_receiver_id']) ? wc_clean(wp_unslash($_REQUEST['wal_fund_transfer_receiver_id'])) : array( $get_user_id );
				$amount = isset($_REQUEST['wal_fund_transfer_amount']) ? wc_clean(wp_unslash($_REQUEST['wal_fund_transfer_amount'])) : '';
				$reason = isset($_REQUEST['wal_fund_transfer_reason']) ? wc_clean(wp_unslash($_REQUEST['wal_fund_transfer_reason'])) : '';

				$user_restriction = $this->prepare_user_restriction();

				$args = array(
					'show' => $valid,
					'user_id' => reset($post_user_id),
					'amount' => $amount,
					'reason' => $reason,
					'fee' => $this->get_fee_amount($amount),
					'show_user_search' => !isset($_GET['wal_receiver_id']),
					'include_users' => $user_restriction['include_users'],
					'include_user_roles' => $user_restriction['include_user_roles'],
					'exclude_users' => $user_restriction['exclude_users'],
					'exclude_user_roles' => $user_restriction['exclude_user_roles'],
				);

				wal_get_template('modules/fund-transfer/fund-transfer-form.php', $args);
			}
		}

		/**
		 * Display the dashboard fund request menu content.
		 * 
		 * @return array
		 */
		public function fund_request_menu_content() {
			$valid = $this->is_valid_user();
			if (!$valid) {
				WAL_Dashboard_Handler::add_error(get_option('wal_module_fund_request_user_restricted_message'));
			}

			wp_enqueue_script('select2');
			wp_enqueue_style('select2');
			wp_enqueue_script('wal-enhanced');

			$get_user_id = isset($_GET['wal_receiver_id']) ? absint($_GET['wal_receiver_id']) : '';
			$post_user_id = isset($_REQUEST['wal_fund_request_receiver_id']) ? wc_clean(wp_unslash($_REQUEST['wal_fund_request_receiver_id'])) : array( $get_user_id );
			$amount = isset($_REQUEST['wal_fund_request_amount']) ? wc_clean(wp_unslash($_REQUEST['wal_fund_request_amount'])) : '';
			$reason = isset($_REQUEST['wal_fund_request_reason']) ? wc_clean(wp_unslash($_REQUEST['wal_fund_request_reason'])) : '';

			$user_restriction = $this->prepare_user_restriction();

			$args = array(
				'show' => $valid,
				'user_id' => reset($post_user_id),
				'amount' => $amount,
				'reason' => $reason,
				'show_user_search' => !isset($_GET['wal_receiver_id']),
				'include_users' => $user_restriction['include_users'],
				'include_user_roles' => $user_restriction['include_user_roles'],
				'exclude_users' => $user_restriction['exclude_users'],
				'exclude_user_roles' => $user_restriction['exclude_user_roles'],
			);

			wal_get_template('modules/fund-transfer/fund-request-form.php', $args);
		}

		/**
		 * Get the fund transfer fee amount.
		 * 
		 * @return float
		 */
		public function get_fee_amount( $amount ) {
			$fee = 0;
			// Prepare transfer fee.
			if (wal_fund_transfer_fee_enabled()) {
				$fee_amount = floatval(get_option('wal_module_fund_transfer_fee_amount'));
				if ('2' == get_option('wal_module_fund_transfer_fee_type')) {
					$fee = ( $fee_amount ) ? ( $fee_amount / 100 ) * floatval($amount) : $fee;
				} else {
					$fee = $fee_amount;
				}
			}
			/**
			 * This hook is used to alter the fund transfer fee amount.
			 * 
			 * @since 1.0
			 */
			return apply_filters('wal_fund_transfer_fee_amount', $fee);
		}

		/**
		 * Prepare the user restriction.
		 * 
		 * @return array
		 */
		public function prepare_user_restriction() {
			$user_restriction = array(
				'include_users' => array(),
				'include_user_roles' => array(),
				'exclude_users' => array( WAL_Current_User_Wallet::get_user_id() ),
				'exclude_user_roles' => array(),
			);

			$type = get_option('wal_module_fund_transfer_user_restriction_type');
			switch ($type) {
				case '2':
					$include_users = array_filter((array) get_option('wal_module_fund_transfer_include_users'), 'wal_array_filter');
					$user_restriction['include_users'] = array_merge($user_restriction['include_users'], $include_users);
					break;
				case '3':
					$exclude_users = array_filter((array) get_option('wal_module_fund_transfer_exclude_users'), 'wal_array_filter');
					$user_restriction['exclude_users'] = array_merge($user_restriction['exclude_users'], $exclude_users);
					break;
				case '4':
					$include_user_roles = array_filter((array) get_option('wal_module_fund_transfer_include_user_roles'), 'wal_array_filter');
					$user_restriction['include_user_roles'] = array_merge($user_restriction['include_user_roles'], $include_user_roles);
					break;
				case '5':
					$exclude_user_roles = array_filter((array) get_option('wal_module_fund_transfer_exclude_user_roles'), 'wal_array_filter');
					$user_restriction['exclude_user_roles'] = array_merge($user_restriction['exclude_user_roles'], $exclude_user_roles);
					break;
			}

			return $user_restriction;
		}

		/**
		 * Process the fund transfer.
		 */
		public function process_fund_transfer() {
			$nonce_value = isset($_POST['wal-fund-transfer-nonce']) ? wc_clean(wp_unslash($_POST['wal-fund-transfer-nonce'])) : null;
			if (!isset($_POST['wal-action']) || empty($_POST['wal-action']) || !wp_verify_nonce($nonce_value, 'wal-fund-transfer')) {
				return;
			}

			try {

				$receiver_id = isset($_POST['wal_fund_transfer_receiver_id']) ? wc_clean(wp_unslash($_POST['wal_fund_transfer_receiver_id'])) : '';
				$receiver_id = wal_check_is_array($receiver_id) ? reset($receiver_id) : false;
				$amount = isset($_POST['wal_fund_transfer_amount']) ? wc_clean(wp_unslash($_POST['wal_fund_transfer_amount'])) : '';
				$reason = isset($_POST['wal_fund_transfer_reason']) ? wc_clean(wp_unslash($_POST['wal_fund_transfer_reason'])) : '';
							   
				$amount = wal_convert_price($amount, true);                                                                
				// Validate the fund transfer form
				$this->validate_fund_transfer_form($receiver_id, $amount);

				global $wal_confirm_action;
				$transfer_action = isset($_POST['wal-action']) ? wc_clean(wp_unslash($_POST['wal-action'])) : '';
				
				$wal_confirm_action = true;
								
				if ('confirm' == $transfer_action) {
					// Verify the OTP.
					$this->verify_otp();

					$args = array(
						'sender_id' => WAL_Current_User_Wallet::get_user_id(),
						'receiver_id' => $receiver_id,
						'amount' => (float) $amount,
						'fee' => $this->get_fee_amount($amount),
						'reason' => $reason,
					);
									
					//Transfer amount to receiver.
					wal_user_fund_transfer($args);

					WAL_Dashboard_Handler::add_message(get_option('wal_module_fund_transfer_success_message'));

					$wal_confirm_action = false;

					$keys = array(
						'wal-fund-transfer-nonce',
						'wal_fund_transfer',
						'wal_receiver_id',
						'wal_fund_transfer_amount',
						'wal_fund_transfer_reason',
					);

					wal_unset_global_variable_values($keys);

					//Reset wallet.
					WAL_Current_User_Wallet::reset();
				}
				/**
				 * This hook is used to do extra action after fund transferred.
				 * 
				 * @since 1.0
				 */
				do_action('wal_processed_fund_transfer', $receiver_id, $amount, $transfer_action);
			} catch (Exception $ex) {

				WAL_Dashboard_Handler::add_error($ex->getMessage());
			}
		}

		/**
		 * Validate the fund transfer form.
		 * 
		 * @return bool
		 */
		public function validate_fund_transfer_form( $receiver_id, $amount ) {
			//Validate the receiver user ID exists.
			if (!$receiver_id) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_transfer_empty_receiver_message')));
			}

			//Validate the transfer amount exists.
			if (empty($amount)) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_transfer_empty_fund_message')));
			}

			$fee = $this->get_fee_amount($amount);                       
			//Validate the user.
			if (!$this->is_valid_user()) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_transfer_user_restricted_message')));
			}
						
						
			// Validate the fund is less than minimum amount.
			$minimum_amount = floatval(get_option('wal_module_fund_transfer_min_amount'));                        
			if ($minimum_amount && $minimum_amount > wal_convert_price($amount , true)) {
				/* translators: %s minimum amount */
				throw new exception(wp_kses_post(str_replace('{minimum_amount}', wal_convert_price_by_currency($minimum_amount), get_option('wal_module_fund_transfer_minimum_amount_message'))));
			}
						
			// Validate the fund is greater than maximum amount.
			$maximum_amount = floatval(get_option('wal_module_fund_transfer_max_amount'));
			if ($maximum_amount && $maximum_amount < wal_convert_price($amount , true )) {
				/* translators: %s maximum amount */
				throw new exception(wp_kses_post(str_replace('{maximum_amount}', wal_convert_price_by_currency($maximum_amount), get_option('wal_module_fund_transfer_maximum_amount_message'))));
			}

			// Validate the transfer amount greater than user balance.
			if ((float) WAL_Current_User_Wallet::get_balance() < ( $amount + $fee )) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_transfer_insufficient_wallet_fund_message')));
			}

			//Validate the wallet transfer count has reached the per day count.
			$max_count_per_day = floatval(get_option('wal_module_fund_transfer_count_per_day'));
			if ($max_count_per_day && $max_count_per_day <= wal_get_user_fund_transfered_count()) {
				/* translators: %s maximum wallet count */
				throw new exception(wp_kses_post(str_replace('{transfer_count}', $max_count_per_day, get_option('wal_module_fund_transfer_maximum_count_per_day_message'))));
			}

			//Validate the wallet transfer count has reached the per day count to a specifc user.
			$max_user_count_per_day = floatval(get_option('wal_module_fund_transfer_user_count_per_day'));
			if ($max_user_count_per_day && $max_user_count_per_day <= wal_get_user_fund_transfered_particular_user_count($receiver_id)) {
				/* translators: %s maximum wallet count */
				throw new exception(wp_kses_post(str_replace('{user_count}', $max_user_count_per_day, get_option('wal_module_fund_transfer_maximum_user_count_per_day_message'))));
			}
		}

		/**
		 * Validate the OTP.
		 */
		public function verify_otp() {
			// Return if the OTP is not enabled.
			if (!wal_fund_transfer_otp_enabled()) {
				return;
			}

			$otp = isset($_REQUEST['wal_fund_transfer_verify_otp']) ? wc_clean(wp_unslash($_REQUEST['wal_fund_transfer_verify_otp'])) : '';
			if (!$otp) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_transfer_invalid_otp_message')));
			}

			// Validate the OTP expiry.
			$validity = get_post_meta(WAL_Current_User_Wallet::get_user_id(), 'wal_fund_transfer_otp_validity', true);
			if (WAL_Date_Time::get_date_time_object('now', true) >= WAL_Date_Time::get_date_time_object($validity)) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_transfer_otp_expired_message')));
			}

			// Validate the entered OTP.
			$saved_otp = get_post_meta(WAL_Current_User_Wallet::get_user_id(), 'wal_fund_transfer_otp', true);
			if ($otp != $saved_otp) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_transfer_invalid_otp_message')));
			}
		}

		/**
		 * Process the fund request.
		 */
		public function process_fund_request() {
			$nonce_value = isset($_POST['wal-fund-request-nonce']) ? wc_clean(wp_unslash($_POST['wal-fund-request-nonce'])) : null;
			if (!isset($_POST['wal-action']) || empty($_POST['wal-action']) || !wp_verify_nonce($nonce_value, 'wal-fund-request')) {
				return;
			}

			try {

				$receiver_id = isset($_POST['wal_fund_request_receiver_id']) ? wc_clean(wp_unslash($_POST['wal_fund_request_receiver_id'])) : '';
				$receiver_id = wal_check_is_array($receiver_id) ? reset($receiver_id) : false;
				$amount = isset($_POST['wal_fund_request_amount']) ? wc_clean(wp_unslash($_POST['wal_fund_request_amount'])) : '';
				$reason = isset($_POST['wal_fund_request_reason']) ? wc_clean(wp_unslash($_POST['wal_fund_request_reason'])) : '';

				$amount = wal_convert_price($amount, true); 
				// Validate the fund request form
				$this->validate_fund_request_form($receiver_id, $amount);

				$args = array(
					'sender_id' => WAL_Current_User_Wallet::get_user_id(),
					'receiver_id' => $receiver_id,
					'amount' => $amount,
					'reason' => $reason,
				);

				//Request amount from a receiver.
				wal_user_fund_request($args);

				WAL_Dashboard_Handler::add_message(get_option('wal_module_fund_request_success_message'));

				// Unset the form values.
				$keys = array(
					'wal-fund-request-nonce',
					'wal_fund_request',
					'wal_receiver_id',
					'wal_fund_request_amount',
					'wal_fund_request_reason',
				);

				wal_unset_global_variable_values($keys);
			} catch (Exception $ex) {

				WAL_Dashboard_Handler::add_error($ex->getMessage());
			}
		}

		/**
		 * Validate the fund request form.
		 * 
		 * @return bool
		 */
		public function validate_fund_request_form( $receiver_id, $amount ) {
			//Validate the receiver user ID exists.
			if (!$receiver_id) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_request_empty_receiver_message')));
			}

			//Validate the request amount exists.
			if (empty($amount)) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_request_empty_fund_message')));
			}

			$amount = wal_convert_price(floatval($amount), true);
			//Validate the user.
			if (!$this->is_valid_user()) {
				throw new Exception(wp_kses_post(get_option('wal_module_fund_request_user_restricted_message')));
			}

			// Validate the fund is less than minimum amount.
			$minimum_amount = floatval(get_option('wal_module_fund_request_min_amount'));
			if ($minimum_amount && $minimum_amount > $amount) {
				/* translators: %s minimum amount */
				throw new exception(wp_kses_post(str_replace('{minimum_amount}', wal_convert_price_by_currency($minimum_amount), get_option('wal_module_fund_request_minimum_amount_message'))));
			}

			// Validate the fund is greater than maximum amount.
			$maximum_amount = floatval(get_option('wal_module_fund_request_max_amount'));
			if ($maximum_amount && $maximum_amount < $amount) {
				/* translators: %s maximum amount */
				throw new exception(wp_kses_post(str_replace('{maximum_amount}', wal_convert_price_by_currency($maximum_amount), get_option('wal_module_fund_request_maximum_amount_message'))));
			}

			//Validate the wallet request count has reached the per day count.
			$max_count_per_day = floatval(get_option('wal_module_fund_request_count_per_day'));
			if ($max_count_per_day && $max_count_per_day <= wal_get_user_fund_requested_count()) {
				/* translators: %s maximum wallet count */
				throw new exception(wp_kses_post(str_replace('{request_count}', $max_count_per_day, get_option('wal_module_fund_request_maximum_count_per_day_message'))));
			}

			//Validate the wallet request count has reached the per day count to a specifc user.
			$max_user_count_per_day = floatval(get_option('wal_module_fund_request_max_amount_per_day'));
			if ($max_user_count_per_day && $max_user_count_per_day <= wal_get_user_fund_requested_particular_user_count($receiver_id)) {
				/* translators: %s maximum wallet count */
				throw new exception(wp_kses_post(str_replace('{user_count}', $max_user_count_per_day, get_option('wal_module_fund_request_maximum_user_count_per_day_message'))));
			}
		}

		/**
		 * Is a valid user?.
		 * 
		 * @return bool
		 */
		public function is_valid_user() {
			$return = true;

			if ($this->validate_user()) {
				$return = false;
			} elseif ('wal_active' != WAL_Current_User_Wallet::get_status()) {
				$return = false;
			}
			/**
			 * This hook is used to validate the fund transfer for user.
			 * 
			 * @since 1.0
			 */
			return apply_filters('wal_is_valid_fund_transfer_user', $return);
		}

		/**
		 * Validate the user.
		 * 
		 * @return bool
		 */
		protected function validate_user() {
			$return = false;
			$user_type = get_option('wal_module_fund_transfer_user_restriction_type');

			switch ($user_type) {
				case '2':
					$include_users = array_filter((array) get_option('wal_module_fund_transfer_include_users'), 'wal_array_filter');
					if (!in_array(WAL_Current_User_Wallet::get_user_id(), $include_users)) {
						$return = true;
					}

					break;
				case '3':
					$exclude_users = array_filter((array) get_option('wal_module_fund_transfer_exclude_users'), 'wal_array_filter');
					if (in_array(WAL_Current_User_Wallet::get_user_id(), $exclude_users)) {
						$return = true;
					}
					break;
				case '4':
					$return = true;
					$include_user_roles = array_filter((array) get_option('wal_module_fund_transfer_include_user_roles'), 'wal_array_filter');

					// Loggedin user restriction
					if (wal_check_is_array(WAL_Current_User_Wallet::get_user()->roles)) {
						foreach (WAL_Current_User_Wallet::get_user()->roles as $role) {
							if (in_array($role, $include_user_roles)) {
								$return = false;
							}
						}
					}

					break;
				case '5':
					$exclude_user_roles = array_filter((array) get_option('wal_module_fund_transfer_exclude_user_roles'), 'wal_array_filter');

					// Loggedin user restriction
					if (wal_check_is_array(WAL_Current_User_Wallet::get_user()->roles)) {
						foreach (WAL_Current_User_Wallet::get_user()->roles as $role) {
							if (in_array($role, $exclude_user_roles)) {
								$return = true;
							}
						}
					}

					break;
			}
			/**
			 * This hook is used to validate the fund transfer for user.
			 * 
			 * @since 1.0
			 */
			return apply_filters('wal_validate_user_fund_transfer', $return);
		}
	}

}
