<?php

/**
 * Module - Auto Top-up.
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Auto_Topup_Module')) {

	/**
	 * Class.
	 * */
	class WAL_Auto_Topup_Module extends WAL_Modules {

		/**
		 * Top up product ID.
		 * 
		 * @var int.
		 */
		protected $topup_product_id = 0;

		/**
		 * Class Constructor.
		 * */
		public function __construct() {

			$this->id = 'auto_topup';
			$this->title = __('Wallet Auto Top-up', 'wallet-for-woocommerce');
			$this->description = __('Users can Auto top-up their wallet', 'wallet-for-woocommerce');
			$this->tooltip = __('By enabling this checkbox, users can Auto top-up their wallet.', 'wallet-for-woocommerce');

			parent::__construct();
		}

		/**
		 * Get the setting tabs.
		 * 
		 * @return array
		 */
		public function setting_tabs() {
			$tabs = array(
				'auto_topup' => __('Wallet Auto Top-up', 'wallet-for-woocommerce'),
				'general' => __('General', 'wallet-for-woocommerce'),
				'localization' => __('Localization', 'wallet-for-woocommerce'),
				'messages' => __('Messages', 'wallet-for-woocommerce'),
			);
			/**
			 * This hook is used to alter the current module tabs.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->get_plugin_slug() . '_get_module_tabs_' . $this->get_id(), $tabs);
		}

		/**
		 * Show buttons?.
		 * 
		 * @return string
		 */
		public function show_buttons() {
			global $current_tab;

			if (!$current_tab || 'auto_topup' == $current_tab) {
				return false;
			}

			return $this->show_buttons;
		}

		/**
		 * Output the tab content.
		 */
		public function output_tab_content() {
			if (!class_exists('WAL_Auto_Topup_Users_List_Table')) {
				require_once WAL_PLUGIN_PATH . '/inc/admin/menu/wp-list-tables/class-wal-auto-topup-users-list-table.php';
			}

			$post_table = new WAL_Auto_Topup_Users_List_Table();
			$post_table->render();
		}

		/**
		 * Get settings for general section array.
		 * 
		 * @return array
		 */
		protected function general_section_array() {
			$section_fields = array();
			$wp_user_roles = wal_get_wp_user_roles();

			$section_fields[] = array(
				'type' => 'title',
				'title' => __('General Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_general_options',
			);
			$section_fields[] = array(
				'title' => __('Predefined Buttons for Auto Top-up', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_predefined_amounts'),
			);
			$section_fields[] = array(
				'title' => __('Predefined Buttons for Threshold', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_predefined_threshold_amounts'),
				'desc' => __('The plugin will try to top-up the customer\'s wallet automatically if their wallet balance is less than the threshold amount.', 'wallet-for-woocommerce'),
				'desc_tip' => true,
			);
			$section_fields[] = array(
				'title' => __('Display Terms and Conditions Link', 'wallet-for-woocommerce'),
				'type' => 'checkbox',
				'default' => 'no',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_terms_and_conditions'),
				'desc' => __('When enabled, Terms and Conditions link will be displayed to the user.', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Terms and Conditions URL', 'wallet-for-woocommerce'),
				'type' => 'text',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_terms_and_conditions_url'),
			);
			$section_fields[] = array(
				'title' => __('Terms and Conditions Content', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'I agree to the <a href="[wallet_auto_topup_terms]">Terms and conditions</a>',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_terms_and_conditions_content'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_general_options',
			);
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Security Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_auto_topup_security_options',
			);
			$section_fields[] = array(
				'title' => __('Minimum Auto Top-up Amount', 'wallet-for-woocommerce'),
				'type' => 'text',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_min_amount'),
				'desc' => __('The minimum amount which the user can set for Auto Top-Up', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Auto Top-up Amount', 'wallet-for-woocommerce'),
				'type' => 'text',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_max_amount'),
				'desc' => __('The maximum amount which the user can set for Auto Top-Up', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Minimum Threshold Amount', 'wallet-for-woocommerce'),
				'type' => 'text',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_min_threshold_amount'),
				'desc' => __('The minimum amount which the user can set as Wallet Threshold', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'title' => __('Maximum Threshold Amount', 'wallet-for-woocommerce'),
				'type' => 'text',
				'default' => '',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_max_threshold_amount'),
				'desc' => __('The maximum amount which the user can set as Wallet Threshold', 'wallet-for-woocommerce'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_auto_topup_security_options',
			);
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Restriction Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_auto_topup_restriction_options',
			);
			$section_fields[] = array(
				'title' => __('Wallet Auto Top-up Applicable for', 'wallet-for-woocommerce'),
				'type' => 'select',
				'default' => '1',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_user_restriction_type'),
				'class' => 'wal-user-restriction-type',
				'options' => array(
					'1' => __('All Users', 'wallet-for-woocommerce'),
					'2' => __('Include Users', 'wallet-for-woocommerce'),
					'3' => __('Exclude Users', 'wallet-for-woocommerce'),
					'4' => __('Include User Roles', 'wallet-for-woocommerce'),
					'5' => __('Exclude User Roles', 'wallet-for-woocommerce'),
				),
			);
			$section_fields[] = array(
				'title' => __('Select Users', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'ajaxmultiselect',
				'list_type' => 'customers',
				'action' => 'wal_json_search_customers',
				'default' => array(),
				'autoload' => false,
				'placeholder' => __('Select a User', 'wallet-for-woocommerce'),
				'class' => 'wal-user-restriction-field wal-include-users-field',
				'id' => $this->get_option_key('auto_topup_include_users'),
			);
			$section_fields[] = array(
				'title' => __('Select Users', 'wallet-for-woocommerce'),
				'type' => 'wal_custom_fields',
				'wal_field' => 'ajaxmultiselect',
				'list_type' => 'customers',
				'class' => 'wal-user-restriction-field wal-exclude-users-field',
				'action' => 'wal_json_search_customers',
				'default' => array(),
				'autoload' => false,
				'placeholder' => __('Select a User', 'wallet-for-woocommerce'),
				'id' => $this->get_option_key('auto_topup_exclude_users'),
			);
			$section_fields[] = array(
				'title' => __('Select User Roles', 'wallet-for-woocommerce'),
				'type' => 'multiselect',
				'default' => array(),
				'autoload' => false,
				'custom_attributes' => array( 'placeholder' => __('Select a User Role', 'wallet-for-woocommerce') ),
				'class' => 'wal_select2 wal-user-restriction-field wal-include-user-roles-field',
				'id' => $this->get_option_key('auto_topup_include_user_roles'),
				'options' => $wp_user_roles,
			);
			$section_fields[] = array(
				'title' => __('Select User Roles', 'wallet-for-woocommerce'),
				'type' => 'multiselect',
				'default' => array(),
				'autoload' => false,
				'custom_attributes' => array( 'placeholder' => __('Select a User Role', 'wallet-for-woocommerce') ),
				'class' => 'wal_select2 wal-user-restriction-field wal-exclude-user-roles-field',
				'id' => $this->get_option_key('auto_topup_exclude_user_roles'),
				'options' => $wp_user_roles,
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_auto_topup_restriction_options',
			);

			return $section_fields;
		}

		/**
		 * Get settings for localization section array.
		 * 
		 * @return array
		 */
		protected function localization_section_array() {
			$section_fields = array();

			// Fund request section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Endpoints', 'wallet-for-woocommerce'),
				'id' => 'wal_dashboard_endpoints_options',
			);
			$section_fields[] = array(
				'title' => __('Auto Top-up Details Endpoint', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'auto-top-up',
				'autoload' => false,
				'id' => $this->get_option_key('dashboard_auto_topup_endpoint'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_dashboard_endpoints_options',
			);
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Customization', 'wallet-for-woocommerce'),
				'id' => 'wal_dashboard_label_options',
			);
			$section_fields[] = array(
				'title' => __('Auto Top-up Menu Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Auto Top-up',
				'autoload' => false,
				'id' => $this->get_option_key('dashboard_auto_topup_menu_label'),
			);
			$section_fields[] = array(
				'title' => __('Auto Top-up Amount Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Auto Top-up Amount',
				'autoload' => false,
				'id' => $this->get_option_key('dashboard_auto_topup_amount_label'),
			);
			$section_fields[] = array(
				'title' => __('Threshold Value Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Auto Top-up when balance drops below',
				'autoload' => false,
				'id' => $this->get_option_key('dashboard_auto_topup_threshold_amount_label'),
			);
			$section_fields[] = array(
				'title' => __('Top-up and Authorize Button Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Top-up and Authorize',
				'autoload' => false,
				'id' => $this->get_option_key('dashboard_auto_topup_and_authorize_button_label'),
			);
			$section_fields[] = array(
				'title' => __('Authorize Button Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Authorize',
				'autoload' => false,
				'id' => $this->get_option_key('dashboard_auto_topup_authorize_button_label'),
			);
			$section_fields[] = array(
				'title' => __('Authorized Amount Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Authorized Amount: ',
				'autoload' => false,
				'id' => $this->get_option_key('dashboard_authorized_auto_topup_amount_label'),
			);
			$section_fields[] = array(
				'title' => __('Authorized Threshold Value Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Auto Top-up when balance drops below: ',
				'autoload' => false,
				'id' => $this->get_option_key('dashboard_authorized_auto_topup_threshold_amount_label'),
			);
			$section_fields[] = array(
				'title' => __('Cancel Authorization Button Label', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Cancel Authorization',
				'autoload' => false,
				'id' => $this->get_option_key('dashboard_cancel_auto_topup_authorized_button_label'),
			);
			$section_fields[] = array(
				'title' => __('Auto Top-up Successful Notice', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Wallet Auto Top-up Successful.',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_successful_notice'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_dashboard_label_options',
			);

			return $section_fields;
		}

		/**
		 * Get the settings for messages section array.
		 * 
		 * @return array
		 */
		protected function messages_section_array() {
			$section_fields = array();

			// Log section start.
			$section_fields[] = array(
				'type' => 'title',
				'title' => __('Messages Settings', 'wallet-for-woocommerce'),
				'id' => 'wal_messages_options',
			);
			$section_fields[] = array(
				'title' => __('Auto Top-up Authorization Message in Checkout Page', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => '{auto_top_up_amount} will be charged as auto Top-up amount.',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_authorization_message'),
			);
			$section_fields[] = array(
				'title' => __('Auto Top-up Authorization Notice in Checkout Page', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Complete your authorization to Auto Top-up your Wallet in future.',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_authorization_notice'),
			);
			$section_fields[] = array(
				'title' => __('Auto Top-up Authorization Cancelled Success Notice', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'You have successfully cancelled your Auto Top-up authorization.',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_authorization_cancelled_notice'),
			);
			$section_fields[] = array(
				'title' => __('User Not Eligible for Auto Top-up Error Notice', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'You are not eligible to auto top-up.',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_user_not_eligible_notice'),
			);
			$section_fields[] = array(
				'title' => __('Terms and Conditions not Agreed Error Notice', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Please agree to the Terms and Conditions.',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_user_not_agreed_terms_notice'),
			);
			$section_fields[] = array(
				'title' => __('Auto Top-up Amount Required Error Notice', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Auto Top-up Amount is required.',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_amount_required_notice'),
			);
			$section_fields[] = array(
				'title' => __('Threshold Amount Required Error Notice', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Threshold Amount is required.',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_threshold_amount_required_notice'),
			);
			$section_fields[] = array(
				'title' => __('Auto Top-up Amount must Greater than Threshold Amount Error Notice', 'wallet-for-woocommerce'),
				'type' => 'textarea',
				'default' => 'Auto Top-up amount must be greater than Threshold Amount.',
				'autoload' => false,
				'id' => $this->get_option_key('auto_topup_amount_must_gt_than_threshold_amount_notice'),
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id' => 'wal_messages_options',
			);

			// Log section end.
			return $section_fields;
		}

		/**
		 * Include files.
		 */
		protected function include_files() {
			include_once WAL_ABSPATH . 'inc/entity/class-wal-auto-topup.php';
		}

		/**
		 * Actions.
		 */
		protected function actions() {
			add_action('plugins_loaded', array( $this, 'load_payment_gateways' ), 20);
			add_action('woocommerce_order_status_completed', array( $this, 'on_authorization_success' ), -1);
			add_action('woocommerce_order_status_processing', array( $this, 'on_authorization_success' ), -1);
			add_action('wal_wallet_amount_debited', array( $this, 'check_wallet_funds_on_debit' ), 10, 2);
			add_action('wal_wallet_payment_paid_and_authorized_for_auto_topup', array( $this, 'topup_funds' ), -1);
			add_action('wal_wallet_auto_topup_successful', array( $this, 'topup_funds' ), -1);
			add_filter('wal_wallet_email_classes', array( $this, 'email_classes' ), 10, 1);
			add_filter('woocommerce_email_enabled_new_order', array( $this, 'prevent_sending_wc_email' ), 99, 2);
			add_filter('woocommerce_email_enabled_customer_completed_order', array( $this, 'prevent_sending_wc_email' ), 99, 2);
			add_filter('woocommerce_email_enabled_customer_processing_order', array( $this, 'prevent_sending_wc_email' ), 99, 2);
		}

		/**
		 * Admin actions.
		 */
		protected function admin_action() {
			//Get auto topup renewal order details popup content.
			add_action('wp_ajax_wal_get_auto_topup_renewal_order_details_popup_content', array( $this, 'get_auto_topup_renewal_order_details_popup_content' ));
		}

		/**
		 * Front end actions.
		 */
		protected function frontend_action() {
			add_filter('wal_wallet_endpoints', array( $this, 'add_endpoint' ));
			add_filter('wal_dashboard_menus', array( $this, 'add_dashboard_menus' ));
			add_action('wal_dashboard_menu_content_autotopup', array( $this, 'render' ));
			add_filter('woocommerce_cart_total', array( $this, 'get_auto_topup_amount_html' ), 99, 2);
			add_action('woocommerce_check_cart_items', array( $this, 'validate_cart_items' ), 1);
			add_filter('woocommerce_available_payment_gateways', array( $this, 'handle_payment_gateways' ), 11);
			add_action('wp_loaded', array( $this, 'process_auto_topup_form' ));
			add_filter('woocommerce_cart_needs_payment', array( $this, 'needs_payment' ), 99);
			add_action('wal_updated_topup_order_meta', array( $this, 'update_order_meta' ));
		}

		/**
		 * Enqueue admin end JS files.
		 */
		protected function admin_external_js_files() {
			wp_enqueue_script('wal-auto-topup', WAL_PLUGIN_URL . '/assets/js/admin/auto-topup.js', array(), WAL_VERSION);
			wp_localize_script(
					'wal-auto-topup', 'wal_auto_topup_params', array(
				'wal_auto_topup_nonce' => wp_create_nonce('wal-auto-topup'),
					)
			);
		}

		/**
		 * Enqueue front end JS files.
		 */
		protected function frontend_external_js_files() {
			wp_enqueue_script('wal-auto-topup', WAL_PLUGIN_URL . '/assets/js/frontend/auto-topup.js', array(), WAL_VERSION);
		}

		/**
		 * Set the Top-up product ID.
		 * 
		 * @return int
		 */
		public function get_topup_product_id() {
			if (!$this->topup_product_id) {
				$topup_product = array_filter((array) get_option('wal_general_topup_product_id', array()), 'wal_array_filter');

				if (wal_check_is_array($topup_product)) {
					$this->topup_product_id = reset($topup_product);
				}
			}

			return $this->topup_product_id;
		}

		/**
		 * Check whether cart contains Auto Topup.
		 * 
		 * @return bool
		 */
		public function cart_contains_auto_topup() {
			if (is_null(WC()->cart)) {
				return false;
			}

			foreach (WC()->cart->cart_contents as $cart_item) {
				if (isset($cart_item['wal_topup']['mode']) && 'auto' === $cart_item['wal_topup']['mode']) {
					return true;
				}
			}

			return false;
		}

		/**
		 * Get Auto Topup from cart.
		 * 
		 * @return bool
		 */
		public function get_auto_topup_from_cart() {
			if (is_null(WC()->cart)) {
				return false;
			}

			foreach (WC()->cart->cart_contents as $cart_item) {
				if (isset($cart_item['wal_topup']) && 'auto' === $cart_item['wal_topup']['mode']) {
					return $cart_item['wal_topup'];
				}
			}

			return false;
		}

		/**
		 * Validate the user restriction.
		 * 
		 * @return bool
		 */
		public function validate_user_restriction() {
			$return = false;
			$user_type = get_option('wal_module_auto_topup_user_restriction_type');

			switch ($user_type) {
				case '2':
					$include_users = array_filter((array) get_option('wal_module_auto_topup_include_users'), 'wal_array_filter');
					if (!in_array(get_current_user_id(), $include_users)) {
						$return = true;
					}
					break;
				case '3':
					$exclude_users = array_filter((array) get_option('wal_module_auto_topup_exclude_users'), 'wal_array_filter');
					if (in_array(get_current_user_id(), $exclude_users)) {
						$return = true;
					}
					break;
				case '4':
					$return = true;
					$user = wp_get_current_user();
					$include_user_roles = array_filter((array) get_option('wal_module_auto_topup_include_user_roles'), 'wal_array_filter');

					// Loggedin user restriction
					if (wal_check_is_array($user->roles)) {
						foreach ($user->roles as $role) {
							if (in_array($role, $include_user_roles)) {
								$return = false;
							}
						}
					}
					break;
				case '5':
					$user = wp_get_current_user();
					$exclude_user_roles = array_filter((array) get_option('wal_module_auto_topup_exclude_user_roles'), 'wal_array_filter');

					// Loggedin user restriction
					if (wal_check_is_array($user->roles)) {
						foreach ($user->roles as $role) {
							if (in_array($role, $exclude_user_roles)) {
								$return = true;
							}
						}
					}

					break;
			}
			/**
			 * This hook is used to validate the user auto top-up.
			 * 
			 * @since 1.0
			 */
			return apply_filters('wal_validate_user_auto_topup', $return);
		}

		/**
		 * Validate the current user.
		 * 
		 * @return boolean
		 */
		public function is_invalid_user() {
			// Return if the wallet is in active.
			if ('wal_inactive' == WAL_Current_User_Wallet::get_status() || $this->validate_user_restriction()) {
				return true;
			}

			return false;
		}

		/**
		 * Add auto topup endpoint.
		 * 
		 * @param array $endpoints
		 * @return array
		 */
		public function add_endpoint( $endpoints ) {
			$endpoints = ( wal_check_is_array($endpoints) ) ? $endpoints : array();
			$endpoints['autotopup'] = get_option('wal_module_dashboard_auto_topup_endpoint');
			return $endpoints;
		}

		/**
		 * Add dashboard menus.
		 * 
		 * @param array $menus
		 * @return array
		 */
		public function add_dashboard_menus( $menus ) {
			$menus = ( wal_check_is_array($menus) ) ? $menus : array();
			$menus['autotopup'] = array(
				'name' => get_option('wal_module_dashboard_auto_topup_menu_label'),
				'endpoint' => get_option('wal_module_dashboard_auto_topup_endpoint'),
			);
			return $menus;
		}

		/**
		 * Display the dashboard auto topup menu content.
		 * 
		 * @return void
		 */
		public function render() {
			$show = true;

			if ($this->is_invalid_user()) {
				$show = false;
				WAL_Dashboard_Handler::add_error(esc_html(get_option('wal_module_auto_topup_user_not_eligible_notice')));
			}

			$authorized = false;
			$auto_topup = false;

			if ($show && WAL_Current_User_Wallet::has_user_wallet()) {
				if (WAL_Current_User_Wallet::get_wallet()->has_status('wal_active')) {
					$auto_topup_id = wal_get_auto_topup_id_by_wallet_id(WAL_Current_User_Wallet::get_wallet_id(), 'wal_active');
					$auto_topup = wal_get_auto_topup($auto_topup_id);
					$authorized = $auto_topup->has_status('wal_active');
				} else {
					$show = false;
				}
			}

			$topup_predefined_amounts = array_map('floatval', array_filter(explode(',', get_option('wal_module_auto_topup_predefined_amounts')), 'wal_array_filter'));
			$threshold_predefined_amounts = array_map('floatval', array_filter(explode(',', get_option('wal_module_auto_topup_predefined_threshold_amounts')), 'wal_array_filter'));

			sort($topup_predefined_amounts);
			sort($threshold_predefined_amounts);

			$args = array(
				'show' => $show,
				'authorized' => $authorized,
				'auto_topup' => $auto_topup,
				'topup_predefined_amounts' => $topup_predefined_amounts,
				'threshold_predefined_amounts' => $threshold_predefined_amounts,
				'topup_min_amount' => get_option('wal_module_auto_topup_min_amount'),
				'topup_max_amount' => get_option('wal_module_auto_topup_max_amount'),
				'min_threshold_amount' => get_option('wal_module_auto_topup_min_threshold_amount'),
				'max_threshold_amount' => get_option('wal_module_auto_topup_max_threshold_amount'),
			);

			wal_get_template('modules/auto-topup/auto-topup-form.php', $args);
		}

		/**
		 * Get payment gateways to load in to the WC checkout
		 */
		public function load_payment_gateways() {
			if (!class_exists('WC_Payment_Gateway')) {
				return;
			}

			include_once WAL_ABSPATH . 'inc/gateways/class-wc-stripe-gateway.php';
		}

		/**
		 * Get Auto Topup amount html.
		 * 
		 * @param string $total
		 * @return string
		 */
		public function get_auto_topup_amount_html( $total ) {
			$auto_topup = $this->get_auto_topup_from_cart();

			if ($auto_topup) {
				$total = str_replace('{auto_top_up_amount}', '<code>' . wal_convert_price_by_currency($auto_topup['auto_topup_price']) . '</code>', get_option('wal_module_auto_topup_authorization_message'));
			}

			return $total;
		}

		/**
		 * Validate the cart items.
		 */
		public function validate_cart_items() {
			if (!$this->cart_contains_auto_topup()) {
				return;
			}

			if ($this->is_invalid_user()) {
				wal_add_wc_notice(esc_html(get_option('wal_module_auto_topup_user_not_eligible_notice')), 'error');
			}
		}

		/**
		 * Handles the payment gateways.
		 * 
		 * @return array.
		 */
		public function handle_payment_gateways( $wc_gateways ) {
			if (is_null(WC()->cart) || !$this->cart_contains_auto_topup()) {
				return $wc_gateways;
			}

			foreach ($wc_gateways as $gateway_id => $gateway) {
				if (!$gateway->supports('wal_wallet')) {
					unset($wc_gateways[$gateway_id]);
				}
			}

			return $wc_gateways;
		}

		/**
		 * May be process the auto topup form.
		 * 
		 * @return void
		 * */
		public function process_auto_topup_form() {
			//To avoid, call by cart ajax
			if (isset($_POST['woocommerce-cart-nonce'])) {
				return;
			}

			$nonce_value = isset($_POST['wal-auto-topup-nonce']) ? wc_clean(wp_unslash($_POST['wal-auto-topup-nonce'])) : null;
			if (!wp_verify_nonce($nonce_value, 'wal-auto-topup')) {
				return;
			}

			try {
				$posted = $_POST;

				if (isset($posted['wal-auto-topup-pay-and-authorize']) || isset($posted['wal-auto-topup-authorize'])) {
					if (empty($posted['wal_auto_topup_agree']) && 'yes' === get_option('wal_module_auto_topup_terms_and_conditions')) {
						throw new Exception(get_option('wal_module_auto_topup_user_not_agreed_terms_notice'));
					}

					$auto_topup_amount = 0;
					if (isset($posted['wal_auto_topup_predefined_amount'])) {
						$auto_topup_amount = 'user-defined' === $posted['wal_auto_topup_predefined_amount'] ? $posted['wal_auto_topup_custom_amount'] : $posted['wal_auto_topup_predefined_amount'];
					} else if (isset($posted['wal_auto_topup_custom_amount'])) {
						$auto_topup_amount = $posted['wal_auto_topup_custom_amount'];
					}

					$auto_topup_amount = wal_convert_price($auto_topup_amount, true);
					if (empty($auto_topup_amount)) {
						throw new Exception(get_option('wal_module_auto_topup_amount_required_notice'));
					}

					$threshold_amount = 0;
					if (isset($posted['wal_auto_topup_threshold_predefined_amount'])) {
						$threshold_amount = 'user-defined' === $posted['wal_auto_topup_threshold_predefined_amount'] ? $posted['wal_auto_topup_threshold_custom_amount'] : $posted['wal_auto_topup_threshold_predefined_amount'];
					} else if (isset($posted['wal_auto_topup_threshold_custom_amount'])) {
						$threshold_amount = $posted['wal_auto_topup_threshold_custom_amount'];
					}

					$threshold_amount = wal_convert_price($threshold_amount, true);
					if (empty($threshold_amount)) {
						throw new Exception(get_option('wal_module_auto_topup_threshold_amount_required_notice'));
					}

					if (floatval($auto_topup_amount) < floatval($threshold_amount)) {
						throw new Exception(get_option('wal_module_auto_topup_amount_must_gt_than_threshold_amount_notice'));
					}

					if (!WC()->cart->is_empty()) {
						WC()->cart->empty_cart();
					}

					$was_added_to_cart = WC()->cart->add_to_cart($this->get_topup_product_id(), 1, 0, array(), array(
						'wal_topup' => array(
							'product_id' => $this->get_topup_product_id(),
							'price' => isset($posted['wal-auto-topup-authorize']) ? 0 : floatval($auto_topup_amount),
							'auto_topup_price' => floatval($auto_topup_amount),
							'auto_topup_threshold_price' => floatval($threshold_amount),
							'mode' => 'auto',
						),
					));

					if (false === $was_added_to_cart) {
						throw new Exception(__('Something went wrong while authorizing future payments.', 'wallet-for-woocommerce'));
					}

					wal_add_wc_notice(get_option('wal_module_auto_topup_authorization_notice'), 'success');

					//redirect to checkout page
					wp_safe_redirect(wc_get_checkout_url());
					exit;
				} else if (isset($posted['wal_cancel_auto_topup'])) {
					$auto_topup = wal_get_auto_topup(absint($posted['wal_cancel_auto_topup']));

					if (!$auto_topup->exists()) {
						throw new Exception(__('We were unable to process your cancel auto Top-up request, please try again.', 'wallet-for-woocommerce'));
					}

					$auto_topup->update_status('wal_cancelled');

					wal_update_auto_topup($auto_topup->get_id(), array( 'wal_last_activity' => __('User has Cancelled their Authorization.', 'wallet-for-woocommerce') ));
					wal_add_wc_notice(get_option('wal_module_auto_topup_authorization_cancelled_notice'), 'success');
					/**
					 * This hook is used to do extra action after wallet auto top-up authorization is cancelled.
					 * 
					 * @since 1.0
					 */
					do_action('wal_wallet_auto_topup_authorization_is_cancelled', $auto_topup->get_id());
				}
			} catch (Exception $e) {
				WAL_Dashboard_Handler::add_error(esc_html($e->getMessage()));
			}
		}

		/**
		 * Needs payment in checkout page?
		 * 
		 * @param bool $bool
		 */
		public function needs_payment( $bool ) {
			if ($this->cart_contains_auto_topup()) {
				return true;
			}

			return $bool;
		}

		/**
		 * Update order meta.
		 * 
		 * @param int $order_id
		 */
		public function update_order_meta( $order_id ) {
			$order = wc_get_order($order_id);

			foreach (WC()->cart->get_cart() as $value) {
				if (!isset($value['wal_topup']) || 'auto' !== $value['wal_topup']['mode']) {
					continue;
				}

				$topup_data = array(
					'wal_auto_topup_price' => $value['wal_topup']['auto_topup_price'],
					'wal_auto_topup_threshold_price' => $value['wal_topup']['auto_topup_threshold_price'],
				);

				foreach ($topup_data as $key => $value) {
					$order->add_meta_data($key, $value);
				}
			}

			$order->save();
		}

		/**
		 * Activate auto topup to the user after the payment method has been authorized.
		 *
		 * @param int $order_id
		 */
		public function on_authorization_success( $order_id ) {
			$order = wc_get_order($order_id);
			if (!$order || 'auto' !== $order->get_meta('wal_topup_mode') || 'yes' === $order->get_meta('wal_auto_topup_authorized')) {
				return;
			}

			$wallet_id = wal_get_wallet_id_by_user_id($order->get_customer_id());
			if (!$wallet_id) {
				$wallet_id = wal_create_new_wallet(array(
					'wal_balance' => 0,
					'wal_current_expiry_date' => wal_get_wallet_current_expiry_date(),
					'wal_currency' => $order->get_currency(),
						), array( 'post_status' => 'wal_active', 'post_parent' => $order->get_customer_id() ));
			}

			$wallet = wal_get_wallet($wallet_id);
			if (!$wallet->exists()) {
				return;
			}

			$auto_topup_id = wal_get_auto_topup_id_by_wallet_id($wallet_id);
			if (!$auto_topup_id) {
				$auto_topup_id = wal_create_new_auto_topup(array(), array(
					'post_type' => WAL_Register_Post_Types::AUTO_TOPUP_USERS_POSTTYPE,
					'post_status' => 'wal_active',
					'post_parent' => $wallet_id,
					'post_author' => 1,
					'post_title' => 'Wallet Auto Topup',
				));
			}

			$auto_topup = wal_get_auto_topup($auto_topup_id);
			if (!$auto_topup->exists()) {
				return;
			}

			$is_authorized_only = $order->get_total('edit') <= 0;
			wal_update_auto_topup($auto_topup_id, array(
				'wal_user_id' => $order->get_customer_id(),
				'wal_topup_amount' => $order->get_meta('wal_auto_topup_price'),
				'wal_threshold_amount' => $order->get_meta('wal_auto_topup_threshold_price'),
				'wal_payment_method' => $order->get_payment_method(),
				'wal_authorized_order' => $order->get_id(),
				'wal_currency' => $order->get_currency(),
				'wal_last_transaction_id' => $order->get_transaction_id(),
				'wal_last_activity' => $is_authorized_only ? __('Authorized Successfully.', 'wallet-for-woocommerce') : __('Authorized and Paid Successfully.', 'wallet-for-woocommerce'),
			));

			$auto_topup->update_status('wal_active');

			if ($is_authorized_only) {
				/**
				 * This hook is used to do extra action after wallet payment authorized for auto top-up.
				 * 
				 * @since 1.0
				 */
				do_action('wal_wallet_payment_authorized_for_auto_topup', $auto_topup_id, $order_id, $wallet_id);
			} else {
				/**
				 * This hook is used to do extra action after wallet payment paid and authorized for auto top-up.
				 * 
				 * @since 1.0
				 */
				do_action('wal_wallet_payment_paid_and_authorized_for_auto_topup', $auto_topup_id, $order_id, $wallet_id);
			}

			$order->update_meta_data('wal_auto_topup_authorized', 'yes');
			$order->save();
		}

		/**
		 * Monitor wallet funds status upon debit.
		 * 
		 * @param int $transaction_log_id
		 * @param object $transaction_log
		 */
		public function check_wallet_funds_on_debit( $transaction_log_id, $transaction_log ) {
			$this->process_payment($transaction_log->get_wallet());
		}

		/**
		 * Process auto topup payment via supported payment method.
		 * 
		 * @param object $wallet
		 * @throws Exception
		 */
		public function process_payment( $wallet ) {
			$auto_topup_id = wal_get_auto_topup_id_by_wallet_id($wallet->get_id());
			$auto_topup = wal_get_auto_topup($auto_topup_id);

			if (!$auto_topup->has_status('wal_active') || floatval($wallet->get_balance()) >= wal_convert_price($auto_topup->get_threshold_amount(), true)) {
				return;
			}

			try {
				if (floatval($auto_topup->get_topup_amount()) <= 0) {
					throw new Exception(__('Top-up amount should be greater than zero.', 'wallet-for-woocommerce'));
				}

				$order = wc_get_order($auto_topup->get_authorized_order());

				if (!$order) {
					throw new Exception(__('Invalid order to auto topup the wallet.', 'wallet-for-woocommerce'));
				}

				/**
				 * This hook is used to alter the auto top-up payment gateway.
				 * 
				 * @since 1.0
				 */
				$result = apply_filters("wal_wallet_auto_topup_payment_via_{$auto_topup->get_payment_method()}", false, $auto_topup, $order);

				if (is_wp_error($result)) {
					throw new Exception($result->get_error_message());
				}

				if (!$result) {
					throw new Exception(__('Unable to Auto Top-up the wallet', 'wallet-for-woocommerce'));
				}
				/**
				 * This hook is used to do extra action after wallet auto top-up successful.
				 * 
				 * @since 1.0
				 */
				do_action('wal_wallet_auto_topup_successful', $auto_topup_id);
			} catch (Exception $e) {
				$auto_topup->update_status('wal_cancelled');

				/**
				 * This hook is used to do extra action after wallet auto top-up is failed.
				 * 
				 * @since 1.0
				 */
				do_action('wal_wallet_auto_topup_is_failed', $auto_topup_id);
			}
		}

		/**
		 * Topup funds on demand.
		 * 
		 * @param int $auto_topup_id
		 */
		public function topup_funds( $auto_topup_id ) {
			$auto_topup = wal_get_auto_topup($auto_topup_id);
			if (!$auto_topup->exists() || !$auto_topup->get_order()) {
				return;
			}

			if (doing_action('wal_wallet_payment_paid_and_authorized_for_auto_topup')) {
				$event_message = str_replace('{order_id}', '#' . $auto_topup->get_authorized_order(), get_option('wal_localization_wallet_topup_fund_credit_log'));
			} else {
				$renewal_order_id = $this->create_renewal_order($auto_topup);
				$auto_topup->update_renewal_details($renewal_order_id);

				$event_message = str_replace('{order_id}', '#' . $renewal_order_id, get_option('wal_module_auto_topup_successful_notice'));
			}

			wal_credit_wallet_fund(array(
				'user_id' => $auto_topup->get_user_id(),
				'amount' => floatval($auto_topup->get_topup_amount()),
				'event_id' => 12,
				'event_message' => $event_message,
				'currency' => $auto_topup->get_currency(),
				'update_topup_total' => true,
			));

			wal_update_auto_topup($auto_topup_id, array(
				'wal_last_charge_date' => WAL_Date_Time::get_mysql_date_time_format('now', true),
				/* translators: 1: transaction id */
				'wal_last_activity' => sprintf(__('Auto Top-up Successful (Transaction ID: %s)', 'wallet-for-woocommerce'), $auto_topup->get_last_transaction_id()),
			));

			$topup_bonus_amount = WAL_Topup_Bonus_Handler::get_amount($auto_topup->get_topup_amount());
			if ($topup_bonus_amount) {
				self::prepare_auto_topup_bonus_log($auto_topup->get_authorized_order(), $auto_topup, $topup_bonus_amount);
			}
		}

		/**
		 * Create a renewal order.
		 * 
		 * @since 2.8.0
		 * @param object $auto_topup
		 * 
		 * @return int.
		 */
		public function create_renewal_order( $auto_topup ) {
			$product = wc_get_product($this->get_topup_product_id());
			if (!is_object($product)) {
				return;
			}

			$product->set_price($auto_topup->get_topup_amount());
			$order = $auto_topup->get_order();

			// Create a order.
			$new_order = wc_create_order(array( 'status' => 'pending', 'customer_id' => $order->get_customer_id() ));
			$new_order->add_product($product, 1, array( 'total' => $auto_topup->get_topup_amount(), 'subtotal' => $auto_topup->get_topup_amount() ));
			$new_order->calculate_totals();
			$new_order->save();

			// It will trigger the WooCommerce Emails for the order.
			$new_order->update_status('completed');

			// Update order meta.
			$new_order->update_meta_data('wal_auto_topup_parent_order_id', $order->get_id());
			$new_order->save();

			$order->update_meta_data('wal_auto_topup_last_renewal_order_id', $new_order->get_id());
			$order->save();

			return $new_order->get_id();
		}

		/**
		 * Prepare auto top-up bonus log.
		 *
		 * @since 2.5.0
		 * @param Object $auto_topup 
		 * $param String $event_message
		 * @param float $topup_bonus_amount 
		 * 
		 * @return void
		 * */
		public static function prepare_auto_topup_bonus_log( $order_id, $auto_topup, $topup_bonus_amount ) {
			if (doing_action('wal_wallet_payment_paid_and_authorized_for_auto_topup')) {
				$event_message = str_replace('{order_id}', '#' . $order_id, wal_topup_bonus_fund_credit_log_label());
			} else {
				$event_message = __('Wallet Bonus Auto Top-up Successful', 'wallet-for-woocommerce');
			}

			wal_credit_wallet_fund(array(
				'user_id' => $auto_topup->get_user_id(),
				'amount' => floatval($topup_bonus_amount),
				'event_id' => 12,
				'event_message' => $event_message,
				'currency' => $auto_topup->get_currency(),
				'update_topup_total' => true,
			));
		}

		/**
		 * Add the custom email classes.
		 * 
		 * @return array
		 */
		public function email_classes( $classes ) {
			$classes = ( wal_check_is_array($classes) ) ? $classes : array();

			$classes['customer-auto-topup-acknowledgement'] = 'WAL_Customer_Auto_Topup_Acknowledgement_Notification';
			$classes['customer-auto-topup-success'] = 'WAL_Customer_Auto_Topup_Success_Notification';
			$classes['customer-auto-topup-cancelled'] = 'WAL_Customer_Auto_Topup_Cancelled_Notification';
			return $classes;
		}

		/**
		 * Check if we need to send WC emails for Authorized orders ?
		 */
		public function prevent_sending_wc_email( $bool, $order ) {
			$order = wc_get_order($order);

			if (is_object($order) && 'shop_order' === $order->get_type() && 'auto' === $order->get_meta('wal_topup_mode')) {
				$bool = false;
			}

			return $bool;
		}

		/**
		 * Get auto top up renewal order details popup content
		 *
		 * @since 2.8.0
		 * 
		 * @return void
		 */
		public static function get_auto_topup_renewal_order_details_popup_content() {
			check_ajax_referer('wal-auto-topup', 'wal_security');

			try {
				if (!isset($_REQUEST['order_id']) || empty($_REQUEST['order_id'])) {
					throw new exception(__('Invalid Request.', 'wallet-for-woocommerce'));
				}

				$order_id = absint($_REQUEST['order_id']);
				$order = wc_get_order($order_id);

				if (!is_object($order)) {
					return;
				}

				$renewal_order_ids = array_filter((array) $order->get_meta('wal_auto_topup_renewal_order_ids'), 'wal_array_filter');

				ob_start();
				include_once WAL_PLUGIN_PATH . '/inc/modules/views/html-auto-topup-renewal-order-details.php';
				$html = ob_get_contents();
				ob_end_clean();

				wp_send_json_success(array( 'html' => $html ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}
	}

}
