<?php

defined( 'ABSPATH' ) || exit ;

if ( ! class_exists( 'WC_Gateway_Stripe' ) ) {
	return ;
}

if ( ! class_exists( 'WAL_WC_Stripe_Gateway' ) ) {

	/**
	 * Handles the WC Stripe Auto Topup Payments
	 * 
	 * @class WAL_WC_Stripe_Gateway
	 * @package Class
	 */
	class WAL_WC_Stripe_Gateway {

		/**
		 * Get the WC Stripe Gateway
		 * 
		 * @var WC_Gateway_Stripe 
		 */
		protected static $stripe ;

		/**
		 * Init WAL_WC_Stripe_Gateway.
		 */
		public static function init() {
			add_filter( 'woocommerce_payment_gateway_supports', __CLASS__ . '::supports', 99, 3 ) ;
			add_filter( 'wc_stripe_display_save_payment_method_checkbox', __CLASS__ . '::maybe_hide_save_checkbox', 99 ) ;
			add_filter( 'wc_stripe_force_save_source', __CLASS__ . '::save_payment_mode' ) ;
			add_filter( 'wc_stripe_payment_metadata', __CLASS__ . '::add_payment_metadata', 10, 2 ) ;
			add_filter( 'wal_wallet_auto_topup_payment_via_stripe', __CLASS__ . '::process_payment', 10, 3 ) ;
		}

		/**
		 * Get the Stripe instance.
		 */
		public static function get_stripe_instance() {
			if ( is_null( self::$stripe ) && WC()->payment_gateways() ) {
				$payment_gateways = WC()->payment_gateways->payment_gateways() ;

				if ( isset( $payment_gateways[ 'stripe' ] ) && is_a( $payment_gateways[ 'stripe' ], 'WC_Gateway_Stripe' ) ) {
					self::$stripe = $payment_gateways[ 'stripe' ] ;
				}
			}

			return self::$stripe ;
		}

		/**
		 * Add gateway supports Wallet Auto Topup.
		 * 
		 * @param bool $bool
		 * @param string $feature
		 * @param WC_Payment_Gateway $gateway
		 * @return bool
		 */
		public static function supports( $bool, $feature, $gateway ) {
			if ( 'stripe' === $gateway->id && in_array( $feature, array( 'wal_wallet' ) ) ) {
				$bool = true ;
			}

			return $bool ;
		}

		/**
		 * Checks to see if we need to hide the save checkbox field.
		 * Because when cart contains a savings plan product, it will save regardless.
		 */
		public static function maybe_hide_save_checkbox( $display_tokenization ) {
			$auto_topup_module = WAL_Module_Instances::get_module_by_id( 'auto_topup' ) ;

			if ( $auto_topup_module && $auto_topup_module->cart_contains_auto_topup() ) {
				$display_tokenization = false ;
			}

			return $display_tokenization ;
		}

		/**
		 * Checks to see if we need to save the payment mode.
		 * Because when cart contains a savings plan product, source should be saved.
		 */
		public static function save_payment_mode( $force_save ) {
			global $wp ;

			$order_id = 0 ;
			if ( isset( $_REQUEST[ 'woocommerce_pay' ], $_GET[ 'key' ] ) ) {
				$nonce_value = '' ;

				if ( isset( $_REQUEST[ 'woocommerce-pay-nonce' ] ) ) {
					$nonce_value = sanitize_key( wp_unslash( $_REQUEST[ 'woocommerce-pay-nonce' ] ) ) ;
				} else if ( isset( $_REQUEST[ '_wpnonce' ] ) ) {
					$nonce_value = sanitize_key( wp_unslash( $_REQUEST[ '_wpnonce' ] ) ) ;
				}

				if ( wp_verify_nonce( $nonce_value, 'woocommerce-pay' ) ) {
					$order_id = absint( $wp->query_vars[ 'order-pay' ] ) ;
				}
			} else if ( WC()->session ) {
				$order_id = absint( WC()->session->get( 'order_awaiting_payment' ) ) ;
			}

			if ($order_id) {
				$order = wc_get_order($order_id);
				
				if (is_object($order) && 'auto' === $order->get_meta('wal_topup_mode')) {
					$force_save = true ;
				}
			}

			return $force_save ;
		}

		/**
		 * Add payment metadata to Stripe.
		 */
		public static function add_payment_metadata( $metadata, $order ) {
			$order = wc_get_order( $order ) ;

			if ( 'auto' !== $order->get_meta('wal_topup_mode') ) {
				return $metadata ;
			}

			$metadata[ 'payment_mode' ] = 'automatic' ;
			$metadata[ 'order_type' ]   = 'wallet-auto-topup' ;
						
			return $metadata ;
		}

		/**
		 * Process the payment.
		 * 
		 * @param bool $bool
		 * @param object $auto_topup
		 * @return bool
		 */
		public static function process_payment( $bool, $auto_topup, $order, $retry = false ) {          
			try {
				$prepared_source = self::get_stripe_instance()->prepare_order_source( $order ) ;

				if ( ! $prepared_source->customer ) {
					throw new WC_Stripe_Exception( 'Failed to process installment for order ' . $order->get_id() . '. Stripe customer id is missing in the order', __( 'Customer not found', 'wallet-for-woocommerce' ) ) ;
				}

				if ( $retry ) {
					// Passing empty source will charge customer default.
					$prepared_source->source = '' ;
				}

				if ( is_callable( array( self::get_stripe_instance(), 'create_and_confirm_intent_for_off_session' ) ) ) {
					$response = self::get_stripe_instance()->create_and_confirm_intent_for_off_session( $order, $prepared_source, $auto_topup->get_topup_amount() ) ;
				} else {
					$request              = self::get_stripe_instance()->generate_payment_request( $order, $prepared_source ) ;
					$request[ 'amount' ]  = WC_Stripe_Helper::get_stripe_amount( $auto_topup->get_topup_amount(), strtolower( $order->get_currency() ) ) ;
					$request[ 'capture' ] = 'true' ;
										
					$response = WC_Stripe_API::request( $request ) ;
				}

				$is_authentication_required = self::get_stripe_instance()->is_authentication_required_for_payment( $response ) ;

				if ( ! empty( $response->error ) && ! $is_authentication_required ) {
					/**
					 * This hook is used to alter the WC stripe default customer source.
					 * 
					 * @since 1.0
					 */
					if ( ! $retry && apply_filters( 'wal_wallet_wc_stripe_use_default_customer_source', true ) ) {
						return self::process_payment( $bool, $auto_topup, $order, true ) ;
					}

					$localized_messages = WC_Stripe_Helper::get_localized_messages() ;

					if ( 'card_error' === $response->error->type ) {
						$localized_message = isset( $localized_messages[ $response->error->code ] ) ? $localized_messages[ $response->error->code ] : $response->error->message ;
					} else {
						$localized_message = isset( $localized_messages[ $response->error->type ] ) ? $localized_messages[ $response->error->type ] : $response->error->message ;
					}

					throw new WC_Stripe_Exception( print_r( $response, true ), $localized_message ) ;
				}

				if ( $is_authentication_required ) {
					/**
					 * This hook is used to alter the wallet WC stripe use default customer source.
					 * 
					 * @since 1.0
					 */
					if ( ! $retry && apply_filters( 'wal_wallet_wc_stripe_use_default_customer_source', true ) ) {
						return self::process_payment( $bool, $auto_topup, $order, true ) ;
					}

					$charge_id = $response->error->charge ;

					wal_update_auto_topup( $auto_topup->get_id(), array(
						'wal_last_charge_date' => WAL_Date_Time::get_mysql_date_time_format( 'now', true ),
						/* translators: 1: Stripe charge id */
						'wal_last_activity'    => sprintf( __( 'Authorization Cancelled Due to Insufficient Funds. Error: Stripe requires authentication by user to charge: %s.', 'wallet-for-woocommerce' ), $charge_id ),
					) ) ;

					/* translators: 1: Stripe charge id */
					$order->add_order_note( sprintf( __( 'Stripe requires authentication by user to charge: %s.', 'wallet-for-woocommerce' ), $charge_id ) ) ;
					$order->set_transaction_id( $charge_id ) ;
					$order->save() ;

					/**
					 * This hook is used to do wallet WC stripe requires authentication.
					 * 
					 * @since 1.0
					 */
					do_action( 'wal_wallet_wc_stripe_requires_authentication', $auto_topup, $order ) ;
					return false ;
				}

				$response = self::get_stripe_instance()->process_response( ( isset( $response->charges->data ) ? end( $response->charges->data ) : $response ), $order ) ;
				wal_update_auto_topup( $auto_topup->get_id(), array( 'wal_last_transaction_id' => ! empty( $response->id ) ? $response->id : '-' ) ) ;
				/**
				 * This hook is used to do stripe process payment.
				 * 
				 * @since 1.0
				 */
				do_action( 'wc_gateway_stripe_process_payment', $response, $order ) ;
				/**
				 * This hook is used to do extra action after stripe payment processed successfully.
				 * 
				 * @since 1.0
				 */
				do_action( 'wal_wallet_wc_stripe_payment_processed_successful', $response, $auto_topup, $order ) ;
				return true ;
			} catch ( WC_Stripe_Exception $e ) {
				WC_Stripe_Logger::log( 'Error: ' . $e->getMessage() ) ;
				/**
				 * This hook is used to process the stripe payment error.
				 * 
				 * @since 1.0
				 */
				do_action( 'wc_gateway_stripe_process_payment_error', $e, $order ) ;

				if ( $e->getLocalizedMessage() ) {
					$order->add_order_note( $e->getLocalizedMessage() ) ;

					wal_update_auto_topup( $auto_topup->get_id(), array(
						'wal_last_charge_date' => WAL_Date_Time::get_mysql_date_time_format( 'now', true ),
						/* translators: 1: Stripe error message */
						'wal_last_activity'    => sprintf( __( 'Authorization Cancelled Due to Insufficient Funds. Error: %s', 'wallet-for-woocommerce' ), $e->getLocalizedMessage() ),
					) ) ;
				}
			}

			return false ;
		}
	}

	WAL_WC_Stripe_Gateway::init() ;
}
