<?php

/**
 * Wallet Gateway.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

use Automattic\WooCommerce\StoreApi\Exceptions\RouteException;

if (!class_exists('WAL_Gateway_Wallet')) {

	/**
	 * Class.
	 */
	class WAL_Gateway_Wallet extends WC_Payment_Gateway {

		/**
		 * Instructions.
		 * 
		 * @var string
		 * */
		protected $instructions;

		/**
		 * Constructor for the gateway.
		 */
		public function __construct() {

			// Setup general properties.
			$this->setup_properties();

			// Load the settings.
			$this->init_form_fields();
			$this->init_settings();

			$this->init_hooks();
			$this->populate_data();

			add_filter( 'woocommerce_available_payment_gateways', array( $this, 'unset_wallet_gateway' ), 10, 1 );
		}

		/**
		 * Setup general properties for the gateway.
		 */
		protected function setup_properties() {
			$this->id = 'wal_wallet';
			$this->has_fields = false;
			/**
			 * This hook is used to alter the wallet gateway icon.
			 * 
			 * @since 1.0
			 */
			$this->icon = apply_filters('wal_wallet_icon', '');
			$this->method_title = __('Wallet', 'wallet-for-woocommerce');
			$this->method_description = __('Use Wallet Funds to Complete this order', 'wallet-for-woocommerce');
			/**
			 * This hook is used to alter the wallet gateway supports.
			 * 
			 * @since 1.0
			 */
			$this->supports = apply_filters('wal_wallet_gateway_supports', array(
				'products',
				'refunds',
			));
		}

		/**
		 * Populate Data.
		 */
		protected function populate_data() {
			$this->title = $this->get_option('title', 'Wallet');
			$this->instructions = $this->get_option('instructions');
			$this->description = $this->get_option('description', 'Pay using your Wallet Funds({available_funds}).');
		}

		/**
		 * Initialize the gateway settings form fields.
		 */
		public function init_form_fields() {
			/**
			 * This hook is used to alter the wallet gateway form fields.
			 * 
			 * @since 1.0
			 */
			$this->form_fields = apply_filters('wal_wallet_gateway_form_fields', array(
				'enabled' => array(
					'title' => __('Enable/Disable', 'wallet-for-woocommerce'),
					'type' => 'checkbox',
					'label' => __('Enable Wallet Gateway', 'wallet-for-woocommerce'),
					'default' => 'yes',
				),
				'title' => array(
					'title' => __('Title', 'wallet-for-woocommerce'),
					'type' => 'text',
					'std' => __('Wallet', 'wallet-for-woocommerce'),
					'default' => __('Wallet', 'wallet-for-woocommerce'),
					'desc_tip' => true,
				),
				'description' => array(
					'title' => __('Description', 'wallet-for-woocommerce'),
					'type' => 'textarea',
					'std' => __('Pay using your Wallet Funds({available_funds}).', 'wallet-for-woocommerce'),
					'default' => __('Pay using your Wallet Funds({available_funds}).', 'wallet-for-woocommerce'),
					'css' => 'max-width:400px;',
					'desc_tip' => true,
				),
				'instructions' => array(
					'title' => __('Instructions', 'wallet-for-woocommerce'),
					'type' => 'textarea',
					'description' => __('Instructions that will be added to the thank you page.', 'wallet-for-woocommerce'),
					'default' => '',
					'css' => 'max-width:400px;',
					'desc_tip' => true,
				),
			));
		}

		/**
		 * Return the gateway's description.
		 *
		 * @return string
		 */
		public function get_description() {
			$balance = class_exists('WAL_Current_User_Wallet') ? WAL_Current_User_Wallet::get_balance() : 0;
			/**
			 * This hook is used to alter the wallet user balance.
			 *
			 * @param float $balance
			 * @since 3.0.0
			 */
			$balance = apply_filters('wal_payment_gateway_user_balance', $balance);
			/**
			 * This hook is used to alter the payment gateway available user  balance.
			 *
			 * @param float $balance
			 * @since 3.0.0
			 */
			$available_funds = apply_filters('wal_payment_gateway_available_user_funds', wal_price($balance), $balance);
			$description = str_replace('{available_funds}', $available_funds, $this->description);

			/**
			 * This hook is used to alter the wallet gateway description.
			 * 
			 * @since 1.0
			 */
			return apply_filters('woocommerce_gateway_description', $description, $this->id);
		}

		/**
		 * Initialize the hook for gateway.
		 */
		public function init_hooks() {
			add_action('woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ));
		}

		/**
		 * Unset Wallet Gateway.
		 */
		public function unset_wallet_gateway( $gateways ) {
			if ( ! is_checkout_pay_page() && ( ! is_object( WC()->cart ) || ! wal_check_is_array( WC()->cart->cart_contents ) ) ) {
				unset( $gateways['wal_wallet'] );
				return $gateways;
			}

			$get_items = WC()->cart->cart_contents;

			global $wp;
			if ( isset($wp->query_vars['order-pay']) && absint($wp->query_vars['order-pay']) > 0 ) {
				$order_id = absint($wp->query_vars['order-pay']);
				$order    = wc_get_order( $order_id );
				$get_items = $order->get_items();
			}

			foreach ( $get_items as $values ) {
				$gateways = $this->get_valid_gateways( $values, $gateways );
			}
			
			return $gateways;
		}

		/**
		 * Get Valid Gateway(s).
		 */
		public function get_valid_gateways( $cart_content, $gateways ) {
			$variation = wc_get_product( $cart_content[ 'variation_id' ] );
			if ( $variation && $variation->is_type( 'variation' ) ) {
				$product_id = $variation->get_parent_id();
			} else {
				$product_id = $cart_content[ 'product_id' ];
			}

			$product_type = get_option( 'wal_general_wallet_usage_product_restriction_type' ) ;
			switch ( $product_type ) {
				case '2':
					$include_products = array_filter( ( array ) get_option( 'wal_general_wallet_usage_include_product' ), 'wal_array_filter' ) ;
					// Return true if any selected products not in the cart.
					if ( ! in_array( $product_id, $include_products ) ) {
						unset( $gateways[ 'wal_wallet' ] );
					}

					break ;
				case '3':
					$exclude_products = array_filter( ( array ) get_option( 'wal_general_wallet_usage_exclude_product' ), 'wal_array_filter' ) ;
					// Return true if any selected products in the cart.
					if ( in_array( $product_id, $exclude_products ) ) {
						unset( $gateways[ 'wal_wallet' ] );
					}
					break ;
				case '4':
					$include_categories = array_filter( ( array ) get_option( 'wal_general_wallet_usage_include_categories' ), 'wal_array_filter' ) ;
					// Included categories.
					$product_categories = get_the_terms( $product_id, 'product_cat' ) ;

					if ( wal_check_is_array( $product_categories ) ) {
						foreach ( $product_categories as $product_category ) {
							$category_ids[] = $product_category->term_id ;
							// Return false if any selected products of category in the cart.
							if ( ! in_array( $product_category->term_id, $include_categories ) ) {
								unset( $gateways[ 'wal_wallet' ] );
							}
						}
					}
					break ;
				case '5':
					$exclude_categories = array_filter( ( array ) get_option( 'wal_general_wallet_usage_exclude_categories' ), 'wal_array_filter' ) ;
					// Excluded categories.
					$product_categories = get_the_terms( $product_id, 'product_cat' ) ;
					if ( wal_check_is_array( $product_categories ) ) {
						foreach ( $product_categories as $product_category ) {
							// Return true if any selected products of category in the cart.
							if ( in_array( $product_category->term_id, $exclude_categories ) ) {
								unset( $gateways[ 'wal_wallet' ] );
								break;
							}
						}
					}
					break ;
			}

			return $gateways;
		}

		/**
		 * Process Payment.
		 */
		public function process_payment( $order_id ) {
			try {

				$order = wc_get_order($order_id);
				// Prevent deducting amounts another order type.
				if ('shop_order' !== $order->get_type()) {
					return;
				}

				//Process wallet debit.
				self::process_wallet_debit($order);

				//Change status to processing
				$order->update_status(get_option('wal_general_default_gateway_order_status', 'processing'), __('Awaiting for Admin Confirmation', 'wallet-for-woocommerce'));

				//Empty the cart after payment processed.
				WC()->cart->empty_cart();

				return array(
					'result' => 'success',
					'redirect' => $this->get_return_url($order),
				);
			} catch (Exception $e) {
				// Rest API Validation.
				throw new RouteException('woocommerce_rest_checkout_process_payment_error', wp_kses_post($e->getMessage()), 402);
			}
		}

		/**
		 * Process Refund.
		 * 
		 * @return bool
		 */
		public function process_refund( $orderid, $amount = null, $reason = '' ) {
			$order = wc_get_order($orderid);

			$payment_gateway_credit_log = get_option('wal_localization_wallet_payment_gateway_credit_log');
			$find = array( '{order_id}', '{order_status}' );
			$replace = array( '#' . $order->get_id(), wal_display_post_status(get_post_status($order->get_id()), false) );
			$event_message = str_replace($find, $replace, $payment_gateway_credit_log);

			$args = array(
				'user_id' => $order->get_customer_id(),
				'order_id' => $order->get_id(),
				'amount' => $amount,
				'event_id' => 7,
				'event_message' => $event_message,
				'currency' => $order->get_currency(),
				'update_usage_total' => true,
			);

			$transaction_log_id = wal_credit_wallet_fund($args);
			if (!$transaction_log_id) {
				throw new exception(esc_html__('Amount not credited.', 'wallet-for-woocommerce'));
			}

			// Add the note.
			/* translators: %1s- Amount, %2s - User name */
			$note = sprintf(__('The amount of %1$s credited to user %2$s', 'wallet-for-woocommerce'), wal_price($amount, array( 'currency' => $order->get_currency() )), $order->get_user()->display_name);
			$order->add_order_note($note);

			// handles the meta when order placed by Wallet gateway.
			if ('yes' === $order->get_meta('wal_gateway_fund_debited')) {
				// Update the fund details.
				$remaining_fund = floatval($order->get_meta('wal_gateway_remaining_fund', $order->get_total()));
				$remaining_fund = $remaining_fund - $amount;
				$remaining_fund = ( $remaining_fund < 0 ) ? 0 : $remaining_fund;

				$order->update_meta_data('wal_gateway_remaining_fund', $remaining_fund);
				$order->update_meta_data('wal_wallet_fund', $remaining_fund);

				// Remove the meta when the fund fully refunded.
				if ($remaining_fund < 0) {
					// delete the meta.
					$order->delete_meta_data('wal_gateway_fund_debited');
					$order->delete_meta_data('wal_wallet_fund_debited');
				}
			}

			$order->save();

			/**
			 * This hook is used to do extra action after gateway order fund credited.
			 * 
			 * @since 1.0
			 */
			do_action('wal_order_gateway_fund_credited', $transaction_log_id, $orderid, $order);

			return true;
		}

		/**
		 * Debit the fund from user wallet when placing order using Wallet payment.
		 * 
		 * @return void
		 */
		public static function process_wallet_debit( &$order ) {
			/**
			 * This hook is used to alter the payment gateway order total.
			 *
			 * @param float $order->get_total()
			 * @since 3.0.0
			 */
			$total = apply_filters('wal_payment_gateway_order_total', $order->get_total());

			/**
			 * This hook is used to alter the payment gateway user balance.
			 *
			 * @param float WAL_Current_User_Wallet::get_balance()
			 * @since 3.0.0
			 */
			$balance = apply_filters('wal_payment_gateway_user_balance', WAL_Current_User_Wallet::get_balance());
			// Throw the error if the wallet amount less than order total. 
			if ($balance < $total) {
				$top_up = sprintf('<a href=%s>%s</a>', wal_dashboard_menu_endpoint_url('topup'), __('top-up', 'wallet-for-woocommerce'));
				/* translators: %s - topup */
				throw new exception(wp_kses_post(sprintf(__('You don"t have sufficient funds in your wallet. Please %s the funds to complete this order.', 'wallet-for-woocommerce'), $top_up)));
			}

			// Return if the current wallet not valid fund usage.
			if (!WAL_Current_User_Wallet::is_valid_fund_usage()) {
				throw new exception(esc_html__('Sorry, You are not valid to use fund.', 'wallet-for-woocommerce'));
			}

			$payment_gateway_debit_log = get_option('wal_localization_wallet_payment_gateway_debit_log');
			$event_message = str_replace('{order_id}', '#' . $order->get_id(), $payment_gateway_debit_log);

			/**
			 * This hook is used to alter the user id.
			 *
			 * @param int $user_id
			 * @since 4.9.0
			 */
			$user_id = apply_filters('wal_get_user_id_to_debit_wallet', $order->get_customer_id());

			$args = array(
				'user_id' => $user_id,
				'order_id' => $order->get_id(),
				'amount' => $total,
				'event_id' => 8,
				'event_message' => $event_message,
				'currency' => $order->get_currency(),
				'update_usage_total' => true,
			);

			$transaction_log_id = wal_debit_wallet_fund($args);

			if (!$transaction_log_id) {
				throw new exception(esc_html__('Amount not debited.', 'wallet-for-woocommerce'));
			}

			// Add the note.
			/* translators: %1s- Amount, %2s - User name */
			$note = sprintf(__('The amount of %1$s debited from user %2$s', 'wallet-for-woocommerce'), wal_price($order->get_total(), array( 'currency' => $order->get_currency() )), $order->get_user()->display_name);
			$order->add_order_note($note);

			// Update the fund details.
			$order->update_meta_data('wal_gateway_total_fund', $total);
			$order->update_meta_data('wal_gateway_remaining_fund', $total);
			$order->update_meta_data('wal_wallet_fund', $total);

			// Update the meta.
			$order->update_meta_data('wal_wallet_fund_debited', 'yes');
			$order->update_meta_data('wal_gateway_fund_debited', 'yes');
			$order->save();

			/**
			 * This hook is used to do extra action after gateway order fund debited.
			 * 
			 * @since 1.0
			 */
			do_action('wal_order_gateway_fund_debited', $transaction_log_id, $order->get_id(), $order);
		}
	}

}
