<?php

/**
 *  Handles the Top up.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Topup_Handler' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Topup_Handler {

		/**
		 * Top up product ID.
		 * 
		 * @var int.
		 */
		protected static $topup_product_id = 0 ;

		/**
		 * Class Initialization.
		 * */
		public static function init() {
			self::set_topup_product_id() ;

			//May be process the topup form.
			add_action( 'wp_loaded', array( __CLASS__, 'process_topup_form' ) ) ;
		}

		/**
		 * Set the Top-up product ID.
		 * 
		 * @return void
		 */
		public static function set_topup_product_id() {
			$topup_product = array_filter( ( array ) get_option( 'wal_general_topup_product_id', array() ), 'wal_array_filter' ) ;

			if ( wal_check_is_array( $topup_product ) ) {
				self::$topup_product_id = reset( $topup_product ) ;
			}
		}

		/**
		 * Get the Top-up product ID.
		 * 
		 * @return void
		 */
		public static function get_topup_product_id() {
			return self::$topup_product_id ;
		}

		/**
		 * Display the content of top-up form.
		 * 
		 * @return void
		 * */
		public static function render() {
			$show           = true ;
			$validate_error = self::validate_render_form() ;

			if ( $validate_error ) {
				$show = false ;
				WAL_Dashboard_Handler::add_error( $validate_error ) ;
			}

			$args = array( 'show' => $show ) ;

			wal_get_template( 'topup-form.php', $args ) ;
		}

		/**
		 * May be process the topup form.
		 * 
		 * @return void
		 * */
		public static function process_topup_form() {
			//To avoid, call by cart ajax
			if ( isset( $_POST[ 'woocommerce-cart-nonce' ] ) ) {
				return ;
			}

			$nonce_value = isset( $_POST[ 'wal-topup-nonce' ] ) ? wc_clean( wp_unslash( $_POST[ 'wal-topup-nonce' ] ) ) : null ;
			if ( ! isset( $_POST[ 'wal-topup-action' ] ) || empty( $_POST[ 'wal-topup-action' ] ) || ! wp_verify_nonce( $nonce_value, 'wal-topup' ) ) {
				return ;
			}

			try {
				$fund = isset( $_POST[ 'wal-topup-form-amount' ] ) ? wc_clean( wp_unslash( $_POST[ 'wal-topup-form-amount' ] ) ) : '' ;
				// Return if the fund is empty.
				if ( ! $fund ) {
					throw new exception( get_option( 'wal_messages_wallet_topup_fund_empty' ) ) ;
				}

				// Return if the fund is not numeric.
				if ( ! is_numeric( $fund ) ) {
					throw new exception( get_option( 'wal_messages_wallet_topup_fund_numeric' ) ) ;
				}

				$incrementor = get_option('wal_general_topup_amount_incrementor');
				if ( ! empty( $incrementor ) ) {
					$remainder = $fund % $incrementor;
					if (!empty($remainder)) {
						$error_notice = get_option('wal_messages_error_notice_for_topup_incrementor');
						$error_notice = str_replace('{incrementor_value}', $incrementor , $error_notice);
						throw new exception( $error_notice ) ;
					}
				}

				// Return if the topup is disabled.
				if ( 'no' == get_option( 'wal_general_enable_topup' ) ) {
					throw new exception( __( 'Currently, top-up form is disabled. Hence, you are unable to top-up your wallet.', 'wallet-for-woocommerce' ) ) ;
				}

				// Return if the wallet is in active.
				if ( 'wal_inactive' == WAL_Current_User_Wallet::get_status() ) {
					throw new exception( get_option( 'wal_messages_topup_wallet_disabled' ) ) ;
				}

				// Return if the user is not valid.
				if ( self::validate_user_restriction() ) {
					throw new exception( get_option( 'wal_messages_topup_wallet_restricted' ) ) ;
				}

				$fund = wal_convert_price( $fund , true );
				// Return if the fund is less than minimum amount.
				$minimum_amount = floatval( get_option( 'wal_general_topup_min_amount' ) ) ;                                
				if ( $minimum_amount && $minimum_amount > $fund ) {
					throw new exception( str_replace( '{topup_min_amount}', wal_convert_price_by_currency( $minimum_amount ), get_option( 'wal_messages_wallet_topup_minimum_amount' ) ) ) ;
				}

				// Return if the fund is greater than maximum amount.
				$maximum_amount = floatval( get_option( 'wal_general_topup_max_amount' ) ) ;                                
				if ( $maximum_amount && $maximum_amount < $fund ) {
					throw new exception( str_replace( '{topup_max_amount}', wal_convert_price_by_currency( $maximum_amount ), get_option( 'wal_messages_wallet_topup_maximum_amount' ) ) ) ;
				}

				// Return if the fund is greater than maximum wallet amount.
				$max_wallet_balance = floatval( get_option( 'wal_general_topup_max_wallet_balance' ) ) ;
				if ( $max_wallet_balance && $max_wallet_balance < ( $fund + WAL_Current_User_Wallet::get_balance() ) ) {
					throw new exception( str_replace( '{max_threshold_value}', wal_convert_price_by_currency( $max_wallet_balance ), get_option( 'wal_messages_wallet_topup_maximum_wallet_amount_error' ) ) ) ;
				}

				// Return if the wallet topup count has reached the per day count.
				$max_count_per_day = floatval( get_option( 'wal_general_topup_count_per_day' ) ) ;
				if ( $max_count_per_day && $max_count_per_day <= wal_customer_wallet_topup_count_per_day() ) {
					throw new exception( '' ) ;
				}

				// Return if the wallet topup total has reached the per day total.
				$max_total_per_day = floatval( get_option( 'wal_general_topup_max_amount_per_day' ) ) ;
				if ( $max_total_per_day && $max_total_per_day < $fund + wal_customer_wallet_topup_total_per_day() ) {
					$topup_balance_amount = $max_total_per_day - wal_customer_wallet_topup_total_per_day() ;
					$topup_balance_amount = ( $topup_balance_amount > 0 ) ? $topup_balance_amount : 0 ;

					$find    = array( '{topup_amount}', '{topup_balance_amount}' ) ;
					$replace = array( wal_convert_price_by_currency( $max_total_per_day ), wal_convert_price_by_currency( $topup_balance_amount ) ) ;

					throw new exception( str_replace( $find, $replace, get_option( 'wal_messages_wallet_topup_maximum_amount_per_day_error' ) ) ) ;
				}
								
				$cart_item_data = array(
					'wal_topup' => array(
						'price'      => $fund,
						'mode'       => 'manual',
						'product_id' => self::get_topup_product_id(),
					),
				);

				self::maybe_save_cart_to_retain();

				// Empty the cart.
				WC()->cart->empty_cart() ;

				WC()->session->set('wal_fund_amount', $fund);

				// Add the topup product in the cart.
				WC()->cart->add_to_cart( self::get_topup_product_id(), '1', 0, array(), $cart_item_data ) ;

				//Redirect to checkout page.
				wp_safe_redirect( wc_get_checkout_url() ) ;
				exit() ;
			} catch ( Exception $ex ) {
				if ( $ex->getMessage() ) {
					WAL_Dashboard_Handler::add_error( $ex->getMessage() ) ;
				}
			}
		}

		/**
		 * May be save the cart in the user meta to retain after topup completed.
		 * 
		 * @since 2.6.0
		 */
		public static function maybe_save_cart_to_retain() {
			// Return if the cart is empty.
			if (!wal_check_is_array(WC()->cart->get_cart())) {
				return;
			}

			// Return if the option is disabled.
			if ('yes' !== get_option('wal_general_retain_cart_after_topup')) {
				return;
			}

			// Update the cart object in the current user meta after retain.
			update_user_meta(get_current_user_id(), 'wal_topup_retain_cart', base64_encode(maybe_serialize(WC()->cart->get_cart())));
		}

		/**
		 * Validate the user restriction.
		 * 
		 * @return bool
		 */
		public static function validate_user_restriction() {
			$return    = false ;
			$user_type = get_option( 'wal_general_topup_user_restriction_type' ) ;

			switch ( $user_type ) {
				case '2':
					$user_id       = get_current_user_id() ;
					$include_users = array_filter( ( array ) get_option( 'wal_general_topup_include_users' ), 'wal_array_filter' ) ;
					if ( ! in_array( $user_id, $include_users ) ) {
						$return = true ;
					}

					break ;
				case '3':
					$user_id       = get_current_user_id() ;
					$exclude_users = array_filter( ( array ) get_option( 'wal_general_topup_exclude_users' ), 'wal_array_filter' ) ;
					if ( in_array( $user_id, $exclude_users ) ) {
						$return = true ;
					}
					break ;
				case '4':
					$return             = true ;
					$user               = wp_get_current_user() ;
					$include_user_roles = array_filter( ( array ) get_option( 'wal_general_topup_include_user_roles' ), 'wal_array_filter' ) ;

					// Loggedin user restriction
					if ( wal_check_is_array( $user->roles ) ) {
						foreach ( $user->roles as $role ) {
							if ( in_array( $role, $include_user_roles ) ) {
								$return = false ;
							}
						}
					}

					break ;
				case '5':
					$user               = wp_get_current_user() ;
					$exclude_user_roles = array_filter( ( array ) get_option( 'wal_general_topup_exclude_user_roles' ), 'wal_array_filter' ) ;

					// Loggedin user restriction
					if ( wal_check_is_array( $user->roles ) ) {
						foreach ( $user->roles as $role ) {
							if ( in_array( $role, $exclude_user_roles ) ) {
								$return = true ;
							}
						}
					}

					break ;
			}
			/**
			 * This hook is used to validate the user top-up.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_validate_user_topup', $return ) ;
		}

		/**
		 * Validate the render top up form.
		 * 
		 * @return void
		 * */
		public static function validate_render_form() {
			// Return if the topup is disabled.
			if ( 'no' == get_option( 'wal_general_enable_topup' ) ) {
				return __( 'Currently, top-up form is disabled. Hence, you are unable to top-up your wallet.', 'wallet-for-woocommerce' ) ;
			}

			// Return if the wallet is in active.
			if ( 'wal_inactive' == WAL_Current_User_Wallet::get_status() ) {
				return get_option( 'wal_messages_topup_wallet_disabled' ) ;
			}

			// Return if the user is not valid.
			if ( self::validate_user_restriction() ) {
				return get_option( 'wal_messages_topup_wallet_restricted' ) ;
			}

			// Return if the fund is greater than maximum wallet amount.
			$max_wallet_balance = floatval( get_option( 'wal_general_topup_max_wallet_balance' ) ) ;
			if ( $max_wallet_balance && $max_wallet_balance <= WAL_Current_User_Wallet::get_balance() ) {
				return str_replace( '{max_threshold_value}', wal_convert_price_by_currency( $max_wallet_balance ), get_option( 'wal_messages_wallet_topup_maximum_wallet_amount' ) ) ;
			}

			// Return if the wallet topup count has reached the per day count.
			$max_count_per_day = floatval( get_option( 'wal_general_topup_count_per_day' ) ) ;
			if ( $max_count_per_day && $max_count_per_day <= wal_customer_wallet_topup_count_per_day() ) {
				return str_replace( '{topup_count}', $max_count_per_day, get_option( 'wal_messages_wallet_topup_maximum_count_per_day' ) ) ;
			}

			// Return if the wallet topup total has reached the per day total.
			$max_total_per_day = floatval( get_option( 'wal_general_topup_max_amount_per_day' ) ) ;
			if ( $max_total_per_day && $max_total_per_day <= wal_customer_wallet_topup_total_per_day() ) {
				return str_replace( '{topup_amount}', wal_convert_price_by_currency( $max_total_per_day ), get_option( 'wal_messages_wallet_topup_maximum_amount_per_day' ) ) ;
			}

			return false ;
		}
	}

	WAL_Topup_Handler::init() ;
}
