<?php

/**
 * Validate the fund usage.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Fund_Usage_Validation' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Fund_Usage_Validation {

		/**
		 * Wallet.
		 * 
		 * @var Object
		 */
		protected $wallet ;

		/**
		 * Wallet usage.
		 * 
		 * @var array
		 */
		protected static $wallet_usage ;

		/**
		 * Class Initialization.
		 */
		public function __construct( &$wallet ) {
			$this->wallet = $wallet ;
		}

		/**
		 * Is valid fund usage?.
		 * 
		 * @return bool
		 */
		public static function is_valid( &$wallet ) {
			if ( isset( self::$wallet_usage[ $wallet->get_id() ] ) ) {
				return self::$wallet_usage[ $wallet->get_id() ] ;
			}

			$validation = new self( $wallet ) ;

			self::$wallet_usage[ $wallet->get_id() ] = $validation->validate_fund_usage() ;

			return self::$wallet_usage[ $wallet->get_id() ] ;
		}

		/**
		 * Validate the fund usage.
		 * 
		 * @return bool
		 */
		public function validate_fund_usage() {
			$return = true ;

			if ( $this->validate_cart_subtotal() ) {
				$return = false ;
			} elseif ( $this->validate_days_restriction() ) {
				$return = false ;
			} elseif ( $this->validate_user_restriction() ) {
				$return = false ;
			} elseif ( $this->validate_security_settings() ) {
				$return = false ;
			}
			/**
			 * This hook is used to validate the fund usage.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_validate_fund_usage', $return, $this->wallet ) ;
		}

		/**
		 * Validate the cart subtotal.
		 * 
		 * @return bool
		 */
		public function validate_cart_subtotal() {
			$return         = false ;
			$subtotal       = wal_get_wc_cart_subtotal() ;
			$min_cart_total = get_option( 'wal_general_wallet_usage_min_cart_total' ) ;
			$max_cart_total = get_option( 'wal_general_wallet_usage_max_cart_total' ) ;

			// Validate the minimum cart subtotal.
			if ( $min_cart_total && floatval( $min_cart_total ) >= $subtotal ) {
				$return = true ;
			}

			// Validate the maximum cart subtotal.
			if ( $max_cart_total && floatval( $max_cart_total ) <= $subtotal ) {
				$return = true ;
			}
			/**
			 * This hook is used to validate the fund usage cart subtotal.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_validate_fund_usage_cart_subtotal', $return, $this->wallet ) ;
		}

		/**
		 * Validate the days restriction.
		 * 
		 * @return bool
		 */
		public function validate_days_restriction() {
			$return = false ;
			if ( $this->validate_date_range() ) {
				$return = true ;
			} elseif ( $this->validate_specific_weekdays() ) {
				$return = true ;
			}
			/**
			 * This hook is used to validate the fund usage days.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_validate_fund_usage_days', $return, $this->wallet ) ;
		}

		/**
		 * Validate the date range restriction.
		 * 
		 * @return bool
		 */
		public function validate_date_range() {
			$return               = false ;
			$fund_usage_from_date = get_option( 'wal_general_wallet_usage_from_date' ) ;
			$fund_usage_to_date   = get_option( 'wal_general_wallet_usage_to_date' ) ;
			$current_date         = current_time( 'timestamp' ) ;

			// Validate the start date and end date.
			if ( $fund_usage_from_date && $fund_usage_to_date ) {
				$from_date_object = WAL_Date_Time::get_gmt_date_time_object( $fund_usage_from_date ) ;
				$to_date_object   = WAL_Date_Time::get_gmt_date_time_object( $fund_usage_to_date ) ;
				$to_date_object->modify( '+1 days' ) ;

				if ( $from_date_object->getTimestamp() <= $current_date && $to_date_object->getTimestamp() >= $current_date ) {
					$return = true ;
				}
				//Validate the start date.
			} elseif ( $fund_usage_from_date ) {
				$from_date_object = WAL_Date_Time::get_gmt_date_time_object( $fund_usage_from_date ) ;

				if ( $from_date_object->getTimestamp() <= $current_date ) {
					$return = true ;
				}
			} elseif ( $fund_usage_to_date ) {
				$to_date_object = WAL_Date_Time::get_gmt_date_time_object( $fund_usage_to_date ) ;
				$to_date_object->modify( '+1 days' ) ;

				if ( $to_date_object->getTimestamp() >= $current_date ) {
					$return = true ;
				}
			}
			/**
			 * This hook is used to validate the fund usage date range.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_validate_fund_usage_date_range', $return, $this->wallet ) ;
		}

		/**
		 * Validate the specific weekdays restriction.
		 * 
		 * @return bool
		 */
		public function validate_specific_weekdays() {
			$return            = false ;
			$today             = gmdate( 'N', current_time( 'timestamp' ) ) ;
			$specific_weekdays = get_option( 'wal_general_wallet_usage_restrict_' . $today ) ;

			if ( 'yes' == $specific_weekdays ) {
				$return = true ;
			}
			/**
			 * This hook is used to validate the fund usage specific weekdays.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_validate_fund_usage_specific_weekdays', $return, $this->wallet ) ;
		}

		/**
		 * Validate the user restriction.
		 * 
		 * @return bool
		 */
		public function validate_user_restriction() {
			$return    = false ;
			$user_type = get_option( 'wal_general_wallet_usage_user_restriction_type' ) ;

			switch ( $user_type ) {
				case '2':
					$include_users = array_filter( ( array ) get_option( 'wal_general_wallet_usage_include_users' ), 'wal_array_filter' ) ;
					if ( ! in_array( $this->wallet->get_user_id(), $include_users ) ) {
						$return = true ;
					}

					break ;
				case '3':
					$exclude_users = array_filter( ( array ) get_option( 'wal_general_wallet_usage_exclude_users' ), 'wal_array_filter' ) ;
					if ( in_array( $this->wallet->get_user_id(), $exclude_users ) ) {
						$return = true ;
					}
					break ;
				case '4':
					$return             = true ;
					$include_user_roles = array_filter( ( array ) get_option( 'wal_general_wallet_usage_include_user_roles' ), 'wal_array_filter' ) ;

					// Logged-in user restriction
					if ( wal_check_is_array( $this->wallet->get_user()->roles ) ) {
						foreach ( $this->wallet->get_user()->roles as $role ) {
							if ( in_array( $role, $include_user_roles ) ) {
								$return = false ;
							}
						}
					}

					break ;
				case '5':
					$exclude_user_roles = array_filter( ( array ) get_option( 'wal_general_wallet_usage_exclude_user_roles' ), 'wal_array_filter' ) ;

					// Logged-in user restriction
					if ( wal_check_is_array( $this->wallet->get_user()->roles ) ) {
						foreach ( $this->wallet->get_user()->roles as $role ) {
							if ( in_array( $role, $exclude_user_roles ) ) {
								$return = true ;
							}
						}
					}

					break ;
			}
			/**
			 * This hook is used to validate the fund usage user.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_validate_fund_usage_user', $return, $this->wallet ) ;
		}

		/**
		 * Validate the security settings.
		 * 
		 * @return bool
		 */
		public function validate_security_settings() {
			$return = false ;

			if ( $this->validate_fund_usage_count_per_day() ) {
				$return = true ;
			} elseif ( $this->validate_fund_usage_total_per_day() ) {
				$return = true ;
			}
			/**
			 * This hook is used to validate the fund usage security settings.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_validate_fund_usage_security_settings', $return, $this->wallet ) ;
		}

		/**
		 * Validate the fund usage count per day.
		 * 
		 * @return bool
		 */
		public function validate_fund_usage_count_per_day() {
			$return = false ;
			$count  = get_option( 'wal_general_fund_usage_count_per_day' ) ;

			if ( $count && $count <= wal_customer_wallet_usage_count_per_day() ) {
				$return = true ;
			}
			/**
			 * This hook is used to validate the fund usage count per day.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_validate_fund_usage_count_per_day', $return, $this->wallet ) ;
		}

		/**
		 * Validate the fund usage total per day.
		 * 
		 * @return bool
		 */
		public function validate_fund_usage_total_per_day() {
			$return = false ;
			$total  = get_option( 'wal_general_fund_usage_max_amount_per_day' ) ;

			if ( $total && $total <= wal_customer_wallet_usage_total_per_day() ) {
				$return = true ;
			}
			/**
			 * This hook is used to validate the fund usage total per day.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_validate_fund_usage_total_per_day', $return, $this->wallet ) ;
		}
	}

}
