<?php

/**
 * Frontend Assets.
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}
if (!class_exists('WAL_Fronend_Assets')) {

	/**
	 * Class.
	 */
	class WAL_Fronend_Assets {

		/**
		 * Suffix.
		 * 
		 * @var string
		 */
		private static $suffix;

		/**
		 * Scripts.
		 *
		 * @since 4.1.0
		 * @var array
		 */
		private static $scripts = array();

		/**
		 * Styles.
		 *
		 * @since 4.1.0
		 * @var array
		 */
		private static $styles = array();

		/**
		 * Localized scripts.
		 *
		 * @since 4.1.0
		 * @var array
		 */
		private static $wp_localized_scripts = array();

		/**
		 * In Footer.
		 * 
		 * @var bool
		 */
		public static $in_footer = false;

		/**
		 * Class Initialization.
		 */
		public static function init() {

			self::$suffix = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

			// Enqueue script in footer.
			if ('2' == get_option('wal_advanced_frontend_enqueue_scripts_type')) {
				self::$in_footer = true;
			}

			add_action('wp_enqueue_scripts', array( __CLASS__, 'load_scripts' ));
			add_action('wp_print_scripts', array( __CLASS__, 'localize_printed_scripts' ), 5);
			add_action('wp_print_footer_scripts', array( __CLASS__, 'localize_printed_scripts' ), 5);
		}

		/**
		 * Get the default scripts to register.
		 * 
		 * @since 4.1.0
		 * @return array
		 */
		private static function get_default_scripts() {
			/**
			 * This hook is used to alter the default register scripts.
			 * 
			 * @since 4.1.0
			 */
			return apply_filters('wal_default_register_scripts',
					array(
						'lightcase' => array(
							'src' => self::get_asset_url('assets/lib/lightcase/lightcase' . self::$suffix . '.js'),
							'can_register' => true,
						),
						'wal-lightcase' => array(
							'src' => self::get_asset_url('assets/js/wal-lightcase-enhanced.js'),
							'can_register' => true,
							'deps' => array( 'lightcase' ),
						),
						'wal-frontend' => array(
							'src' => self::get_asset_url('assets/js/frontend/frontend.js'),
							'deps' => array( 'jquery-blockui' ),
							'can_register' => true,
						),
						'wal-enhanced' => array(
							'src' => self::get_asset_url('assets/js/wal-enhanced.js'),
							'can_register' => true,
							'deps' => array( 'select2' ),
						),
						'accounting' => array(
							'src' => self::get_asset_url('assets/js/accounting/accounting.js'),
							'can_register' => true,
						),
			));
		}

		/**
		 * Get the default styles to register.
		 * 
		 * @since 4.1.0
		 * @return array
		 */
		private static function get_default_styles() {
			/**
			 * This hook is used to alter the default register styles.
			 * 
			 * @since 4.1.0
			 */
			return apply_filters('wal_default_register_styles',
					array(
						'lightcase' => array(
							'src' => self::get_asset_url('assets/lib/lightcase/lightcase' . self::$suffix . '.css'),
							'can_register' => true,
						),
						'wal-frontend' => array(
							'src' => self::get_asset_url('assets/css/frontend.css'),
							'can_register' => true,
						),
						'wal-module' => array(
							'src' => self::get_asset_url('assets/css/module.css'),
							'can_register' => true,
						),
			));
		}

		/**
		 * Get script data.
		 * 
		 * @since 4.1.0
		 * @param string $handle
		 * @return array/false
		 */
		public static function get_script_data( $handle ) {
			switch ($handle) {
				case 'wal-frontend':
					$params = array(
						'ajax_url' => WAL_ADMIN_AJAX_URL,
						'pagination_nonce' => wp_create_nonce('wal-pagination-nonce'),
						'partial_fund_usage_nonce' => wp_create_nonce('wal-partial-fund-usage-nonce'),
						'pagination_action_nonce' => wp_create_nonce('wal-pagination-action-nonce'),
						'shipping_list_nonce' => wp_create_nonce('wal-shipping-list'),
						'currency_symbol' => get_woocommerce_currency_symbol(),
						'currency_num_decimals' => wc_get_price_decimals(),
						'currency_decimal_sep' => esc_attr(wc_get_price_decimal_separator()),
						'currency_thousand_sep' => esc_attr(wc_get_price_thousand_separator()),
						'currency_format' => esc_attr(str_replace(array( '%1$s', '%2$s' ), array( '%s', '%v' ), get_woocommerce_price_format())),
						'is_checkout' => is_checkout(),
						'is_gateway_discount_enabled' => get_option('wal_general_apply_discount_for_wallet_gateway'),
						'discount_nonce' => wp_create_nonce('wal-discount-nonce'),
					);
					break;

				case 'wal-enhanced':
					$params = array(
						'i18n_no_matches' => __('No matches found', 'wallet-for-woocommerce'),
						'i18n_input_too_short_1' => __('Please enter 1 or more characters', 'wallet-for-woocommerce'),
						'i18n_input_too_short_n' => __('Please enter %qty% or more characters', 'wallet-for-woocommerce'),
						'i18n_input_too_long_1' => __('Please delete 1 character', 'wallet-for-woocommerce'),
						'i18n_input_too_long_n' => __('Please delete %qty% characters', 'wallet-for-woocommerce'),
						'i18n_selection_too_long_1' => __('You can only select 1 item', 'wallet-for-woocommerce'),
						'i18n_selection_too_long_n' => __('You can only select %qty% items', 'wallet-for-woocommerce'),
						'i18n_load_more' => __('Loading more results&hellip;', 'wallet-for-woocommerce'),
						'i18n_searching' => __('Searching&hellip;', 'wallet-for-woocommerce'),
						'search_nonce' => wp_create_nonce('wal-search-nonce'),
						'ajaxurl' => WAL_ADMIN_AJAX_URL,
						);
					break;

				default:
					$params = false;
					break;
			}
			
			return $params;
		}

		/**
		 * Register and enqueue frontend scripts.
		 * 
		 * @since 4.1.0
		 */
		public static function load_scripts() {
			global $post;

			self::register_scripts();
			self::register_styles();
			
			// Enqueue scripts in cart & checkout and block cart & checkout page.
			if ( is_account_page() || is_cart() || is_checkout() || wal_is_block_cart() || wal_is_block_checkout()) {
				self::enqueue_registered_scripts();
				self::enqueue_registered_styles();
			}

			// Enqueue scripts in short code page.
			if (is_object($post) && !empty($post->post_content) && strstr($post->post_content, '[wal_')) {
				self::enqueue_registered_scripts();
				self::enqueue_registered_styles();
			}

			self::add_inline_style();
		}

		/**
		 * Localize scripts only when enqueued.
		 * 
		 * @since 4.1.0
		 */
		public static function localize_printed_scripts() {
			foreach (self::$scripts as $handle) {
				self::localize_script($handle);
			}
		}

		/**
		 * Register all scripts.
		 * 
		 * @since 4.1.0
		 */
		private static function register_scripts() {
			$default_scripts = self::get_default_scripts();
			// Returns if there is no scripts to register.
			if (!wal_check_is_array($default_scripts)) {
				return;
			}

			foreach ($default_scripts as $handle => $script) {
				if (!isset($script['src'])) {
					continue;
				}

				if (!$script['can_register']) {
					continue;
				}

				$deps = isset($script['deps']) ? array_merge(array( 'jquery' ), $script['deps']) : array( 'jquery' );
				$version = isset($script['version']) ? $script['version'] : WAL_VERSION;
				$in_footer = isset($script['in_footer']) ? $script['in_footer'] : self::$in_footer;
				if (!wp_register_script($handle, $script['src'], $deps, $version, $in_footer)) {
					continue;
				}

				self::$scripts[] = $handle;
			}
		}

		/**
		 * Register all styles.
		 * 
		 * @since 4.1.0
		 */
		private static function register_styles() {
			$default_styles = self::get_default_styles();
			// Returns if there is no styles to register.
			if (!wal_check_is_array($default_styles)) {
				return;
			}
			
			foreach ($default_styles as $handle => $style) {
				if (!isset($style['src'])) {
					continue;
				}

				if (!$style['can_register']) {
					continue;
				}
				
				$deps = isset($style['deps']) ? $style['deps'] : array();
				$version = isset($style['version']) ? $style['version'] : WAL_VERSION;
				$media = isset($style['media']) ? $style['media'] : 'all';
				$has_rtl = isset($style['has_rtl']) ? $style['has_rtl'] : false;
				if (!wp_register_style($handle, $style['src'], $deps, $version, $media)) {
					continue;
				}
				
				self::$styles[] = $handle;
				
				if ($has_rtl) {
					wp_style_add_data($handle, 'rtl', 'replace');
				}
			}
		}

		/**
		 * Enqueue all registered scripts.
		 * 
		 * @since 4.1.0
		 */
		private static function enqueue_registered_scripts() {
			foreach (self::$scripts as $handle) {
				self::enqueue_script($handle);
			}
		}

		/**
		 * Enqueue script.
		 * 
		 * @param string $handle
		 * @since 4.1.0
		 */
		private static function enqueue_script( $handle ) {
			if (!wp_script_is($handle, 'registered')) {
				return;
			}

			wp_enqueue_script($handle);
		}

		/**
		 * Enqueue all registered styles.
		 * 
		 * @since 4.1.0
		 */
		private static function enqueue_registered_styles() {
			foreach (self::$styles as $handle) {
				self::enqueue_style($handle);
			}
		}

		/**
		 * Enqueue style.
		 * 
		 * @param string $handle
		 * @since 4.1.0
		 */
		private static function enqueue_style( $handle ) {
			if (!wp_style_is($handle, 'registered')) {
				return;
			}

			wp_register_style('wal-inline-style', false, array(), WAL_VERSION); // phpcs:ignore
			wp_enqueue_style('wal-inline-style');

			wp_enqueue_style($handle);
		}

		/**
		 * Localize the enqueued script.
		 * 
		 * @since 4.1.0
		 * @param string $handle
		 * @return null
		 */
		private static function localize_script( $handle ) {
			// Return if already localized script or not enqueued script.
			if (in_array($handle, self::$wp_localized_scripts, true) || !wp_script_is($handle)) {
				return;
			}
			
			// Get the data for current script.
			$data = self::get_script_data($handle);
			
			if (!$data) {
				return;
			}

			$name = str_replace('-', '_', $handle) . '_params';

			/**
			 * This hook is used to alter the script data.
			 * 
			 * @since 4.1.0
			 */
			if (wp_localize_script($handle, $name, apply_filters($name, $data))) {
				self::$wp_localized_scripts[] = $handle;
			}
		}

		/**
		 * Get asset URL.
		 *
		 * @since 4.1.0
		 * @param string $path Assets path.
		 * @return string
		 */
		private static function get_asset_url( $path ) {
			/**
			 * This hook is used to alter the asset URL.
			 * 
			 * @since 4.1.0
			 */
			return apply_filters('wal_get_asset_url', WAL_PLUGIN_URL . '/' . $path, $path);
		}

		/**
		 * Add Inline style.
		 */
		public static function add_inline_style() {
			$contents = get_option('wal_advanced_custom_css', '');
			if (!$contents) {
				return;
			}

			//Add custom css as inline style.
			wp_add_inline_style('wal-inline-style', $contents);
		}
	}

	WAL_Fronend_Assets::init();
}
