<?php

/**
 *  Handles the Dashboard.
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Dashboard_Handler')) {

	/**
	 * Class.
	 * */
	class WAL_Dashboard_Handler {

		/**
		 * Error messages.
		 * 
		 * @var array
		 */
		private static $errors = array();

		/**
		 * Info messages.
		 * 
		 * @var array
		 */
		private static $infos = array();

		/**
		 * Success messages.
		 * 
		 * @var array
		 */
		private static $messages = array();

		/**
		 * Add a message.
		 */
		public static function add_message( $text ) {
			self::$messages[] = $text;
		}

		/**
		 * Add an error.
		 */
		public static function add_error( $text ) {
			self::$errors[] = $text;
		}

		/**
		 * Add an info.
		 */
		public static function add_info( $text ) {
			self::$infos[] = $text;
		}

		/**
		 * Output messages + errors.
		 */
		public static function show_messages() {
			if (count(self::$errors) > 0) {
				foreach (self::$errors as $error) {
					self::show_error($error);
				}
			} elseif (count(self::$messages) > 0) {
				foreach (self::$messages as $message) {
					self::show_message($message);
				}
			}

			foreach (self::$infos as $info) {
				self::show_info($info);
			}

			self::$infos = array();
			self::$errors = array();
			self::$messages = array();
		}

		/**
		 * Output a message.
		 */
		public static function show_message( $message ) {
			wc_print_notice($message, 'success');
		}

		/**
		 * Output a error.
		 */
		public static function show_error( $error ) {
			wc_print_notice($error, 'error');
		}

		/**
		 * Output a info.
		 */
		public static function show_info( $info ) {
			wc_print_notice($info, 'notice');
		}

		/**
		 * Class Initialization.
		 * */
		public static function init() {

			//Add the custom rewrite endpoint.
			add_action('init', array( __CLASS__, 'custom_rewrite_endpoint' ));
			//Flush rewrite rules.
			add_action('wp_loaded', array( __CLASS__, 'flush_rewrite_rules' ));
			//Add the custom query vars.
			add_filter('query_vars', array( __CLASS__, 'custom_query_vars' ), 0);
			//Add the hook for overview menu content.
			add_action('wal_dashboard_menu_content_overview', array( __CLASS__, 'render_overview' ));
			//Add the hook for activity menu content.
			add_action('wal_dashboard_menu_content_activity', array( __CLASS__, 'render_activity' ));
			//Add the hook for topup menu content.
			add_action('wal_dashboard_menu_content_topup', array( __CLASS__, 'render_topup' ));
		}

		/**
		 * Add the custom rewrite endpoint.
		 * 
		 * @return void
		 * */
		public static function custom_rewrite_endpoint() {
			foreach (wal_wallet_endpoints() as $endpoint) {
				add_rewrite_endpoint($endpoint, EP_ROOT | EP_PAGES);
			}
		}

		/**
		 * Flush rewrite rules.
		 * 
		 * @return void.
		 * */
		public static function flush_rewrite_rules() {
			flush_rewrite_rules();
		}

		/**
		 * Add the custom query variable.
		 * 
		 * @return array.
		 * */
		public static function custom_query_vars( $query_vars ) {
			foreach (wal_wallet_endpoints() as $endpoint) {
				$query_vars[] = $endpoint;
			}

			return $query_vars;
		}

		/**
		 * Render the dashboard overview menu content.
		 * 
		 * @return void
		 */
		public static function render_overview() {
			wal_get_template('dashboard/overview.php');
		}

		/**
		 * Render the dashboard activity menu content.
		 * 
		 * @return void
		 */
		public static function render_activity() {
			$transaction_log_ids = wal_user_transaction_logs();
			$post_per_page = wal_transaction_logs_pagination_count();

			$current_page = empty($current_page) ? 1 : absint($current_page);
			$offset = ( $post_per_page * $current_page ) - $post_per_page;
			$page_count = ceil(count($transaction_log_ids) / $post_per_page);
			$pagination_range = wal_transaction_logs_pagination_range();

			$table_args = array(
				'transaction_log_ids' => array_slice($transaction_log_ids, $offset, $post_per_page),
				'pagination' => array(
					'page_count' => $page_count,
					'current_page' => $current_page,
					'pagination_range' => $pagination_range,
					'prev_page_count' => ( ( $current_page - 1 ) == 0 ) ? ( $current_page ) : ( $current_page - 1 ),
					'next_page_count' => ( ( $current_page + 1 ) <= ( $page_count ) ) ? ( $current_page + 1 ) : ( $current_page ),
					'prev_page_row' => ( ( $current_page - $pagination_range ) > 0 ) ? $current_page - $pagination_range : $current_page,
					'next_page_row' => ( ( $current_page + $pagination_range ) <= $page_count ) ? $current_page + $pagination_range : $current_page,
				),
			);

			wal_get_template('dashboard/activity-wrapper.php', $table_args);
		}

		/**
		 * Render the dashboard top up menu content.
		 * 
		 * @return void
		 */
		public static function render_topup() {
			wal_get_template('dashboard/topup-form.php');
		}
	}

	WAL_Dashboard_Handler::init();
}
