<?php

/**
 *  Handles the Checkout.
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Checkout_Handler')) {

	/**
	 * Class.
	 * */
	class WAL_Checkout_Handler {

		/**
		 * Class Initialization.
		 * */
		public static function init() {
			// May be render the redeem wallet fund form in the checkout.
			add_action('woocommerce_before_checkout_form', array( __CLASS__, 'render_checkout_notice' ), 20);
			//Handle the payment gateways.
			add_filter('woocommerce_available_payment_gateways', array( __CLASS__, 'handle_payment_gateways' ), 10, 1);
			//Handle the topup payment gateways.
			add_filter('woocommerce_available_payment_gateways', array( __CLASS__, 'handle_topup_payment_gateways' ), 10, 1);
			// Validate the checkout postdata.
			add_action('woocommerce_after_checkout_validation', array( __CLASS__, 'validate_checkout_postdata' ), 10, 2);
			// Maybe add wallet details in the order review fragments.
			add_filter('woocommerce_update_order_review_fragments', array( __CLASS__, 'maybe_add_wallet_details_in_order_review_fragments' ), 10, 1);
		}

		/**
		 * Render Checkout Notices.
		 * 
		 * @since 4.6.0
		 */
		public static function render_checkout_notice() {
			self::render_checkout_redeem_wallet_fund_form();

			self::render_discount_notice_for_gateway();
		}

		/**
		 * May be add wallet details in the order review fragments.
		 * 
		 * @since 4.0.0
		 * @param array $fragments
		 * @return array
		 */
		public static function maybe_add_wallet_details_in_order_review_fragments( $fragments ) {
			$fragments = wal_check_is_array($fragments) ? $fragments : array();

			$fragments['wal_redeem_wallet_fund_form_html'] = wal_get_checkout_redeem_wallet_fund_form_html();
			$fragments['wal_discount_notice_for_gateway'] = wal_get_discount_notice_for_gateway();

			return $fragments;
		}

		/**
		 * May be render the redeem wallet fund form in the checkout.
		 * 
		 * @since 1.0.0
		 * */
		public static function maybe_render_redeem_wallet_fund_form() {
			// Return if the redeem wallet fund form is not valid to display.
			if (!wal_can_render_redeem_wallet_fund_form_in_checkout()) {
				return '';
			}

			if ( ! WAL_Current_User_Wallet::get_balance() ) {
				$notice_msg = "You don't have any funds in your wallet.";
				echo wp_kses_post( wal_get_template_html( 'error-notice.php', array( 'notice_msg' => $notice_msg, 'class_name' => '' ) ) );
				return;
			}

			wal_get_template('checkout-partial-fund.php');
		}

		/**
		 * Render the checkout redeem wallet fund form.
		 * 
		 * @since 4.0.0
		 * */
		public static function render_checkout_redeem_wallet_fund_form() {
			echo '<div id="wal-redeem-wallet-fund-forms-wrapper">';
			self::maybe_render_redeem_wallet_fund_form();
			echo '</div>';
		}

		/**
		 * May be render the discount notice for wallet gateway in the checkout.
		 * 
		 * @since 4.6.0
		 * */
		public static function maybe_render_discount_notice_for_wallet_gateway() {
			if ('yes' != get_option('wal_general_apply_discount_for_wallet_gateway')) {
				return;
			}

			if (wal_cart_topup_product_exists()) {
				return;
			}

			$discount_value = get_option('wal_general_discount_value');

			$discount_type = get_option('wal_general_discount_type');
			if ('2' == $discount_type) {
				$discount_value = ( ( $discount_value / 100 ) * wal_get_wc_cart_subtotal() );
			} else if ('3' == $discount_type) {
				$discount_value = ( ( $discount_value / 100 ) * wal_get_wc_cart_total() );
			}

			if (empty($discount_value)) {
				return;
			}

			$notice_msg = get_option('wal_messages_notice_for_discount');
			$notice_msg = str_replace('{discount_value}', wal_price($discount_value) , $notice_msg);

			wal_get_template('discount-notice.php', array( 'notice_msg' => $notice_msg ));
		}

		/**
		 * Render the checkout redeem wallet fund form.
		 * 
		 * @since 4.0.0
		 * */
		public static function render_discount_notice_for_gateway() {           
			echo '<div id="wal-discount-notice-for-gateway-wrapper">';
			self::maybe_render_discount_notice_for_wallet_gateway();
			echo '</div>';
		}

		/**
		 * Handles the payment gateways.
		 * 
		 * @return array.
		 */
		public static function handle_payment_gateways( $wc_gateways ) {
			if (is_null(WC()->cart) || wal_cart_topup_product_exists()) {
				return $wc_gateways;
			}

			if ( ! isset($wc_gateways['wal_wallet']) ) {
				return $wc_gateways;
			}
			
			// Return if the partial fund already applied.
			if (is_object(WC()->session) && WC()->session->get('wal_partial_fund')) {
				// Unset the wallet gateway.
				unset($wc_gateways['wal_wallet']);

				return $wc_gateways;
			}

			// Wallet having valid fund usage.
			if (WAL_Current_User_Wallet::is_valid_fund_usage()) {
				// Hide the other payment gateways.
				if ('yes' == get_option('wal_general_hide_other_wc_payment_gateways')) {
					$wallet_gateway = array( 'wal_wallet' => $wc_gateways['wal_wallet'] );
					unset($wc_gateways);

					$wc_gateways = $wallet_gateway;
				}
			} else {
				// Unset the wallet gateway, If the wallet not valid fund usage.
				unset($wc_gateways['wal_wallet']);
			}
						
			if ( ( 'yes' === get_option( 'wal_general_hide_wallet_payment_gateway_in_checkout' , 'no' ) ) && ( wal_convert_price(WAL_Current_User_Wallet::get_balance()) < wal_get_wc_cart_total() ) ) {
				unset( $wc_gateways[ 'wal_wallet' ] ) ;
			}

			return $wc_gateways;
		}

		/**
		 * Handles the topup payment gateways.
		 * 
		 * @return array.
		 */
		public static function handle_topup_payment_gateways( $wc_gateways ) {
			if (is_null(WC()->cart) || !wal_cart_topup_product_exists()) {
				return $wc_gateways;
			}

			// Remove the wallet payment gateway when purchasing the topup product.
			if (array_key_exists('wal_wallet', $wc_gateways)) {
				unset($wc_gateways['wal_wallet']);
			}

			$restricted_gateways = array_filter((array) get_option('wal_general_topup_restrict_payment_gateways'), 'wal_array_filter');
			if (!wal_check_is_array($restricted_gateways)) {
				return $wc_gateways;
			}

			foreach ($restricted_gateways as $gateway_id) {
				if (!isset($wc_gateways[$gateway_id])) {
					continue;
				}

				unset($wc_gateways[$gateway_id]); //unset payment gateways
			}

			return $wc_gateways;
		}

		/**
		 * Validate the checkout post data.
		 *
		 * @since 3.5.0
		 * @param  array $posted_data  An array of posted data.
		 * @param  WP_Error $errors Validation errors.
		 */
		public static function validate_checkout_postdata( $posted_data, $errors ) {
			if (!is_object(WC()->session)) {
				return;
			}

			if ('wal_wallet' !== WC()->session->get('chosen_payment_method')) {
				return;
			}
			/**
			 * This hook is used to alter the payment gateway order total.
			 *
			 * @param float WC()->cart->total
			 * @since 3.5.0 
			 */
			$total = apply_filters('wal_payment_gateway_order_total', WC()->cart->total);
			/**
			 * This hook is used to alter the payment gateway user balance.
			 *
			 * @param float WAL_Current_User_Wallet::get_balance()
			 * @since 3.5.0
			 */
			$balance = apply_filters('wal_payment_gateway_user_balance', WAL_Current_User_Wallet::get_balance());

			// Throw the error if the wallet amount less than order total. 
			if ($balance < $total) {
				$top_up = sprintf('<a href=%s>%s</a>', wal_dashboard_menu_endpoint_url('topup'), __('top-up', 'wallet-for-woocommerce'));
				/* translators: %s - topup */
				wal_add_wc_notice(sprintf(__('You don"t have sufficient funds in your wallet. Please %s the funds to complete this order.', 'wallet-for-woocommerce'), $top_up), 'error');
			}

			// Return if the current wallet not valid fund usage.
			if (!WAL_Current_User_Wallet::is_valid_fund_usage()) {
				wal_add_wc_notice(esc_html__('Sorry, You are not valid to use fund.', 'wallet-for-woocommerce'), 'error');
			}
		}
	}

	WAL_Checkout_Handler::init();
}
