<?php

/**
 *  Handles the Cart.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Cart_Handler' ) ) {

	/**
	 * Class.
	 * */
	class WAL_Cart_Handler {

		/**
		 * Class Initialization.
		 * */
		public static function init() {
			// May be render the redeem wallet fund form in the cart.
			add_action( 'woocommerce_after_cart_table', array( __CLASS__, 'maybe_render_redeem_wallet_fund_form' ), 20 );
			// Add error notice for wallet fund form in the cart.
			add_action( 'woocommerce_before_cart', array( __CLASS__, 'maybe_add_notice_for_partial_usage' ), 5 );
			// Add error notice for wallet fund form in the cart.
			add_action( 'woocommerce_before_checkout_form', array( __CLASS__, 'maybe_add_notice_for_wallet_fund_form' ), 20 );
			// May be render the auto deduct wallet balance in the cart.
			add_action( 'wp', array( __CLASS__, 'maybe_render_auto_deduct_wallet' ), 20 );
			// Calculate the custom fees.
			add_action( 'woocommerce_cart_calculate_fees', array( __CLASS__, 'custom_fees' ) );
			// Calculate the discount for wallet gateway.
			add_action( 'woocommerce_cart_calculate_fees', array( __CLASS__, 'discount_for_wallet_gateway' ) );
			// Add Gateway Fee in Cart/Checkout.
			add_action( 'woocommerce_blocks_loaded', array( __CLASS__, 'register_wc_blocks' ), 10 );
			// Remove the custom fees.
			add_filter( 'woocommerce_calculated_total', array( __CLASS__, 'remove_custom_fees' ), 10, 2 );
			// Add the custom cart totals fee html.
			add_filter( 'woocommerce_cart_totals_fee_html', array( __CLASS__, 'maybe_add_custom_cart_totals_fee_html' ), 10, 2 );
			// Restrict the apply coupon discount when the topup products only exists in the cart.
			add_filter( 'woocommerce_coupon_is_valid', array( __CLASS__, 'restrict_apply_coupon' ), 10, 3 );
			// Restrict the coupon discount for the topup products.
			add_filter( 'woocommerce_coupon_get_discount_amount', array( __CLASS__, 'restrict_topup_product_coupon_discount' ), 9999, 5 );
			// Restrict the shipping for the topup products.
			add_filter('woocommerce_shipping_packages' , array( __CLASS__, 'remove_shipping_for_topup' ) , 999);
			// Validate the other product add to cart when topup product in the cart.
			add_filter( 'woocommerce_add_to_cart_validation', array( __CLASS__, 'validate_other_product_add_to_cart' ), 10 );
			// Alter product price based on top-up amount.
			add_action( 'woocommerce_get_cart_item_from_session', array( __CLASS__, 'set_price' ), 10, 3 );
			// Alter product price based on top-up amount.
			add_filter( 'woocommerce_product_get_price', array( __CLASS__, 'set_price_from_session' ), 10, 2 );
			// Maybe alert cart item quantity HTML.
			add_filter( 'woocommerce_cart_item_quantity', array( __CLASS__, 'maybe_alert_cart_item_quantity_html' ), 10, 3 );
			// Validate the cart items.
			add_action( 'woocommerce_check_cart_items', array( __CLASS__, 'validate_cart_items' ), 1 );
			// Remove the tax for wallet usage fees.
			add_filter( 'woocommerce_cart_totals_get_fees_from_cart_taxes', array( __CLASS__, 'remove_tax_wallet_usage_fees' ), 10, 3 );
			// Calculate Fund after product removed from cart.
			add_action( 'woocommerce_cart_item_removed', array( __CLASS__, 'recalculate_fund_after_product_removed' ), 10, 3 );
		}

		/**
		 * May be render the redeem wallet fund form in the cart.
		 *
		 * @since 1.0.0
		 * */
		public static function maybe_render_redeem_wallet_fund_form() {
			// Return if the redeem wallet fund form is not valid to display.
			if ( ! wal_can_render_redeem_wallet_fund_form_in_cart() ) {
				return;
			}

			if ( ! WAL_Current_User_Wallet::get_balance() ) {
				$notice_msg = "You don't have any funds in your wallet.";
				echo wp_kses_post( wal_get_template_html( 'error-notice.php', array( 'notice_msg' => $notice_msg, 'class_name' => '' ) ) );
				return;
			}

			wal_get_template( 'cart-partial-fund.php' );
		}

		/**
		 * Add error notice for wallet fund form in the cart.
		 *
		 * @since 4.4.0
		 * */
		public static function maybe_add_notice_for_partial_usage() {
			if ( 'yes' != get_option( 'wal_general_allow_partial_payments' ) ) {
				return;
			}

			$user_id = get_current_user_id();

			$max_count_per_day = floatval( get_option( 'wal_general_fund_usage_count_per_day' ) ) ;
			$top_up_count_per_day = wal_customer_wallet_usage_count_per_day( $user_id );
			if ( $max_count_per_day && $max_count_per_day <= $top_up_count_per_day ) {
				$count_err_message = get_option('wal_messages_partial_fund_usage_limit_count');
				wc_add_notice($count_err_message, 'error');
			}

			$max_total_per_day = floatval( get_option( 'wal_general_fund_usage_max_amount_per_day' ) ) ;
			$top_total_per_day = wal_customer_wallet_usage_total_per_day( $user_id );
			if ( $max_total_per_day && $max_total_per_day <= $top_total_per_day ) {
				$spent_err_message = get_option('wal_messages_partial_fund_usage_spent_value');
				wc_add_notice($spent_err_message, 'error');
			}
		}

		/**
		 * Add error notice for wallet fund form in the cart.
		 *
		 * @since 4.4.0
		 * */
		public static function maybe_add_notice_for_wallet_fund_form() {
			$user_id = get_current_user_id();

			$max_count_per_day = floatval( get_option( 'wal_general_fund_usage_count_per_day' ) ) ;
			$top_up_count_per_day = wal_customer_wallet_usage_count_per_day( $user_id );
			if ( $max_count_per_day && $max_count_per_day <= $top_up_count_per_day ) {
				$limit_count_error_message = get_option('wal_messages_wallet_gateway_usage_limit_count');
				wc_add_notice($limit_count_error_message, 'error');
			}

			$max_total_per_day = floatval( get_option( 'wal_general_fund_usage_max_amount_per_day' ) ) ;
			$top_total_per_day = wal_customer_wallet_usage_total_per_day( $user_id );
			if ( $max_total_per_day && $max_total_per_day <= $top_total_per_day ) {
				$spent_value_error_message = get_option('wal_messages_wallet_usage_spent_value');
				wc_add_notice($spent_value_error_message, 'error');
			}
		}

		/**
		 * AUto deduct wallet amount in cart.
		 *
		 * @return void
		 */
		public static function maybe_render_auto_deduct_wallet() {

			if ('yes' != get_option('wal_general_allow_partial_payments')) {
				return;
			}

			if ('yes' != get_option('wal_general_auto_deduct_wallet_for_partial_payments')) {
				return;
			}

			if ( ( 'yes' == WC()->session->get('wal_auto_deduct_partial_fund') ) || ( 'no' == WC()->session->get('wal_auto_deduct_partial_fund') )) {
				return;
			}

			// Return if the current wallet not valid fund usage.
			if (!WAL_Current_User_Wallet::is_valid_fund_usage()) {
				return;
			}

			$wallet_balance = WAL_Current_User_Wallet::get_balance();
			$maximum_amount = wal_get_maximum_fund_usage_limit();
			// Return if the fund is greater than maximum amount.
			if (false !== $maximum_amount && $maximum_amount < $wallet_balance) {
				return;
			}

			$cart_total = WAL_Current_User_Wallet::get_fund_to_apply(wal_get_wc_cart_total());

			if ( empty($cart_total) ) {
				wal_add_wc_notice(get_option('wal_messages_product_filter_error_for_usage'));
				return;
			}

			// Return if the wallet balance is less than order total.
			if ('2' != get_option('wal_general_partial_payments_mode') && $wallet_balance >= $cart_total) {
				return;
			}
			
			// Return if the fund is greater than cart total.
			if (wal_get_wc_cart_total() < $wallet_balance) {
				WC()->session->set('wal_partial_fund', $cart_total);
			} else {
				WC()->session->set('wal_partial_fund', $wallet_balance);
			}

			wal_add_wc_notice(get_option('wal_messages_partial_fund_added'));

			WC()->session->set('wal_auto_deduct_partial_fund', 'yes');
		}

		/**
		 * Register & Apply Gateway Fee for WooCommerce Blocks
		 *
		 * @since 5.5.0
		 */
		public static function register_wc_blocks() {
			woocommerce_store_api_register_update_callback(
				array( 
					'namespace' => 'wal-add-gateway-discount',
					'callback'  => function ( $data ) {
						self::add_gateway_discount_for_wc_blocks( $data );
					},
				)
			);
		}

		/**
		 * Add Gateway Discount for WooCommerce Blocks
		 *
		 * @since 5.5.0
		 */
		public static function add_gateway_discount_for_wc_blocks( $data ) {

			if ('add-discount' != $data['action']) {
				return;
			}

			if (empty($data['gateway_id'])) {
				WC()->session->__unset( 'wal_gateway_id' );
				return;
			}

			if ( 'wal_wallet' != $data['gateway_id'] ) {
				WC()->session->__unset( 'wal_gateway_id' );
				return;
			}

			WC()->session->set( 'wal_gateway_id', $data['gateway_id'] );
		}

		/**
		 * Add the custom fees.
		 *
		 * @return void
		 */
		public static function discount_for_wallet_gateway() {
			// Return if the session object is not initialized.
			if ( ! is_object( WC()->session ) ) {
				return;
			}

			if ('yes' != get_option('wal_general_apply_discount_for_wallet_gateway')) {
				return;
			}

			$gateway_id = WC()->session->get( 'wal_gateway_id' );
			$gateway_id = empty( $gateway_id ) ? WC()->session->get( 'chosen_payment_method' ) : $gateway_id;
			if ( empty( $gateway_id ) ) {
				return;                                                                                                                                                 
			}

			if ( 'wal_wallet' != $gateway_id ) {
				return;
			}

			$discount_value = get_option('wal_general_discount_value');

			$discount_type = get_option('wal_general_discount_type');
			if ('2' == $discount_type) {
				$discount_value = ( ( $discount_value / 100 ) * wal_get_wc_cart_subtotal() );
			} else if ('3' == $discount_type) {
				$discount_value = ( ( $discount_value / 100 ) * wal_get_wc_cart_total() );
			}

			if ( empty( $discount_value ) ) {
				return;
			}
			

			WC()->cart->fees_api()->add_fee(
				array(
					'id'          => 'wal_gateway_discount',
					'name'         => get_option('wal_localization_wallet_discount_label', 'Wallet Discount'),
					'amount'       => '-' . $discount_value,
					'taxable'      => false,
					'wal-mode'     => 'redeem',
					'wal-negative' => true,
					'wal-amount'   => '150',
				)
			);

			$discount_msg = get_option('wal_messages_success_msg_for_discount');
			wal_add_wc_notice($discount_msg, 'success');
		}

		/**
		 * Add the custom fees.
		 *
		 * @return void
		 */
		public static function custom_fees() {
			// Return if the session object is not initialized.
			if ( ! is_object( WC()->session ) ) {
				return;
			}

			// Return if the partial fund not apply in the cart.
			if ( ! WC()->session->get( 'wal_partial_fund' ) ) {
				return;
			}

			try {
				// Return if the partial fund is disabled.
				if ( 'yes' != get_option( 'wal_general_allow_partial_payments' ) ) {
					throw new Exception( '' );
				}

				// Return if the current wallet not valid fund usage.
				if ( ! WAL_Current_User_Wallet::is_valid_fund_usage() ) {
					throw new Exception( '' );
				}

				// Return if the topup product exists.
				if ( wal_cart_topup_product_exists() ) {
					throw new Exception( '' );
				}

				// Add fee.
				WC()->cart->fees_api()->add_fee(
					array(
						'id'          => WAL()->wallet_fee_name(),
						'name'         => wal_partial_usage_fee_title_label(),
						'amount'       => '-' . WC()->session->get( 'wal_partial_fund' ),
						'taxable'      => false,
						'wal-mode'     => 'redeem',
						'wal-negative' => true,
						'wal-amount'   => WC()->session->get( 'wal_partial_fund' ),
					)
				);
			} catch ( Exception $ex ) {
				// Remove the partial fund in the WC session.
				WC()->session->set( 'wal_partial_fund', null );
			}
		}

		/**
		 * Remove the custom fees.
		 *
		 * @return float
		 */
		public static function remove_custom_fees( $total, $cart ) {
			// Return if the session object is not initialized.
			if ( ! is_object( WC()->session ) ) {
				return $total;
			}

			// Return if the partial fund not apply in the cart.
			if ( ! WC()->session->get( 'wal_partial_fund' ) ) {
				return $total;
			}

			// Return if the wallet balance is less than order total.
			$cart_total = $total + floatval( WC()->session->get( 'wal_partial_fund' ) );
			if ( '2' === get_option( 'wal_general_partial_payments_mode' ) || wal_convert_price( WAL_Current_User_Wallet::get_balance() ) < $cart_total ) {
				return $total;
			}

			// Remove the partial fund in the WC session.
			WC()->session->set( 'wal_partial_fund', null );

			return $cart_total;
		}

		/**
		 * May be add custom cart totals fee html.
		 *
		 * @return string
		 */
		public static function maybe_add_custom_cart_totals_fee_html( $cart_totals_fee_html, $fee ) {
			if ( ! isset( $fee->id ) ) {
				return $cart_totals_fee_html;
			}

			if ( WAL()->wallet_fee_name() != $fee->id ) {
				return $cart_totals_fee_html;
			}

			if ( is_checkout() ) {
				$class_name = 'wal-checkout-remove-partial-fund';
			} else {
				$class_name = 'wal-cart-remove-partial-fund';
			}

			$cart_totals_fee_html .= ' <a href="javascript:void(0)" class="' . $class_name . '">' . wal_get_partial_usage_remove_button_label() . '</a>';

			return $cart_totals_fee_html;
		}

		/**
		 * Restrict the apply coupon discount when the top-up products only exists in the cart.
		 *
		 * @param bool   $bool
		 * @param object $coupon
		 * @param float  $discount
		 * @since 2.0
		 *
		 * @return type
		 * @throws Exception
		 */
		public static function restrict_apply_coupon( $bool, $coupon, $discount ) {
			// Return if the coupon discount restriction is not enabled.
			if ( 'yes' != get_option( 'wal_general_disable_topup_product_coupon_discount' ) ) {
				return $bool;
			}

			// throw the error if the top-up products only exists in the cart
			if ( wal_topup_products_only_exists_in_cart() ) {
				throw new Exception( wp_kses_post( get_option( 'wal_messages_wallet_topup_product_coupon_discount_restriction' ) ) );
			}

			return $bool;
		}

		/**
		 * Restrict the coupon discount for the topup product.
		 *
		 * @param float   $discount_amount
		 * @param float   $price_to_discount
		 * @param array   $cart_item
		 * @param boolean $single
		 * @param object  $coupon
		 *
		 * @since 2.0
		 *
		 * @return float
		 */
		public static function restrict_topup_product_coupon_discount( $discount_amount, $price_to_discount, $cart_item, $single, $coupon ) {
			// Return if the current product is not a topup product.
			if ( ! isset( $cart_item['wal_topup'] ) ) {
				return $discount_amount;
			}

			// Return if the coupon discount restriction is not enabled.
			if ( 'yes' != get_option( 'wal_general_disable_topup_product_coupon_discount' ) ) {
				return $discount_amount;
			}

			return 0;
		}

		/**
		 * Remove shipping for top up product.
		 * 
		 * @since 4.7.0
		 * @return array()
		 * */
		public static function remove_shipping_for_topup( $packages ) {
			if (wal_cart_topup_product_exists()) {
				return array();
			}

			return $packages;
		}

		/**
		 * Validate the other product add to cart when topup product in the cart.
		 *
		 * @return bool.
		 */
		public static function validate_other_product_add_to_cart( $passed ) {
			if ( ! wal_cart_topup_product_exists() ) {
				return $passed;
			}

			wal_add_wc_notice( get_option( 'wal_messages_wallet_topup_validate_other_product_cart' ), 'error' );

			return false;
		}

		/**
		 * Alter product price based on top-up amount.
		 *
		 * @return array.
		 */
		public static function set_price( $session_data, $values, $key ) {
			// Return if the current product is not a top-up product.
			if ( ! isset( $session_data['wal_topup'] ) ) {
				return $session_data;
			}

			if ( ! is_object( $session_data['data'] ) ) {
				return $session_data;
			}
			/**
			 * This hook is used to alter the top-up product cart price.
			 *
			 * @since 1.0
			 */
			$price = apply_filters( 'wal_topup_product_cartitem_price', $session_data['wal_topup']['price'], $session_data['data'] );

			$session_data['data']->set_price( $price );

			return $session_data;
		}

		/**
		 * Alter product price based on top-up amount.
		 *
		 * @return array.
		 */
		public static function set_price_from_session( $price, $product ) {
			if ('yes' != get_option('wal_general_troubleshoot_for_topup_product')) {
				return $price;
			}

			// Return if the current product is not a top-up product.
			$fund = isset( $_REQUEST[ 'wal-topup-form-amount' ] ) ? wc_clean( wp_unslash( $_REQUEST[ 'wal-topup-form-amount' ] ) ) : WC()->session->get('wal_fund_amount') ;
			// Return if the fund is empty.
			if ( ! $fund ) {
				return $price ;
			}

			// Return if the fund is not numeric.
			if ( ! is_numeric( $fund ) ) {
				return $price ;
			}

			$fund = wal_convert_price( $fund , true );

			return $fund;
		}

		/**
		 * Maybe alert cart item quantity HTML.
		 *
		 * @since 3.7.0
		 * @param int    $quantity
		 * @param string $cart_item_key
		 * @param array  $cart_item
		 * @return int
		 */
		public static function maybe_alert_cart_item_quantity_html( $quantity, $cart_item_key, $cart_item ) {
			// Omit it if the current cart item is not topup cart item.
			if ( ! isset( $cart_item['wal_topup'] ) ) {
				return $quantity;
			}

			return 1;
		}

		/**
		 * Validate the cart items.
		 *
		 * @return bool
		 * */
		public static function validate_cart_items() {
			$return = true;
			if ( ! is_object( WC()->cart ) ) {
				return $return;
			}

			$cart_items = WC()->cart->get_cart();
			if ( ! wal_check_is_array( $cart_items ) ) {
				return $return;
			}

			foreach ( $cart_items as $cart_item_key => $value ) {

				$result = self::validate_topup_product_cart_items( $value );
				if ( ! is_wp_error( $result ) ) {
					continue;
				}

				// Remove the product from the cart.
				WC()->cart->set_quantity( $cart_item_key, 0 );

				wal_add_wc_notice( $result->get_error_message(), 'error' );

				$return = false;
			}

			return $return;
		}

		/**
		 * Validate the top up product cart items.
		 *
		 * @return bool/error message.
		 * */
		public static function validate_topup_product_cart_items( $value ) {
			$result = true;
			if ( ! isset( $value['wal_topup']['mode'] ) || 'manual' !== $value['wal_topup']['mode'] ) {
				return $result;
			}

			// Return if the topup is not enabled.
			if ( 'yes' != get_option( 'wal_general_enable_topup' ) ) {
				return new WP_Error( 'invalid', __( 'Sorry, the top-up product was removed from the cart.', 'wallet-for-woocommerce' ) );
			}

			// Return if the wallet is in active.
			if ( 'wal_inactive' == WAL_Current_User_Wallet::get_status() ) {
				return new WP_Error( 'invalid', __( 'Sorry, the top-up product was removed from the cart.', 'wallet-for-woocommerce' ) );
			}

			// Validate the user restriction.
			if ( WAL_Topup_Handler::validate_user_restriction() ) {
				return new WP_Error( 'invalid', __( 'Sorry, the top-up product was removed from the cart.', 'wallet-for-woocommerce' ) );
			}

			// Validate the user restriction.
			$minimum_amount = floatval( get_option( 'wal_general_topup_min_amount' ) );
			if ( $minimum_amount && $minimum_amount > $value['wal_topup']['price'] ) {
				return new WP_Error( 'invalid', __( 'Sorry, the top-up product was removed from the cart.', 'wallet-for-woocommerce' ) );
			}

			// Validate the user restriction.
			$maximum_amount = floatval( get_option( 'wal_general_topup_max_amount' ) );
			if ( $maximum_amount && $maximum_amount < $value['wal_topup']['price'] ) {
				return new WP_Error( 'invalid', __( 'Sorry, the top-up product was removed from the cart.', 'wallet-for-woocommerce' ) );
			}

			// Validate the user restriction.
			$max_wallet_balance = floatval( get_option( 'wal_general_topup_max_wallet_balance' ) );
			if ( $max_wallet_balance && $max_wallet_balance < ( $value['wal_topup']['price'] + wal_convert_price( WAL_Current_User_Wallet::get_balance() ) ) ) {
				return new WP_Error( 'invalid', __( 'Sorry, the top-up product was removed from the cart.', 'wallet-for-woocommerce' ) );
			}

			// Return if the wallet topup count has reached the per day count.
			$max_count_per_day = floatval( get_option( 'wal_general_topup_count_per_day' ) );
			if ( $max_count_per_day && $max_count_per_day <= wal_customer_wallet_topup_count_per_day() ) {
				return new WP_Error( 'invalid', __( 'Sorry, the top-up product was removed from the cart.', 'wallet-for-woocommerce' ) );
			}

			// Return if the wallet topup total has reached the per day total.
			$max_total_per_day = floatval( get_option( 'wal_general_topup_max_amount_per_day' ) );
			if ( $max_total_per_day && $max_total_per_day < $value['wal_topup']['price'] + wal_customer_wallet_topup_total_per_day() ) {
				return new WP_Error( 'invalid', __( 'Sorry, the top-up product was removed from the cart.', 'wallet-for-woocommerce' ) );
			}

			return $result;
		}

		/**
		 * Remove the tax for wallet usage fees.
		 *
		 * @since 4.0.0
		 * @param array  $taxes Taxes.
		 * @param float  $fee Fee for the tax.
		 * @param object $cart Cart object.
		 * @return array
		 */
		public static function remove_tax_wallet_usage_fees( $taxes, $fee, $cart ) {
			if ( ! isset( $fee->object->id ) ) {
				return $taxes;
			}

			if ( WAL()->wallet_fee_name() != $fee->object->id ) {
				return $taxes;
			}

			return array();
		}

		/**
		 * Remove the tax for wallet usage fees.
		 *
		 * @since 4.0.0
		 * @param array  $taxes Taxes.
		 * @param float  $fee Fee for the tax.
		 * @param object $cart Cart object.
		 * @return array
		 */
		public static function recalculate_fund_after_product_removed( $cart_item_key, $cart_object ) {
			$fund = WC()->session->get( 'wal_partial_fund' );
			$cart_total = WAL_Current_User_Wallet::get_fund_to_apply( $fund );

			$wallet_balance = WAL_Current_User_Wallet::get_balance();
			
			// Return if the fund is greater than cart total.
			if (wal_get_wc_cart_total() < $wallet_balance) {
				WC()->session->set('wal_partial_fund', $cart_total);
			} else {
				WC()->session->set('wal_partial_fund', $wallet_balance);
			}
		}
	}

	WAL_Cart_Handler::init();
}
