<?php

/**
 * Handles the transaction logs exports.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

/**
 * Include dependencies.
 */
if ( ! class_exists( 'WC_CSV_Exporter', false ) ) {
	require_once WC_ABSPATH . 'includes/export/abstract-wc-csv-exporter.php' ;
}

if ( ! class_exists( 'WAL_Transaction_Logs_Export_CSV' ) ) {

	/**
	 * Class.
	 */
	class WAL_Transaction_Logs_Export_CSV extends WC_CSV_Exporter {

		/**
		 * Type of export used in filter names.
		 *
		 * @var string
		 */
		protected $export_type = 'transaction_logs' ;

		/**
		 * Filename to export to.
		 *
		 * @var string
		 */
		protected $filename = 'transaction-logs.csv' ;

		/**
		 * Wallet.
		 *
		 * @var object
		 */
		protected $wallet ;

		/**
		 * Set wallet.
		 */
		public function set_wallet( &$wallet ) {
			$this->wallet = $wallet ;
		}

		/**
		 * Get the wallet.
		 * 
		 * @return object
		 */
		public function get_wallet() {
			return $this->wallet ;
		}

		/**
		 * Generate and return a filename.
		 *
		 * @return string
		 */
		public function get_filename() {
			$file_name = $this->filename ;
			if ( isset( $this->wallet ) ) {
				$file_name = $this->wallet->get_user_name() . '-' . $this->filename ;
			}
			/**
			 * This hook is used to alter the transaction logs export file name.
			 * 
			 * @since 1.0
			 */
			return sanitize_file_name( apply_filters( "woocommerce_{$this->export_type}_export_get_filename", $file_name ) ) ;
		}

		/**
		 * Return default columns.
		 * 
		 * @return array
		 */
		public function get_default_column_names() {
			$headings = array(
				'username'     => __( 'Username', 'wallet-for-woocommerce' ),
				'user_email'   => __( 'User Email', 'wallet-for-woocommerce' ),
				'event'        => __( 'Event', 'wallet-for-woocommerce' ),
				'currency'     => __( 'Currency', 'wallet-for-woocommerce' ),
				'amount'       => __( 'Amount', 'wallet-for-woocommerce' ),
				'total'        => __( 'Total', 'wallet-for-woocommerce' ),
				'status'       => __( 'Status', 'wallet-for-woocommerce' ),
				'created_date' => __( 'Created Date', 'wallet-for-woocommerce' ),
					) ;
			/**
			 * This hook is used to alter the transaction logs export heading.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_transaction_log_export_heading', $headings ) ;
		}

		/**
		 * Prepare data that will be exported.
		 * 
		 * @return void.
		 */
		public function prepare_data_to_export() {
			// Prepare column names.
			$this->column_names = $this->get_default_column_names() ;

			$transaction_log_ids = $this->get_transaction_log_ids() ;
			foreach ( $transaction_log_ids as $transaction_log_id ) {

				$transaction_log = wal_get_transaction_log( $transaction_log_id ) ;
				if ( ! is_object( $transaction_log ) ) {
					continue ;
				}

				$this->row_data[] = self::generate_row_data( $transaction_log ) ;
			}
		}

		/**
		 * Get the transaction log data.
		 * 
		 * @return array
		 */
		protected function generate_row_data( $transaction_log ) {
			$row = array(
				'username'     => $transaction_log->get_user_name(),
				'user_email'   => $transaction_log->get_user_email(),
				'event'        => $transaction_log->get_event_message(),
				'currency'     => $transaction_log->get_currency(),
				'amount'       => $transaction_log->get_amount(),
				'total'        => $transaction_log->get_total(),
				'status'       => wal_display_post_status( $transaction_log->get_status(), false ),
				'created_date' => $transaction_log->get_formatted_created_date(),
					) ;
			/**
			 * This hook is used to alter the transaction logs export row data.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_transaction_log_export_row_data', $row ) ;
		}

		/**
		 * Get the transaction log IDs.
		 * 
		 * @return array
		 */
		protected function get_transaction_log_ids() {
			$args = array(
				'post_type'   => WAL_Register_Post_Types::TRANSACTION_LOG_POSTTYPE,
				'post_status' => wal_get_transaction_log_statuses(),
				'fields'      => 'ids',
				'numberposts' => '-1',
					) ;

			if ( isset( $this->wallet ) ) {
				$args[ 'meta_key' ]   = 'wal_user_id' ;
				$args[ 'meta_value' ] = $this->wallet->get_user_id() ;
			}

			return get_posts( $args ) ;
		}
	}

}
