<?php

/**
 * Handles the gift vouchers exports.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

/**
 * Include dependencies.
 */
if ( ! class_exists( 'WC_CSV_Exporter', false ) ) {
	require_once WC_ABSPATH . 'includes/export/abstract-wc-csv-exporter.php' ;
}

if ( ! class_exists( 'WAL_Gift_Vouchers_Export_CSV' ) ) {

	/**
	 * Class.
	 */
	class WAL_Gift_Vouchers_Export_CSV extends WC_CSV_Exporter {

		/**
		 * Type of export used in filter names.
		 *
		 * @var string
		 */
		protected $export_type = 'gift_vouchers' ;

		/**
		 * Filename to export to.
		 *
		 * @var string
		 */
		protected $filename = 'gift-vouchers.csv' ;

		/**
		 * Generate and return a filename.
		 *
		 * @return string
		 */
		public function get_filename() {
			/**
			 * This hook is used to alter the gift vouchers export file name.
			 * 
			 * @since 1.0
			 */
			return sanitize_file_name( apply_filters( "woocommerce_{$this->export_type}_export_get_filename", $this->filename ) ) ;
		}

		/**
		 * Return default columns.
		 * 
		 * @return array
		 */
		public function get_default_column_names() {
			$headings = array(
				'code'          => __( 'Code', 'wallet-for-woocommerce' ),
				'currency'      => __( 'Currency', 'wallet-for-woocommerce' ),
				'balance'       => __( 'Wallet Balance', 'wallet-for-woocommerce' ),
				'created_date'  => __( 'Created On', 'wallet-for-woocommerce' ),
				'expired_date'  => __( 'Expiry Date', 'wallet-for-woocommerce' ),
				'status'        => __( 'Status', 'wallet-for-woocommerce' ),
				'username'      => __( 'Redeemed user Name', 'wallet-for-woocommerce' ),
				'user_email'    => __( 'Redeemed user Email', 'wallet-for-woocommerce' ),
				'redeemed_date' => __( 'Redeemed Date', 'wallet-for-woocommerce' ),
					) ;
			/**
			 * This hook is used to alter the gift vouchers export heading.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_gift_voucher_export_heading', $headings ) ;
		}

		/**
		 * Prepare data that will be exported.
		 * 
		 * @return void.
		 */
		public function prepare_data_to_export() {
			// Prepare column names.
			$this->column_names = $this->get_default_column_names() ;

			$gift_voucher_ids = $this->get_gift_voucher_ids() ;
			foreach ( $gift_voucher_ids as $gift_voucher_id ) {

				$gift_voucher = wal_get_gift_voucher( $gift_voucher_id ) ;
				if ( ! is_object( $gift_voucher ) ) {
					continue ;
				}

				$this->row_data[] = self::generate_row_data( $gift_voucher ) ;
			}
		}

		/**
		 * Get the transaction log data.
		 * 
		 * @return array
		 */
		protected function generate_row_data( $gift_voucher ) {
			$row = array(
				'code'          => $gift_voucher->get_code(),
				'currency'      => $gift_voucher->get_currency(),
				'balance'       => $gift_voucher->get_amount(),
				'created_date'  => $gift_voucher->get_formatted_created_date(),
				'expired_date'  => $gift_voucher->get_formatted_expiry_date(),
				'status'        => wal_display_post_status( $gift_voucher->get_status(), false ),
				'username'      => $gift_voucher->get_user_name(),
				'user_email'    => $gift_voucher->get_user_email(),
				'redeemed_date' => $gift_voucher->get_formatted_redeemed_date(),
					) ;
			/**
			 * This hook is used to alter the gift vouchers export row data.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_gift_voucher_export_row_data', $row ) ;
		}

		/**
		 * Get the gift voucher IDs.
		 * 
		 * @return array
		 */
		protected function get_gift_voucher_ids() {
			$args = array(
				'post_type'   => WAL_Register_Post_Types::GIFT_VOUCHER_POSTTYPE,
				'post_status' => wal_get_gift_voucher_statuses(),
				'fields'      => 'ids',
				'numberposts' => '-1',
					) ;

			return get_posts( $args ) ;
		}
	}

}
