<?php

/**
 * Wallet Withdrawal.
 * 
 * @since 1.8
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Wallet_Withdrawal' ) ) {

	/**
	 * Class.
	 */
	class WAL_Wallet_Withdrawal extends WAL_Post {

		/**
		 * Post Type.
		 * 
		 * @var string
		 */
		protected $post_type = WAL_Register_Post_Types::WALLET_WITHDRAWAL_POSTTYPE ;

		/**
		 * Post Status.
		 * 
		 * @var string
		 */
		protected $post_status = 'wal_pending' ;

		/**
		 * Created Date.
		 * 
		 * @var string
		 */
		protected $created_date ;

		/**
		 * Wallet.
		 * 
		 * @var object
		 */
		protected $wallet ;
				
		/**
		 * Wallet ID.
		 * 
		 * @since 3.3.0
		 * @var int
		 */
		protected $wallet_id ;

		/**
		 * Meta data keys.
		 */
		protected $meta_data_keys = array(
			'wal_user_id'                    => '',
			'wal_amount'                     => '',
			'wal_fee'                        => '',
			'wal_currency'                   => '',
			'wal_payment_method'             => '',
			'wal_payment_method_credentials' => array(),
			'wal_withdrawal_cancelled_reason' => '',
		);

		/**
		 * Prepare extra post data.
		 */
		protected function load_extra_postdata() {
			$this->wallet_id    = $this->post->post_parent ;
			$this->created_date = $this->post->post_date_gmt ;
		}

		/**
		 * Get the wallet.
		 * 
		 * @return object
		 */
		public function get_wallet() {
			if ( isset( $this->wallet ) ) {
				return $this->wallet ;
			}

			$this->wallet = wal_get_wallet( $this->get_wallet_id() ) ;

			return $this->wallet ;
		}

		/**
		 * Get the formatted created date time.
		 * 
		 * @return string
		 */
		public function get_formatted_created_date() {
			return WAL_Date_Time::get_wp_format_datetime_from_gmt( $this->get_created_date() ) ;
		}

		/**
		 * Get the user name.
		 * 
		 * @return string
		 */
		public function get_user_name() {
			if ( ! is_object( $this->get_wallet() ) ) {
				return '' ;
			}

			return $this->get_wallet()->get_user_name() ;
		}

		/**
		 * Get the user email.
		 * 
		 * @return string
		 */
		public function get_user_email() {
			if ( ! is_object( $this->get_wallet() ) ) {
				return '' ;
			}

			return $this->get_wallet()->get_user_email() ;
		}

		/**
		 * Get the payment gateway.
		 * 
		 * @return object / bool
		 */
		public function get_payment_gateway() {
			return WAL_Module_Instances::get_module_by_id( 'wallet_withdrawal' )->get_payment_gateway_by_id( $this->get_payment_method() ) ;
		}

		/**
		 * Format the wallet withdrawal gateway credentials.
		 * 
		 * @return array
		 */
		public function get_formatted_payment_gateway_credentials() {
			$formatted_credentials = array() ;
			if ( ! $this->get_payment_gateway() ) {
				return $formatted_credentials ;
			}

			return wal_format_payment_gateway_credentials( $this->get_payment_method_credentials(), $this->get_payment_gateway()->get_credential_labels() ) ;
		}

		/**
		 * Get the payment gateway title.
		 * 
		 * @return string
		 */
		public function get_payment_gateway_title() {
			if ( ! $this->get_payment_gateway() ) {
				return '' ;
			}

			return $this->get_payment_gateway()->get_title() ;
		}

		/**
		 * Get the total amount.
		 * 
		 * @return float
		 */
		public function get_total() {
			return floatval( $this->get_amount() ) + floatval( $this->get_fee() ) ;
		}

		/**
		 * Setters and Getters.
		 * */

		/**
		 * Set wallet ID
		 */
		public function set_wallet_id( $value ) {
			$this->wallet_id = $value ;
		}

		/**
		 * Set created date.
		 */
		public function set_created_date( $value ) {
			$this->created_date = $value ;
		}

		/**
		 * Set user ID.
		 * */
		public function set_user_id( $value ) {
			$this->set_prop( 'wal_user_id', $value ) ;
		}

		/**
		 * Set amount.
		 * */
		public function set_amount( $value ) {
			$this->set_prop( 'wal_amount', $value ) ;
		}

		/**
		 * Set fee.
		 */
		public function set_fee( $value ) {
			$this->set_prop( 'wal_fee', $value ) ;
		}

		/**
		 * Set currency.
		 */
		public function set_currency( $value ) {
			$this->set_prop( 'wal_currency', $value ) ;
		}

		/**
		 * Set payment method.
		 */
		public function set_payment_method( $value ) {
			$this->set_prop( 'wal_payment_method', $value ) ;
		}

		/**
		 * Set payment method credentials.
		 */
		public function set_payment_method_credentials( $value ) {
			$this->set_prop('wal_payment_method_credentials', $value);
		}

		/**
		 * Set cancelled reason.
		 *
		 * @since 2.8.0
		 */
		public function set_cancelled_reason( $value ) {
			$this->set_prop('wal_withdrawal_cancelled_reason', $value);
		}

		/**
		 * Get wallet ID.
		 */
		public function get_wallet_id() {
			return $this->wallet_id ;
		}

		/**
		 * Get created date.
		 */
		public function get_created_date() {
			return $this->created_date ;
		}

		/**
		 * Get user ID.
		 */
		public function get_user_id() {
			return $this->get_prop( 'wal_user_id' ) ;
		}

		/**
		 * Get amount.
		 */
		public function get_amount() {
			return $this->get_prop( 'wal_amount' ) ;
		}

		/**
		 * Get fee.
		 */
		public function get_fee() {
			return $this->get_prop( 'wal_fee' ) ;
		}

		/**
		 * Get currency.
		 */
		public function get_currency() {
			return $this->get_prop( 'wal_currency' ) ;
		}

		/**
		 * Get payment method.
		 */
		public function get_payment_method() {
			return $this->get_prop( 'wal_payment_method' ) ;
		}

		/**
		 * Get payment method credentials.
		 */
		public function get_payment_method_credentials() {
			return $this->get_prop('wal_payment_method_credentials');
		}

		/**
		 * Get cancelled reason.
		 *
		 * @since 2.8.0
		 */
		public function get_cancelled_reason() {
			return $this->get_prop('wal_withdrawal_cancelled_reason');
		}
	}

}
