<?php

/**
 * Transaction Log.
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WAL_Auto_Topup')) {

	/**
	 * WAL_Auto_Topup Class.
	 */
	class WAL_Auto_Topup extends WAL_Post {

		/**
		 * Post Type.
		 * 
		 * @var string
		 */
		protected $post_type = WAL_Register_Post_Types::AUTO_TOPUP_USERS_POSTTYPE;

		/**
		 * Post Status.
		 * 
		 * @var string
		 */
		protected $post_status = 'wal_active';

		/**
		 * Wallet ID.
		 * 
		 * @var int
		 */
		protected $wallet_id;

		/**
		 * Created Date.
		 * 
		 * @var string
		 */
		protected $created_date;

		/**
		 * Wallet.
		 * 
		 * @var object
		 */
		protected $wallet;

		/**
		 * Order.
		 * 
		 * @since 2.8.0
		 * @var object
		 */
		protected $order;

		/**
		 * Last renewal order ID.
		 * 
		 * @since 2.8.0
		 * @var object
		 */
		protected $last_renewal_order_id;

		/**
		 * User.
		 * 
		 * @var object
		 */
		protected $user;

		/**
		 * Meta data keys.
		 */
		protected $meta_data_keys = array(
			'wal_user_id' => '',
			'wal_topup_amount' => '',
			'wal_threshold_amount' => '',
			'wal_payment_method' => '',
			'wal_authorized_order' => '',
			'wal_currency' => '',
			'wal_last_charge_date' => '',
			'wal_last_activity' => '',
			'wal_last_transaction_id' => '',
			'wal_last_renewal_order_id' => '',
			'wal_renewal_order_ids' => array(),
		);

		/**
		 * Prepare extra post data.
		 */
		protected function load_extra_postdata() {
			$this->wallet_id = $this->post->post_parent;
			$this->created_date = $this->post->post_date_gmt;
		}

		/**
		 * Get the wallet.
		 * 
		 * @return object
		 */
		public function get_wallet() {
			if (isset($this->wallet)) {
				return $this->wallet;
			}

			$this->wallet = wal_get_wallet($this->get_wallet_id());

			return $this->wallet;
		}

		/**
		 * Get the user.
		 * 
		 * @return object
		 */
		public function get_user() {

			if (isset($this->user)) {
				return $this->user;
			}

			$this->user = get_userdata($this->get_user_id());

			return $this->user;
		}

		/**
		 * Get the order.
		 * 
		 * @since 2.8.0
		 * @return object
		 */
		public function get_order() {
			if (isset($this->order)) {
				return $this->order;
			}

			$this->order = wc_get_order($this->get_authorized_order());

			return $this->order;
		}

		/**
		 * Get the last renewal order.
		 * 
		 * @since 2.8.0
		 * @return object
		 */
		public function get_last_renewal_order() {
			if (isset($this->last_renewal_order)) {
				return $this->last_renewal_order;
			}

			$this->last_renewal_order = wc_get_order($this->get_last_renewal_order_id());

			return $this->last_renewal_order;
		}

		/**
		 * Get the user name.
		 * 
		 * @return string
		 */
		public function get_user_name() {
			if (!is_object($this->get_user())) {
				return '';
			}

			return $this->get_user()->display_name;
		}

		/**
		 * Get the user email.
		 * 
		 * @return string
		 */
		public function get_user_email() {
			if (!is_object($this->get_user())) {
				return '';
			}

			return $this->get_user()->user_email;
		}

		/**
		 * Update the renewal ID.
		 * 
		 * @since 2.8.0
		 * @param int $renewal_order_id Renewal Order ID
		 */
		public function update_renewal_details( $renewal_order_id ) {    
			$renewal_ids = (array) $this->get_renewal_order_ids(); 
			$renewal_ids[] = $renewal_order_id;
			$this->update_meta('wal_renewal_order_ids', array_filter(array_unique($renewal_ids), 'wal_array_filter'));

			$authorized_order_renewal_ids = (array) $this->get_order()->get_meta('wal_auto_topup_renewal_order_ids');                          
			$authorized_order_renewal_ids[] = $renewal_order_id;
											
			$this->get_order()->update_meta_data('wal_auto_topup_renewal_order_ids', array_filter(array_unique($authorized_order_renewal_ids), 'wal_array_filter'));                        
			$this->get_order()->save();
		}

		/**
		 * Get the formatted created datetime.
		 * 
		 * @return string
		 */
		public function get_formatted_created_date() {
			return WAL_Date_Time::get_wp_format_datetime_from_gmt($this->get_created_date());
		}

		/**
		 * Get the formatted last charge datetime.
		 * 
		 * @return string
		 */
		public function get_formatted_last_charge_date() {
			return WAL_Date_Time::get_wp_format_datetime_from_gmt($this->get_last_charge_date());
		}

		/**
		 * Setters and Getters.
		 * */

		/**
		 * Set wallet ID
		 */
		public function set_wallet_id( $value ) {
			$this->wallet_id = $value;
		}

		/**
		 * Set created date.
		 */
		public function set_created_date( $value ) {
			$this->created_date = $value;
		}

		/**
		 * Set user ID.
		 * */
		public function set_user_id( $value ) {
			$this->set_prop('wal_user_id', $value);
		}

		/**
		 * Set topup amount.
		 * */
		public function set_topup_amount( $value ) {
			$this->set_prop('wal_topup_amount', $value);
		}

		/**
		 * Set threshold amount.
		 * */
		public function set_threshold_amount( $value ) {
			$this->set_prop('wal_threshold_amount', $value);
		}

		/**
		 * Set payment method.
		 * */
		public function set_payment_method( $value ) {
			$this->set_prop('wal_payment_method', $value);
		}

		/**
		 * Set authorized order.
		 * */
		public function set_authorized_order( $value ) {
			$this->set_prop('wal_authorized_order', $value);
		}

		/**
		 * Set last charge date.
		 * */
		public function set_last_charge_date( $value ) {
			$this->set_prop('wal_last_charge_date', $value);
		}

		/**
		 * Set currency.
		 */
		public function set_currency( $value ) {
			$this->set_prop('wal_currency', $value);
		}

		/**
		 * Set last activity.
		 */
		public function set_last_activity( $value ) {
			$this->set_prop('wal_last_activity', $value);
		}

		/**
		 * Set last transaction id.
		 */
		public function set_last_transaction_id( $value ) {
			$this->set_prop('wal_last_transaction_id', $value);
		}

		/**
		 * Set last renewal order ID.
		 * 
		 * @since 2.8.0
		 * @param int $value Renewal Order ID
		 */
		public function set_last_renewal_order_id( $value ) {
			$this->set_prop('wal_last_renewal_order_id', $value);
		}

		/**
		 * Set renewal order IDs.
		 * 
		 * @since 2.8.0
		 * @param array $value Renewal Order IDs
		 */
		public function set_renewal_order_ids( $value ) {
			$this->set_prop('wal_renewal_order_ids', $value);
		}

		/**
		 * Get wallet ID.
		 */
		public function get_wallet_id() {
			return $this->wallet_id;
		}

		/**
		 * Get created date.
		 */
		public function get_created_date() {
			return $this->created_date;
		}

		/**
		 * Get user ID.
		 */
		public function get_user_id() {
			return $this->get_prop('wal_user_id');
		}

		/**
		 * Get topup amount.
		 * */
		public function get_topup_amount() {
			return $this->get_prop('wal_topup_amount');
		}

		/**
		 * Get threshold amount.
		 * */
		public function get_threshold_amount() {
			return $this->get_prop('wal_threshold_amount');
		}

		/**
		 * Get payment method.
		 * */
		public function get_payment_method() {
			return $this->get_prop('wal_payment_method');
		}

		/**
		 * Get authorized order.
		 * */
		public function get_authorized_order() {
			return $this->get_prop('wal_authorized_order');
		}

		/**
		 * Get last charge date.
		 * */
		public function get_last_charge_date() {
			return $this->get_prop('wal_last_charge_date');
		}

		/**
		 * Get currency.
		 */
		public function get_currency() {
			return $this->get_prop('wal_currency');
		}

		/**
		 * Get last activity.
		 */
		public function get_last_activity() {
			return $this->get_prop('wal_last_activity');
		}

		/**
		 * Get last transaction id.
		 */
		public function get_last_transaction_id() {
			return $this->get_prop('wal_last_transaction_id');
		}

		/**
		 * Get the last renewal order id.
		 * 
		 * @since 2.8.0
		 * @return int
		 */
		public function get_last_renewal_order_id() {
			return $this->get_prop('wal_last_renewal_order_id');
		}

		/**
		 * Get the renewal order IDs.
		 * 
		 * @since 2.8.0
		 * @return array
		 */
		public function get_renewal_order_ids() {
			return $this->get_prop('wal_renewal_order_ids');
		}
	}

}
