<?php

/**
 * Compatibility - YaYCommerce Currency Plugin.
 * 
 * @since 3.0.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

use Yay_Currency\Helpers\Helper;
use Yay_Currency\Helpers\YayCurrencyHelper;

if (!class_exists('WAL_YaYCommerce_Currency_Compatibility')) {

	/**
	 * Class.
	 *
	 * @since 3.0.0
	 */
	class WAL_YaYCommerce_Currency_Compatibility extends WAL_Compatibility {

		/**
		 * Class Constructor.
		 *
		 * @since 3.0.0
		 */
		public function __construct() {
			$this->id = 'yaycommerce-currency';

			parent::__construct();
		}

		/**
		 * Is plugin enabled?.
		 *
		 * @return bool
		 * */
		public function is_plugin_enabled() {
			return class_exists('Yay_Currency\\Initialize');
		}

		/**
		 * Front end action.
		 *
		 * @since 3.0.0
		 */
		public function actions() {
			// Convert the price based on default currency.
			add_filter('wal_convert_price', array( __CLASS__, 'convert_price' ), 10, 2);
			// Convert the price by currency based on default currency.
			add_filter('wal_convert_price_by_currency', array( __CLASS__, 'convert_price_by_currency' ), 10, 1);
			// Convert Price based on currency.
			add_filter('wal_payment_gateway_order_total', array( __CLASS__, 'convert_payment_gateway_currency_amount' ), 10, 1);
			// Convert Price based on currency.
			add_filter('wal_payment_gateway_user_balance', array( __CLASS__, 'convert_payment_gateway_currency_amount' ), 10, 1);
			// Convert Wallet Gateway amount display
			add_filter('wal_payment_gateway_available_user_funds', array( __CLASS__, 'convert_gateway_available_user_balance' ), 10, 2);
			// Convert Wallet partial debit amount
			add_filter('wal_partial_debit_amount', array( __CLASS__, 'convert_partial_fund_debit' ), 10, 1);
		}

		/**
		 * Convert the price based on current currency in the site.
		 * 
		 * @since 3.0.0
		 * @param float/int $price
		 * @param bool $convert_to_default_currency
		 * @return float
		 */
		public static function convert_price( $price, $convert_to_default_currency = false ) {
			if (!$price) {
				return $price;
			}

			$apply_currency = YayCurrencyHelper::detect_current_currency();
			if (!wal_check_is_array($apply_currency)) {
				return $price;
			}

			if ($convert_to_default_currency) {
				return floatval($price) / floatval($apply_currency['rate']);
			}

			$converted_discount = YayCurrencyHelper::calculate_price_by_currency($price, false, $apply_currency);

			return floatval($converted_discount);
		}

		/**
		 * Convert the price by currency based on current currency in the site.
		 * 
		 * @since 3.0.0
		 * @param float/int $price        
		 * @return float
		 */
		public static function convert_price_by_currency( $price ) {
			if (!$price) {
				return $price;
			}

			$apply_currency = YayCurrencyHelper::detect_current_currency();
			if (!wal_check_is_array($apply_currency)) {
				return $price;
			}

			$price = YayCurrencyHelper::calculate_price_by_currency($price, false, $apply_currency);

			return $price;
		}

		/**
		 * Convert the payment gateway amount on currency in the site.
		 * 
		 * @since 3.0.0
		 * @param float/int $total
		 * @return float
		 */
		public static function convert_payment_gateway_currency_amount( $total ) {
			if (empty(get_option('yay_currency_checkout_different_currency'))) {
				return floatval($total);
			}

			return self::convert_price($total);
		}

		/**
		 * Convert the price by convert gateway available user balance.
		 * 
		 * @since 3.0.0
		 * @param string $price
		 * @param float/int $amount   
		 * @return string
		 */
		public static function convert_gateway_available_user_balance( $price, $amount ) {
			if (!empty(get_option('yay_currency_checkout_different_currency'))) {
				return $price;
			}

			$apply_currency = YayCurrencyHelper::detect_current_currency();
			$currency_symbol = isset($apply_currency['symbol']) ? $apply_currency['symbol'] : '';
			if (get_woocommerce_currency_symbol() === $currency_symbol) {
				return $price;
			}

			$balance = wal_price(wal_convert_price($amount), array( 'currency' => $apply_currency['currency'] ));
			return $price . YayCurrencyHelper::converted_approximately_html($balance);
		}

		/**
		 * Convert the partial debit amount.
		 * 
		 * @since 3.0.0
		 * @param float/int $amount
		 * @return float
		 */
		public static function convert_partial_fund_debit( $amount ) {
			if (empty(get_option('yay_currency_checkout_different_currency'))) {
				return floatval($amount);
			}

			return self::convert_price($amount, true);
		}
	}

}
