<?php

/**
 * WooCommerce Subscriptions Compatibility.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Woocommerce_Subscriptions_Compatibility' ) ) {

	/**
	 * Class.
	 */
	class WAL_Woocommerce_Subscriptions_Compatibility extends WAL_Compatibility {

		/**
		 * Class Constructor.
		 */
		public function __construct() {
			$this->id = 'woocommerce_subscriptions' ;

			parent::__construct() ;
		}

		/**
		 * Is plugin enabled?.
		 * 
		 *  @return bool
		 * */
		public function is_plugin_enabled() {
			return class_exists( 'WC_Subscriptions' ) ;
		}

		/**
		 * Action.
		 */
		public function actions() {
			add_filter( 'wal_wallet_gateway_supports', array( $this, 'supports' ) ) ;
			add_action( 'woocommerce_scheduled_subscription_payment_wal_wallet', array( $this, 'scheduled_subscription_payment' ), 10, 2 ) ;
		}

		/**
		 * Add gateway supports subscriptions.
		 * 
		 * @param bool $bool
		 * @param string $feature
		 * @param WC_Payment_Gateway $gateway
		 * @return bool
		 */
		public function supports( $supports ) {
			$supports = array_merge( $supports, array(
				'products',
				'subscriptions',
				'subscription_cancellation',
				'subscription_suspension',
				'subscription_reactivation',
				'subscription_amount_changes',
				'subscription_date_changes',
				'subscription_payment_method_change',
				'subscription_payment_method_change_customer',
				'subscription_payment_method_change_admin',
				'multiple_subscriptions',
					) ) ;

			return $supports ;
		}

		/**
		 * Process subscription payment on scheduled.
		 *
		 * @param $amount_to_charge float The amount to charge.
		 * @param $renewal_order WC_Order A WC_Order object created to record the renewal payment.
		 */
		public function scheduled_subscription_payment( $amount_to_charge, $renewal_order ) {
			try {
				$wallet_id = wal_get_wallet_id_by_user_id( $renewal_order->get_customer_id() ) ;

				if ( empty( $wallet_id ) ) {
					/* translators: %s - Renewal order ID */
					throw new Exception( sprintf( __( 'Failed to process renewal for order #%s. Wallet for the user is missing', 'wallet-for-woocommerce' ), $renewal_order->get_id() ) ) ;
				}

				$wallet = wal_get_wallet( $wallet_id ) ;
				if ( ! $wallet->has_status( 'wal_active' ) ) {
					/* translators: %s - Renewal order ID */
					throw new Exception( sprintf( __( 'Failed to process renewal for order #%s. Wallet for the user is not active', 'wallet-for-woocommerce' ) ), $renewal_order->get_id() ) ;
				}

				if ( $amount_to_charge > floatval( $wallet->get_balance() ) ) {
					/* translators: %s - Renewal order ID */
					throw new Exception( sprintf( __( 'Insufficient balance to process renewal for order #%s' ), $renewal_order->get_id() ) ) ;
				}

				$transaction_log_id = wal_debit_wallet_fund( array(
					'user_id'            => $renewal_order->get_customer_id(),
					'order_id'           => $renewal_order->get_id(),
					'amount'             => $amount_to_charge,
					'event_id'           => 8,
					/* translators: %s - Renewal order ID */
					'event_message'      => sprintf( __( 'Wallet Gateway used on Renewal Order #%s.', 'wallet-for-woocommerce' ), $renewal_order->get_id() ),
					'currency'           => $renewal_order->get_currency(),
					'update_usage_total' => true,
						) ) ;

				if ( ! $transaction_log_id ) {
					/* translators: %s - Renewal order ID */
					throw new exception( sprintf( __( 'Sorry, we are unable to process renewal for order #%s at this time. Please retry later.', 'wallet-for-woocommerce' ), $renewal_order->get_id() ) ) ;
				}

				$renewal_order->payment_complete() ;
								
								// Set the gateway order status.
				$renewal_order->update_status( get_option( 'wal_general_default_gateway_order_status', 'processing' ), __( 'Awaiting for Admin Confirmation', 'wallet-for-woocommerce' ) ) ;
								
				/* translators: %1s - Amount , %2s - User name */
				$renewal_order->add_order_note( sprintf( __( 'The amount of %1$s debited from user %2$s', 'wallet-for-woocommerce' ), wal_price( $amount_to_charge ), $renewal_order->get_user()->display_name ) ) ;

				// Update the fund details.
				$renewal_order->update_meta_data('wal_gateway_total_fund', $amount_to_charge);
				$renewal_order->update_meta_data('wal_gateway_remaining_fund', $amount_to_charge);
				$renewal_order->update_meta_data('wal_wallet_fund', $amount_to_charge);

				// Update the meta.
				$renewal_order->update_meta_data('wal_wallet_fund_debited', 'yes');
				$renewal_order->update_meta_data('wal_gateway_fund_debited', 'yes');
				$renewal_order->save();

				/**
				 * This hook is used to do extra action after wallet payment gateway processed.
				 * 
				 * @since 1.0
				 */
				do_action( 'wal_wallet_gateway_payment_processed', $transaction_log_id, $renewal_order ) ;
			} catch ( Exception $e ) {
				/**
				 * This hook is used to do extra action after wallet payment gateway process error.
				 * 
				 * @since 1.0
				 */
				do_action( 'wal_wallet_gateway_process_payment_error', $e, $renewal_order ) ;

				/* translators: error message */
				$renewal_order->update_status( 'failed' ) ;
			}
		}
	}

}
