<?php

/**
 * Wallet for WooCommerce Main Class.
 * 
 * @since 1.0.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('FP_Wallet')) {

	/**
	 * Class.
	 * 
	 * @since 1.0.0
	 * */
	final class FP_Wallet {

		/**
		 * Version.
		 * 
		 * @var string
		 * */
		private $version = '5.1.0';

		/**
		 * Locale.
		 * 
		 * @var string
		 * */
		private $locale = 'wallet-for-woocommerce';

		/**
		 * Folder Name.
		 * 
		 * @var string
		 * */
		private $folder_name = 'wallet-for-woocommerce';

		/**
		 * WC minimum version.
		 *
		 * @var string
		 */
		public static $wc_minimum_version = '3.5.0';

		/**
		 * WP minimum version.
		 *
		 * @var string
		 */
		public static $wp_minimum_version = '4.6.0';

		/**
		 * Wallet Fee name.
		 *
		 * @since 3.7.0
		 * @var string
		 */
		private $wallet_fee_name = 'wallet-credits';

		/**
		 * The single instance of the class.
		 * 
		 * @var object
		 * */
		protected static $_instance = null;

		/**
		 * Load FP_Wallet Class in Single Instance.
		 * 
		 * @since 1.0.0
		 * @return object
		 */
		public static function instance() {
			if (is_null(self::$_instance)) {
				self::$_instance = new self();
			}

			return self::$_instance;
		}

		/* Cloning has been forbidden */

		public function __clone() {
			_doing_it_wrong(__FUNCTION__, 'You are not allowed to perform this action!!!', esc_html($this->version));
		}

		/**
		 * Unserialize the class data has been forbidden.
		 * */
		public function __wakeup() {
			_doing_it_wrong(__FUNCTION__, 'You are not allowed to perform this action!!!', esc_html($this->version));
		}

		/**
		 * Constructor.
		 * 
		 * @since 1.0.0
		 * */
		public function __construct() {
			$this->define_constants();
			$this->include_files();
			$this->init_hooks();
		}

		/**
		 * Load the plugin translate files.
		 * 
		 * @since 1.0.0
		 * */
		private function load_plugin_textdomain() {
			$locale = determine_locale();
			/**
			 * This hook is used to alter the plugin locale.
			 * 
			 * @since 1.0.0
			 */
			$locale = apply_filters('plugin_locale', $locale, WAL_LOCALE);

			// Unload the text domain if other plugins/themes loaded the same text domain by mistake.
			unload_textdomain(WAL_LOCALE);

			// Load the text domain from the "wp-content" languages folder. we have handles the plugin folder in languages folder for easily handle it.
			load_textdomain(WAL_LOCALE, WP_LANG_DIR . '/' . WAL_FOLDER_NAME . '/' . WAL_LOCALE . '-' . $locale . '.mo');

			// Load the text domain from the "wp-content" languages folder. we have handles the plugin folder in languages folder for easily handle it.
			load_textdomain(WAL_LOCALE, WP_LANG_DIR . '/plugins/' . WAL_LOCALE . '-' . $locale . '.mo');

			// Load the text domain from the current plugin languages folder.
			load_plugin_textdomain(WAL_LOCALE, false, dirname(plugin_basename(WAL_PLUGIN_FILE)) . '/languages');
		}

		/**
		 * Define the constant values.
		 * 
		 * @since 1.0.0
		 * */
		private function define_constants() {

			$constant_array = array(
				'WAL_VERSION' => $this->version,
				'WAL_LOCALE' => $this->locale,
				'WAL_FOLDER_NAME' => $this->folder_name,
				'WAL_ABSPATH' => dirname(WAL_PLUGIN_FILE) . '/',
				'WAL_ADMIN_URL' => admin_url('admin.php'),
				'WAL_ADMIN_AJAX_URL' => admin_url('admin-ajax.php'),
				'WAL_PLUGIN_SLUG' => plugin_basename(WAL_PLUGIN_FILE),
				'WAL_PLUGIN_PATH' => untrailingslashit(plugin_dir_path(WAL_PLUGIN_FILE)),
				'WAL_PLUGIN_URL' => untrailingslashit(plugins_url('/', WAL_PLUGIN_FILE)),
			);
			/**
			 * This hook is used to alter the plugin constants.
			 * 
			 * @since 1.0.0
			 */
			$constant_array = apply_filters('wal_define_constants', $constant_array);

			if (is_array($constant_array) && !empty($constant_array)) {
				foreach ($constant_array as $name => $value) {
					if (defined($name)) {
						continue;
					}

					define($name, $value);
				}
			}
		}

		/**
		 * Include required files.
		 * 
		 * @since 1.0.0
		 * */
		private function include_files() {
			// Function.
			include_once WAL_ABSPATH . 'inc/wal-common-functions.php';
			include_once WAL_ABSPATH . 'inc/wal-frontend-functions.php';
			include_once WAL_ABSPATH . 'inc/class-wal-topup-bonus-handler.php';
			include_once WAL_ABSPATH . 'inc/admin/wal-admin-functions.php';

			// Abstract classes.
			include_once WAL_ABSPATH . 'inc/abstracts/abstract-wal-post.php';

			include_once WAL_ABSPATH . 'inc/class-wal-register-post-types.php';
			include_once WAL_ABSPATH . 'inc/class-wal-register-post-status.php';

			include_once WAL_ABSPATH . 'inc/class-wal-install.php';
			include_once WAL_ABSPATH . 'inc/class-wal-pages.php';
			include_once WAL_ABSPATH . 'inc/class-wal-date-time.php';
			include_once WAL_ABSPATH . 'inc/class-wal-query.php';
			include_once WAL_ABSPATH . 'inc/privacy/class-wal-privacy.php';

			// Instances.
			include_once WAL_ABSPATH . 'inc/modules/class-wal-module-instances.php';
			include_once WAL_ABSPATH . 'inc/widgets/class-wal-widget-instances.php';
			include_once WAL_ABSPATH . 'inc/notifications/class-wal-notification-instances.php';
			include_once WAL_ABSPATH . 'inc/compatibility/class-wal-compatibility-instances.php';

			// Entity.
			include_once WAL_ABSPATH . 'inc/entity/class-wal-wallet.php';
			include_once WAL_ABSPATH . 'inc/entity/class-wal-transaction-log.php';

			include_once WAL_ABSPATH . 'inc/class-wal-order-handler.php';
			include_once WAL_ABSPATH . 'inc/class-wal-cron-handler.php';
			include_once WAL_ABSPATH . 'inc/class-wal-action-scheduler-handler.php';

			// Current user wallet.
			include_once WAL_ABSPATH . 'inc/class-wal-current-user-wallet.php';

			// Block compatibility.
			include_once WAL_ABSPATH . 'inc/wc-blocks/class-wal-wc-blocks-compatibility.php';

			if (is_admin()) {
				$this->include_admin_files();
			}

			if (!is_admin() || defined('DOING_AJAX')) {
				$this->include_frontend_files();
			}
		}

		/**
		 * Include admin files.
		 * 
		 * @since 1.0.0
		 * */
		private function include_admin_files() {
			include_once WAL_ABSPATH . 'inc/admin/class-wal-admin-assets.php';
			include_once WAL_ABSPATH . 'inc/admin/class-wal-admin-ajax.php';
			include_once WAL_ABSPATH . 'inc/admin/menu/class-wal-menu-management.php';
			include_once WAL_ABSPATH . 'inc/admin/class-wal-admin-user-handler.php';
			include_once WAL_ABSPATH . 'inc/admin/class-wal-admin-post-type-handler.php';
		}

		/**
		 * Include frontend files.
		 * 
		 * @since 1.0.0
		 * */
		private function include_frontend_files() {
			include_once WAL_ABSPATH . 'inc/frontend/class-wal-frontend-assets.php';
			include_once WAL_ABSPATH . 'inc/frontend/class-wal-cart-handler.php';
			include_once WAL_ABSPATH . 'inc/frontend/class-wal-fund-usage-validation.php';
			include_once WAL_ABSPATH . 'inc/frontend/class-wal-checkout-handler.php';
			include_once WAL_ABSPATH . 'inc/frontend/class-wal-myaccount-handler.php';
			include_once WAL_ABSPATH . 'inc/frontend/class-wal-dashboard-handler.php';
			include_once WAL_ABSPATH . 'inc/frontend/class-wal-topup-handler.php';
			include_once WAL_ABSPATH . 'inc/frontend/class-wal-shortcodes.php';
		}

		/**
		 * Define the hooks.
		 * 
		 * @since 1.0.0
		 * */
		private function init_hooks() {
			// WC compatibility to the plugin.
			add_action('before_woocommerce_init', array( $this, 'declare_WC_compatibility' ));

			// Init the plugin.
			add_action('init', array( $this, 'init' ), 5);
			// Plugins loaded.
			add_action('plugins_loaded', array( $this, 'plugins_loaded' ), 11);
			// Register the plugin.
			register_activation_hook(WAL_PLUGIN_FILE, array( 'WAL_Install', 'install' ));
		}

		/**
		 * Declare the plugin is compatibility with WC features.
		 * 
		 * @since 3.7.0
		 * @return void
		 */
		public function declare_WC_compatibility() {
			// HPOS compatibility.
			$this->declare_WC_HPOS_compatibility();

			// Block compatibility.
			$this->declare_WC_Block_compatibility();
		}

		/**
		 * Declare the plugin is compatibility with WC HPOS.
		 * 
		 * @since 2.9.0
		 * @return void
		 */
		public function declare_WC_HPOS_compatibility() {
			if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', WAL_PLUGIN_FILE, true);
			}
		}

		/**
		 * Declare the plugin is compatibility with WC block.
		 * 
		 * @since 3.7.0
		 * @return void
		 */
		public function declare_WC_Block_compatibility() {
			if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('cart_checkout_blocks', WAL_PLUGIN_FILE, true);
			}
		}

		/**
		 * Init.
		 * 
		 * @since 1.0.0
		 * */
		public function init() {
			$this->load_plugin_textdomain();

			// Modules.
			WAL_Module_Instances::get_modules();
			// Widgets.
			WAl_Widget_Instances::get_widgets();
			// Notifications.
			WAL_Notification_Instances::get_notifications();
			// Compatibility.
			WAL_Compatibility_Instances::instance();
		}

		/**
		 * Plugins Loaded.
		 * 
		 * @since 1.0.0
		 * */
		public function plugins_loaded() {
			/**
			 * This hook is used to do extra action before plugin loaded.
			 * 
			 * @since 1.0.0
			 */
			do_action('wal_before_plugin_loaded');

			// Current User Wallet Init.
			WAL_Current_User_Wallet::init();

			//Payment Gateway.
			include_once WAL_ABSPATH . 'inc/gateways/class-wal-gateway-wallet.php';

			add_filter('woocommerce_payment_gateways', array( $this, 'add_custom_gateways' ));

			/**
			 * This hook is used to do extra action after plugin loaded.
			 * 
			 * @since 1.0
			 */
			do_action('wal_after_plugin_loaded');
		}

		/**
		 * Add the custom gateways.
		 * 
		 * @since 1.0.0
		 * @return array
		 */
		public function add_custom_gateways( $wc_gateways ) {
			$wc_gateways[] = 'WAL_Gateway_Wallet';

			return $wc_gateways;
		}

		/**
		 * Templates.
		 * 
		 * @since 1.0.0
		 * @return string
		 * */
		public function templates() {
			return WAL_PLUGIN_PATH . '/templates/';
		}

		/**
		 * Get the menu name.
		 * 
		 * @since 1.0.0
		 * @return string
		 * */
		public function menu_name() {
			return __('Wallet', 'wallet-for-woocommerce');
		}

		/**
		 * Get the wallet fee name.
		 * 
		 * @since 1.0.0
		 * @return string
		 * */
		public function wallet_fee_name() {
			return $this->wallet_fee_name;
		}
	}

}
