<?php

/**
 * Handles Top-up Bonus Handler.
 * 
 * @since 2.5.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!class_exists('WAL_Topup_Bonus_Handler')) {

	/**
	 * Class.
	 */
	class WAL_Topup_Bonus_Handler {

		/**
		 * Amount.
		 * 
		 * @var string
		 * */
		private $topup_amount;

		/**
		 * Bonus amount.
		 * 
		 * @var int
		 * */
		private $bonus_amount = 0;

		/**
		 * Valid rules.
		 * 
		 * @var Array
		 * */
		private $valid_rules = array();

		/**
		 * Bonus rules.
		 * 
		 * @var Array
		 * */
		private $bonus_rule = array();

		/**
		 * Class initialization.
		 * */
		public function __construct( $topup_amount ) {
			$this->topup_amount = $topup_amount;

			$this->prepare_amount();
		}

		/**
		 * Get the amount.
		 *
		 * @param float $topup_amount
		 * 
		 * @return float
		 */
		public static function get_amount( $topup_amount ) {
			$bonus = new self($topup_amount);

			return $bonus->get_bonus_amount();
		}

		/**
		 * Get the bonus amount.
		 * 
		 * @return String
		 */
		private function get_bonus_amount() {
			return $this->bonus_amount;
		}

		/**
		 * Prepare amount.
		 * 
		 * @return String
		 * */
		private function prepare_amount() {
			$bonus_topup_enable = get_option('wal_general_enable_topup_bonus', 'no');
			//Return if Bonus top-up disable
			if ('yes' != $bonus_topup_enable) {
				return $this->bonus_amount;
			}

			$topup_bonus_rules = get_option('wal_general_topup_bonus_rules', array());
			//Return if Top-up bonus rules empty
			if (!wal_check_is_array($topup_bonus_rules)) {
				return $this->bonus_amount;
			}

			foreach ($topup_bonus_rules as $key => $bonus_rule) {
				$this->bonus_rule = $bonus_rule;

				if (!$this->validate_rule()) {
					continue;
				}

				$this->valid_rules[$key] = '1' == $this->bonus_rule['bonus_type'] ? $this->bonus_rule['bonus_amount'] : $this->bonus_rule['bonus_amount'] / 100 * $this->topup_amount;
			}

			$this->prepare_amount_by_priority();
		}

		/**
		 * Prepare amount by priority
		 * 
		 * @return Array
		 * */
		private function prepare_amount_by_priority() {
			if (!wal_check_is_array($this->valid_rules)) {
				return $this->bonus_amount;
			}

			switch (get_option('wal_general_topup_bonus_rule_type', '1')) {
				case '2':
					$this->bonus_amount = end($this->valid_rules);
					break;
				case '3':
					$this->bonus_amount = min($this->valid_rules);
					break;
				case '4':
					$this->bonus_amount = max($this->valid_rules);
					break;
				default:
					$this->bonus_amount = reset($this->valid_rules);
					break;
			}
		}

		/**
		 * Validate rule 
		 *
		 * @return bool.
		 * */
		private function validate_rule() {
			$min_amount = isset($this->bonus_rule['min_amount']) ? $this->bonus_rule['min_amount'] : '';
			$max_amount = isset($this->bonus_rule['max_amount']) ? $this->bonus_rule['max_amount'] : '';

			if (!$min_amount && !$max_amount) {
				return true;
			}

			if ($min_amount <= $this->topup_amount && $max_amount >= $this->topup_amount) {
				return true;
			}

			if (( !$min_amount && $max_amount ) && $max_amount >= $this->topup_amount) {
				return true;
			}

			if (( !$max_amount && $min_amount ) && $min_amount <= $this->topup_amount) {
				return true;
			}

			return false;
		}
	}

}
