<?php

/**
 * Custom Post Type.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}

if ( ! class_exists( 'WAL_Register_Post_Types' ) ) {

	/**
	 * Class.
	 */
	class WAL_Register_Post_Types {

		/**
		 * Wallet Post Type.
		 * 
		 * @var string
		 */
		const WALLET_POSTTYPE = 'wal_wallet' ;

		/**
		 * Transaction Log Post Type.
		 * 
		 * @var string
		 */
		const TRANSACTION_LOG_POSTTYPE = 'wal_transaction_log' ;

		/**
		 * Fund Transfer Post Type.
		 * 
		 * @var string
		 */
		const FUND_TRANSFER_POSTTYPE = 'wal_fund_transfer' ;

		/**
		 * Fund Transfer Log Post Type.
		 * 
		 * @var string
		 */
		const FUND_TRANSFER_LOG_POSTTYPE = 'wal_fund_trans_log' ;

		/**
		 * Auto Top-up Users Post Type.
		 * 
		 * @var string
		 */
		const AUTO_TOPUP_USERS_POSTTYPE = 'wal_autotopup_users' ;

		/**
		 * Gift Voucher Post Type.
		 * 
		 * @var string
		 */
		const GIFT_VOUCHER_POSTTYPE = 'wal_gift_voucher' ;

		/**
		 * Wallet Withdrawal Post Type.
		 * 
		 * @var string
		 */
		const WALLET_WITHDRAWAL_POSTTYPE = 'wal_wallet_withdraw' ;

		/**
		 * Class initialization.
		 */
		public static function init() {
			add_action( 'init', array( __CLASS__, 'register_custom_post_types' ) ) ;
		}

		/**
		 * Register Custom Post types.
		 * 
		 * @return void
		 */
		public static function register_custom_post_types() {
			if ( ! is_blog_installed() ) {
				return ;
			}

			$custom_post_types = array(
				self::WALLET_POSTTYPE            => array( 'WAL_Register_Post_Types', 'wallet_post_type_args' ),
				self::TRANSACTION_LOG_POSTTYPE   => array( 'WAL_Register_Post_Types', 'transaction_log_post_type_args' ),
				self::FUND_TRANSFER_POSTTYPE     => array( 'WAL_Register_Post_Types', 'fund_transfer_post_type_args' ),
				self::FUND_TRANSFER_LOG_POSTTYPE => array( 'WAL_Register_Post_Types', 'fund_transfer_log_post_type_args' ),
				self::AUTO_TOPUP_USERS_POSTTYPE  => array( 'WAL_Register_Post_Types', 'auto_topup_users_post_type_args' ),
				self::GIFT_VOUCHER_POSTTYPE      => array( 'WAL_Register_Post_Types', 'gift_voucher_post_type_args' ),
				self::WALLET_WITHDRAWAL_POSTTYPE => array( 'WAL_Register_Post_Types', 'wallet_withdrawal_post_type_args' ),
					) ;
			/**
			 * This hook is used to alter the custom post types.
			 * 
			 * @since 1.0
			 */
			$custom_post_types = apply_filters( 'wal_add_custom_post_types', $custom_post_types ) ;

			// Return if no post type has to register.
			if ( ! wal_check_is_array( $custom_post_types ) ) {
				return ;
			}

			foreach ( $custom_post_types as $post_type => $args_function ) {

				$args = array() ;
				if ( $args_function ) {
					$args = call_user_func_array( $args_function, $args ) ;
				}

				// Check if the already current post type is register.
				if ( post_type_exists( $post_type ) ) {
					continue ;
				}

				// Register the custom post type.
				register_post_type( $post_type, $args ) ;
			}
		}

		/**
		 * Prepare the wallet post type arguments.
		 * 
		 * @return array
		 */
		public static function wallet_post_type_args() {
			/**
			 * This hook is used to alter the wallet post type arguments.
			 * 
			 * @since 1.0
			 */
			return apply_filters(
					'wal_wallet_post_type_args', array(
				'labels'              => array(
					'name'               => __( 'All Wallet', 'wallet-for-woocommerce' ),
					'singular_name'      => __( 'Wallet', 'wallet-for-woocommerce' ),
					'menu_name'          => __( 'All Wallet', 'wallet-for-woocommerce' ),
					'add_new'            => __( 'Add New', 'wallet-for-woocommerce' ),
					'add_new_item'       => __( 'Add New Wallet', 'wallet-for-woocommerce' ),
					'new_item'           => __( 'New Wallet', 'wallet-for-woocommerce' ),
					'edit_item'          => __( 'Edit Wallet', 'wallet-for-woocommerce' ),
					'view_item'          => __( 'View Wallet', 'wallet-for-woocommerce' ),
					'all_items'          => __( 'All Wallet', 'wallet-for-woocommerce' ),
					'search_items'       => __( 'Search Users', 'wallet-for-woocommerce' ),
					'not_found'          => __( 'No records found.', 'wallet-for-woocommerce' ),
					'not_found_in_trash' => __( 'No records found in Trash.', 'wallet-for-woocommerce' ),
					),
					'description'         => __( 'This is where store Wallets are stored.', 'wallet-for-woocommerce' ),
					'public'              => false,
					'hierarchical'        => false, // Hierarchical causes memory issues - WP loads all records!
					'exclude_from_search' => true,
					'publicly_queryable'  => false,
					'show_ui'             => true,
					'show_in_menu'        => 'wal_wallet',
					'capability_type'     => 'post',
					'capabilities'        => array(
					'create_posts' => 'do_not_allow',
					),
					'map_meta_cap'        => true,
					'supports'            => false,
					'has_archive'         => false,
					'rewrite'             => false,
					)
					) ;
		}

		/**
		 * Prepare the transaction log post type arguments.
		 * 
		 * @return array
		 */
		public static function transaction_log_post_type_args() {
			/**
			 * This hook is used to alter the transaction logs post type arguments.
			 * 
			 * @since 1.0
			 */
			return apply_filters(
					'wal_transaction_log_post_type_args', array(
				'labels'              => array(
					'name'               => __( 'Transactions', 'wallet-for-woocommerce' ),
					'singular_name'      => __( 'Transaction', 'wallet-for-woocommerce' ),
					'menu_name'          => __( 'Transaction', 'wallet-for-woocommerce' ),
					'view_item'          => __( 'View Transaction', 'wallet-for-woocommerce' ),
					'all_items'          => __( 'Transactions', 'wallet-for-woocommerce' ),
					'search_items'       => __( 'Search Users', 'wallet-for-woocommerce' ),
					'not_found'          => __( 'No records found.', 'wallet-for-woocommerce' ),
					'not_found_in_trash' => __( 'No records found in Trash.', 'wallet-for-woocommerce' ),
					),
					'description'         => __( 'This is where store transactions are stored.', 'wallet-for-woocommerce' ),
					'public'              => false,
					'hierarchical'        => false, // Hierarchical causes memory issues - WP loads all records!
					'exclude_from_search' => true,
					'publicly_queryable'  => false,
					'show_ui'             => true,
					'show_in_menu'        => 'wal_wallet',
					'capabilities'        => array(
					'create_posts' => 'do_not_allow',
					),
					'map_meta_cap'        => true,
					'supports'            => false,
					'has_archive'         => false,
					'rewrite'             => false,
					)
					) ;
		}

		/**
		 * Prepare the fund transfer post type arguments.
		 * 
		 * @return array
		 */
		public static function fund_transfer_post_type_args() {
			/**
			 * This hook is used to alter the fund transfer post type arguments.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_fund_transfer_post_type_args', array(
				'label'           => __( 'Fund Transfer', 'wallet-for-woocommerce' ),
				'public'          => false,
				'hierarchical'    => false,
				'supports'        => false,
				'capability_type' => 'post',
				'rewrite'         => false,
					)
					) ;
		}

		/**
		 * Prepare the fund transfer log post type arguments.
		 * 
		 * @return array
		 */
		public static function fund_transfer_log_post_type_args() {
			/**
			 * This hook is used to alter the fund transfer log post type arguments.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_fund_transfer_log_post_type_args', array(
				'label'           => __( 'Fund Transfer Log', 'wallet-for-woocommerce' ),
				'public'          => false,
				'hierarchical'    => false,
				'supports'        => false,
				'capability_type' => 'post',
				'rewrite'         => false,
					)
					) ;
		}

		/**
		 * Prepare the auto top-up users post type arguments.
		 * 
		 * @return array
		 */
		public static function auto_topup_users_post_type_args() {
			/**
			 * This hook is used to alter the auto top-up users post type arguments.
			 * 
			 * @since 1.0
			 */
			return apply_filters(
					'wal_autotopup_users_post_type_args', array(
				'labels'              => array(
					'name'               => __( 'Auto Topup Users', 'wallet-for-woocommerce' ),
					'singular_name'      => __( 'Auto Topup User', 'wallet-for-woocommerce' ),
					'menu_name'          => __( 'Auto Topup User', 'wallet-for-woocommerce' ),
					'view_item'          => __( 'View Auto Topup User', 'wallet-for-woocommerce' ),
					'all_items'          => __( 'Auto Topup Users', 'wallet-for-woocommerce' ),
					'search_items'       => __( 'Search Auto Topup Users', 'wallet-for-woocommerce' ),
					'not_found'          => __( 'No records found.', 'wallet-for-woocommerce' ),
					'not_found_in_trash' => __( 'No records found in Trash.', 'wallet-for-woocommerce' ),
					),
					'description'         => __( 'This is where store auto topup users are stored.', 'wallet-for-woocommerce' ),
					'public'              => false,
					'hierarchical'        => false, // Hierarchical causes memory issues - WP loads all records!
					'exclude_from_search' => true,
					'publicly_queryable'  => false,
					'show_ui'             => true,
					'show_in_menu'        => false,
					'capabilities'        => array(
					'create_posts' => 'do_not_allow',
					),
					'map_meta_cap'        => true,
					'supports'            => false,
					'has_archive'         => false,
					'rewrite'             => false,
					)
					) ;
		}

		/**
		 * Prepare the gift voucher post type arguments.
		 * 
		 * @return array
		 */
		public static function gift_voucher_post_type_args() {
			/**
			 * This hook is used to alter the gift voucher post type arguments.
			 * 
			 * @since 1.0
			 */
			return apply_filters( 'wal_gift_voucher_post_type_args', array(
				'label'           => __( 'Gift Voucher', 'wallet-for-woocommerce' ),
				'public'          => false,
				'hierarchical'    => false,
				'supports'        => false,
				'capability_type' => 'post',
				'rewrite'         => false,
					)
					) ;
		}

		/**
		 * Prepare the wallet withdrawal post type arguments.
		 * 
		 * @since 1.8
		 * 
		 * @return array
		 */
		public static function wallet_withdrawal_post_type_args() {
			/**
			 * This hook is used to alter the wallet withdrawal post type arguments.
			 * 
			 * @since 1.8
			 * 
			 * @param array
			 */
			return apply_filters(
					'wal_wallet_withdrawal_post_type_args', array(
				'labels' => array(
					'name' => __('Wallet Withdrawals', 'wallet-for-woocommerce'),
					'singular_name' => __('Wallet Withdrawal', 'wallet-for-woocommerce'),
					'menu_name' => _x('Wallet Withdrawal', 'Admin menu name', 'wallet-for-woocommerce'),
					'edit_item' => __('Edit Wallet Withdrawal', 'wallet-for-woocommerce'),
					'view_item' => __('View Wallet Withdrawal', 'wallet-for-woocommerce'),
					'all_items' => self::get_wallet_withdrawal_menu_name(),
					'search_items' => __('Search Users', 'wallet-for-woocommerce'),
					'not_found' => __('No records found.', 'wallet-for-woocommerce'),
					'not_found_in_trash' => __('No records found in Trash.', 'wallet-for-woocommerce'),
					),
					'description'         => __( 'This is where store wallet withdrawals are stored.', 'wallet-for-woocommerce' ),
					'public'              => false,
					'hierarchical'        => false, // Hierarchical causes memory issues - WP loads all records!
					'exclude_from_search' => true,
					'publicly_queryable'  => false,
					'show_ui'             => WAL_Module_Instances::get_module_by_id( 'wallet_withdrawal' )->is_enabled() ? true : false,
					'show_in_menu'        => 'wal_wallet',
					'capabilities'        => array(
					'create_posts' => 'do_not_allow',
					),
					'map_meta_cap'        => true,
					'supports'            => false,
					'has_archive'         => false,
					'rewrite'             => false,
					)
			);
		}

		/**
		 * Get Wallet Withdrawal menu name.
		 * 
		 * @since 2.8.0
		 * 
		 * @return String
		 */
		public static function get_wallet_withdrawal_menu_name() {
			$menu_name = esc_html__('Wallet Withdrawals', 'wallet-for-woocommerce');
			$pending_count = wal_get_pending_wallet_withdrawal_ids();
			if (!empty($pending_count)) {
				$menu_name .= '<span class="wal-pending-count-wrap awaiting-mod update-plugins count-' . esc_attr($pending_count) . '"><span class="wal-pending-count">' . number_format_i18n($pending_count) . '</span></span>';
			}

			/**
			 * This hook is used to alter the wallet withdrawal menu name.
			 * 
			 * @since 2.8.0
			 * 
			 */
			return apply_filters('wal_wallet_withdrawal_menu_name', $menu_name);
		}
	}

	WAL_Register_Post_Types::init() ;
}
