<?php

/**
 * Handles Purchased Product Fund Handler.
 * 
 * @since 3.1.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!class_exists('WAL_Purchased_Product_Fund_Handler')) {

	/**
	 * Class.
	 */
	class WAL_Purchased_Product_Fund_Handler {

		/**
		 * Product ID.
		 *
		 * @var int
		 * @since 3.1.0
		 * */
		private $product_id;

		/**
		 * Variation ID.
		 *
		 * @var int
		 * @since 3.1.0
		 * */
		private $variation_id;

		/**
		 * Current product ID.
		 *
		 * @var int
		 * @since 3.1.0
		 * */
		private $current_product_id;

		/**
		 * Quantity
		 *
		 * @var int
		 * @since 3.1.0
		 * */
		private $quantity;

		/**
		 * Regular price.
		 *
		 * @var float
		 * @since 3.1.0
		 * */
		private $regular_price;

		/**
		 * Purchased product fund.
		 *
		 * @var int
		 * @since 3.1.0
		 * */
		private $purchased_product_fund = 0;

		/**
		 * Class initialization.
		 * 
		 * @param int $product_id
		 * @param int $variation_id
		 * @param int $qty
		 * @param float $regular_price
		 * @since 3.1.0
		 * 
		 * */
		public function __construct( $product_id, $variation_id, $qty, $regular_price ) {
			$this->product_id = $product_id;
			$this->variation_id = $variation_id;
			$this->quantity = $qty;
			$this->regular_price = floatval($regular_price);
			$this->current_product_id = !empty($variation_id) ? $variation_id : $product_id;

			$this->prepare_purchased_product_fund();
		}

		/**
		 * Get fund
		 *
		 * @param int $product_id
		 * @param int $variation_id
		 * @param int $qty
		 * @param float $regular_price
		 * @since 3.1.0
		 * 
		 * @return float
		 */
		public static function get_fund( $product_id, $variation_id, $qty, $regular_price ) {
			$purchased_product = new self($product_id, $variation_id, $qty, $regular_price);

			return $purchased_product->get_purchased_product_fund();
		}

		/**
		 * Get purchased product fund.
		 *
		 * @since 3.1.0
		 * @return string
		 */
		private function get_purchased_product_fund() {
			return $this->purchased_product_fund;
		}

		/**
		 * Prepare purchased product fund.
		 *
		 * @since 3.1.0
		 * @return String
		 * */
		private function prepare_purchased_product_fund() {
			$fund_type = get_option('wal_module_product_purchase_mode', '1');
			if ('1' !== $fund_type) {
				return $this->purchased_product_fund;
			}

			$purchased_product_fund = $this->get_product_level_fund();
			if (!$purchased_product_fund) {
				return $this->purchased_product_fund;
			}

			$this->purchased_product_fund = $purchased_product_fund;
		}

		/**
		 * Get product level fund
		 *
		 * @since 3.1.0
		 * @return float.
		 */
		private function get_product_level_fund() {
			$block_fund = get_post_meta($this->current_product_id, '_wal_block_fund', true);
			if ('yes' === $block_fund) {
				return $this->purchased_product_fund;
			}

			$product_fund_value = get_post_meta($this->current_product_id, '_wal_fund_value', true);
			if (empty($product_fund_value)) {
				return $this->get_category_level_fund();
			}

			$product_fund_type = get_post_meta($this->current_product_id, '_wal_fund_type', true);
			if ('2' === $product_fund_type) {
				$this->purchased_product_fund = floatval($product_fund_value) * $this->quantity;
			} else {
				$this->purchased_product_fund = ( $this->regular_price / 100 ) * floatval($product_fund_value);
			}

			return floatval($this->purchased_product_fund);
		}

		/**
		 * Get category level fund
		 *
		 * @since 3.1.0
		 * @return float.
		 */
		private function get_category_level_fund() {
			$terms = get_the_terms($this->product_id, 'product_cat');
			if (!wal_check_is_array($terms)) {
				return $this->purchased_product_fund;
			}
			
			$fund_value = array();
			foreach ($terms as $term) {
				$fund_blocked = get_term_meta($term->term_id, '_wal_block_fund', true);
				if ('on' === $fund_blocked) {
					continue;
				}

				$category_fund_value = get_term_meta($term->term_id, '_wal_fund_value', true);
				if (empty($category_fund_value)) {
					return $this->get_global_level_fund();
				}
				
				if ('2' === get_term_meta($term->term_id, '_wal_fund_type', true)) {
					$fund_value[] = floatval($category_fund_value) * $this->quantity;
				} else {
					$fund_value[] = ( $this->regular_price / 100 ) * floatval($category_fund_value);
				}
			}

			return wal_check_is_array($fund_value) ? floatval(max($fund_value)) : 0;
		}

		/**
		 * Get global level fund
		 *
		 * @since 3.1.0
		 * @return float.
		 */
		private function get_global_level_fund() {
			if ('2' === get_option('wal_module_product_purchase_fund_type')) {
				$this->purchased_product_fund = floatval(get_option('wal_module_product_purchase_amount')) * $this->quantity;
			} else {
				$this->purchased_product_fund = ( $this->regular_price / 100 ) * floatval(get_option('wal_module_product_purchase_amount'));
			}

			return floatval($this->purchased_product_fund);
		}
	}

}
